////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////
/* -*- Mode:c++; c-basic-offset:4; tab-width:4; indent-tabs-mode:t -*- */

/**************************************************************************
  WorkloadApp.cpp

begin           : Nov 8, 2003
version         : $Id: WorkloadApp.cpp 2382 2005-11-03 22:54:59Z ashu $
copyright       : (C) 2003      Jeff Pang        ( jeffpang@cs.cmu.edu )
(C) 2003      Justin Weisz     (  jweisz@cs.cmu.edu  )

***************************************************************************/

#include "sim_shared.h"
#include "WorkloadApp.h"
#include "SimObject.h"

    WorkloadApp::WorkloadApp(Server *server, gtime_t migrateInterval,
			     gtime_t interestInterval) : 
	GApplication(), server(server), migrateInterval(migrateInterval),
	interestInterval(interestInterval) {
    server->registerApp(this);
  }

WorkloadApp::~WorkloadApp() {
}

void WorkloadApp::initWorld() {
    if (migrateInterval != (uint32)-1) {
	// Jitter initial timer by interval so servers 
	// will be out of phase with each other
	server->handleRegisterTimer(migrateInterval + 
				    (gtime_t)( drand48()*migrateInterval ),
				    TIMER_MIGRATE);
    }
    if (interestInterval != (uint32)-1) {
	server->handleRegisterTimer(interestInterval + 
				    (gtime_t)( drand48()*interestInterval ),
				    TIMER_INTEREST);
    }
}

void WorkloadApp::handleTraceEvent(TraceEvent *tev) {
    //	debug("%d: app on server:%d handling tev", 
    //	  server->getTime(), server->sid.addr);

    switch(tev->type) {
    case EV_MOD: {
	handleMod(tev);
	break;
    }
    case EV_SUB: {
	handleSub(tev);
	break;
    }
    case EV_CREATE: {
	handleCreate(tev);
	break;
    }
    case EV_DESTROY: {
	handleDestroy(tev);
	break;
    }
    default:
	Debug::die("unknown trace event type %d", tev->type);
    }

    return;
}

void WorkloadApp::handleMod(TraceEvent *ev) {
    ObjectStore store = server->getObjectStore();
    ObjectStoreIter iter = store.find(ev->origin);

#ifndef DEBUG
    // HACK JUST IN CASE
    if (iter == store.end()) return;
#endif

    ASSERT(iter != store.end());
    ASSERT(iter->second->getReplicaStatus() == PRIMARY);

    ObjectStoreIter iter_targ = store.find(ev->target);

    //#ifndef DEBUG
    // xxx HACK JUST IN CASE
    // for some *wierd* cases using the region-based policy, we really need
    // to ignore this MOD. Suppose the following scenario:
    //
    // - missile X is in region 1
    // - player  Y is in region 2
    // - both are initially on the same server S1
    // - missile X moves to region 3, which is handled by a server S2
    // - this will cause missile X to publish itself so Y knows where he is
    // - BUT missile X now does NOT know where player Y is unless S2 already
    //   had a replica for Y
    // - ... so if missle X now hits Y, it will try to modify a replica
    //   which does not yet exit on S2
    // - the *problem* is that X is being migrated mid-frame in the simulated
    //   trace, which would probably never happen in real-life.
    // - since we've established updates are cheap anyway, this does not
    //   really affect our bandwidth or migration results since X will
    //   learn about Y in the next frame of the simulation anyway
    if (iter_targ == store.end()) return;
    //#endif

    ASSERT(iter_targ != store.end());
    SimObject *target = dynamic_cast<SimObject *>(iter_targ->second);
    for (AttributeMapIter i = ev->attributes.begin(); 
	 i != ev->attributes.end(); i++) {
	target->setAttribute(i->first, i->second, MOD_GLOBAL);
    }
    target->setMod(ev);

    // TODO: app shouldn't have to decide if its primary or replica...
    if (target->getReplicaStatus() == PRIMARY)
	server->handleObjPublish(target);
    else
	server->handleObjUpdate(target);
}

void WorkloadApp::handleSub(TraceEvent *ev) {
    ObjectStore store = server->getObjectStore();
    ObjectStoreIter iter = store.find(ev->origin);
    ASSERT(iter != store.end());
    ASSERT(iter->second->getReplicaStatus() == PRIMARY);

    iter->second->setInterests(ev->subs);
    ev->subs.clear();
}

void WorkloadApp::handleCreate(TraceEvent *ev) {
    if (ev->origin != GUID_NONE) {
	ObjectStore store = server->getObjectStore();
	ObjectStoreIter iter = store.find(ev->origin);
	ASSERT(iter != store.end());
	ASSERT(iter->second->getReplicaStatus() == PRIMARY);
    }

    Value vals[NUM_ATTRS];
    // initial attributes
    for (int i=0; i<NUM_ATTRS; i++) {
	vals[i] = ev->attributes[ATTRS[i]];
	ASSERT(vals[i].m_Type != ATTR_INVALID);
    }
    // create object
    SimObject *obj = new SimObject(ev->target, vals);

    server->handleAddObject(obj);
}

void WorkloadApp::handleDestroy(TraceEvent *ev) {
    ObjectStore store = server->getObjectStore();
    ObjectStoreIter iter = store.find(ev->origin);
    ASSERT(iter != store.end());
    ASSERT(iter->second->getReplicaStatus() == PRIMARY);

    server->handleRemoveObject(iter->second);
}

void WorkloadApp::handleSystemEvent(SystemEvent *sev) {

    /*
      debug("SERVER %d: OBJECT STORE:", server->sid.addr);
      ObjectStore store = server->getObjectStore();
      for (ObjectStoreIter i = store.begin(); i != store.end(); i++) {
      DEBUG_RUN(cerr << "   " << i->second << endl);
      }
    */

    switch(sev->type) {
    case EV_TIMER: {
	server->handleTimer(sev->timer_type);
	switch(sev->timer_type) {
	case TIMER_MIGRATE:
	    //debug("%d: app on server:%d handling migrate timer", 
	    //	  server->getTime(), server->sid.addr);

	    server->handleRegisterTimer(migrateInterval, TIMER_MIGRATE);
	    break;
	case TIMER_INTEREST:
	    server->handleRegisterTimer(interestInterval, TIMER_INTEREST);
	    break;
	default:
	    Debug::die("unknown timer type $d", sev->timer_type);
	}
	break;
    }
    default:
	Debug::die("unknown system event type %d", sev->type);
    }

    return;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
