////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////
/* -*- Mode:c++; c-basic-offset:4; tab-width:4; indent-tabs-mode:t -*- */

/**************************************************************************
  SimProtocolLayer.cpp

begin           : Nov 8, 2003
version         : $Id: SimProtocolLayer.cpp 2382 2005-11-03 22:54:59Z ashu $
copyright       : (C) 2003      Jeff Pang        ( jeffpang@cs.cmu.edu )
(C) 2003      Justin Weisz     (  jweisz@cs.cmu.edu  )

***************************************************************************/

#include "sim_shared.h"
#include "SimProtocolLayer.h"
#include <framework2/TypeTable.h>

    // define this to update the lightly loaded servers every time we send a packet
    //#define INLINE_LOAD_CHECK

    static uint32 realSize(uint32 size) {
    /*
    // assume 1500byte packets
    uint32 pkts = size/1500;
    if (pkts == 0) pkts = 1;
    // assume UDP/IP
    return size + (20 + 8)*pkts;
    */
    // FIXME: ignore IP overhead for now...
    return size;
  }

SimProtocolLayer::SimProtocolLayer(God *god, Logger *logger, 
				   gtime_t bwidthWindowSize) : 
    god(god), logger(logger), windowSize(bwidthWindowSize) {
    god->registerProto(this);
}

SimProtocolLayer::~SimProtocolLayer() {
}

void SimProtocolLayer::registerHandler(Server *s) {
    // do nothing... this isn't needed in the simulator
}

void SimProtocolLayer::publish(GEvent *e) {
    GUID guid = e->getGUID();
    Server *curr = god->getCurrServer();

    //debug("publishing event for type %s", e->GetAttribute(TYPE)->m_Sval->c_str());

    uint32 size = realSize( pubSize(e) );

    // log send
    logger->logPubSend(curr->sid, guid, size);
    log_out(size);

    GUID_DEBUG_MSG(guid, "publishing %d", guid.localOID);

    SIDSet seen; // only send to each receiver once
    seen.insert(god->getCurrServer()->sid); // don't send it back to me
    GUIDSet interested = god->getInterested(guid);
    for (GUIDSetIter i = interested.begin(); i != interested.end(); i++) {
	Server *serv = god->locateObj(*i);

#ifndef DEBUG
	// HACK just in case
	if (serv == NULL) continue;
#endif

	GUID_DEBUG_MSG(*i, "%d is interested in pub[%d]",
		       i->localOID, guid.localOID);

	if (seen.find(serv->sid) == seen.end()) {
	    // have the receiver handle it
	    god->setCurrServer(serv);
	    // log receive
	    logger->logPubRecv(serv->sid, guid, size);
	    GUID_DEBUG_MSG(guid, "sending pub to %d", serv->sid.addr);
	    log_in(size);
	    serv->handleEvent(e);
	    god->setCurrServer(curr);
	    seen.insert(serv->sid);
	}
    }

    delete( e );
}

void SimProtocolLayer::subscribe(Interest *i) {
    // don't actually need to do anything except log this
    // because God will automatically decide what objects
    // each GObject is interested in using the trace

    uint32 size = realSize( subSize(i) );

    logger->logSub(god->getCurrServer()->sid, size);
    log_out(size);

    //delete( i ); -- these stay in the GObject!
}

void SimProtocolLayer::update(SID target, GUpdate *delta) {
    ASSERT(target != god->getCurrServer()->sid);
    Server *curr = god->getCurrServer();

    uint32 size = realSize( updateSize(delta) );

    // log update
    logger->logUpdate(curr->sid, target, delta->getTarget(), size);
    log_out(size);

    // have the receiver handle it
    Server *serv = god->getServer(target);
    god->setCurrServer(serv);
    log_in(size);
    serv->handleUpdate(delta);
    god->setCurrServer(curr);

    delete( delta );
}

void SimProtocolLayer::migrate(GObject *o, SID sid) {
    SystemEvent *ev = new SystemEvent(god->getCurrTime(), EV_MIGRATE);
    ev->sid  = god->getCurrServer()->sid;
    ev->targ = sid;
    ev->obj  = o;

    god->addEvent(ev);
}

void SimProtocolLayer::handleMigrate(SystemEvent *ev) {
    ASSERT(ev->type == EV_MIGRATE);
    ASSERT(ev->sid  == god->getCurrServer()->sid);

    GObject *o = ev->obj;
    SID sid    = ev->targ;

    Server *curr = god->getCurrServer();
    Server *targ = god->getServer(sid);
    ASSERT(curr != targ);

    uint32 size = realSize( objSize(o) );

    // log migration
    logger->logMigration(curr->sid, targ->sid, o->guid, size);
    log_out(size);

    // serialize at current server
    Packet pkt(sizeof(gtype_t) + o->GetLength());
    TypeTable::serialize(&pkt, o);

    // remove from current server
    // TODO: this is before or after added to remote server?
    curr->handleRemovePrimary(o->guid);

    // tell God that the location of this object has changed
    // careful about when this is called... (in between ownership)
    god->objMoved(o->guid, sid);

    // deserialize at target server
    god->setCurrServer(targ);
    log_in(size);
    pkt.ResetBufPosition();
    GObject *copy = TypeTable::deserialize( &pkt );
    // add to target server
    targ->handleAddPrimary(copy);
    god->setCurrServer(curr);

    // log the client load change if attached to a player
    if (*(copy->getAttribute(TYPE).m_Sval) == PLAYER) {
	int curr_size = curr->getNumClients();
	int targ_size = targ->getNumClients();
	logger->logClientLoad(curr->sid, 
			      clientIn(curr_size),
			      clientOut(curr_size));
	logger->logClientLoad(targ->sid,
			      clientIn(targ_size),
			      clientOut(targ_size));
    }

}

gtime_t SimProtocolLayer::getTime() {
    return god->getCurrTime();
}

void SimProtocolLayer::registerInterrupt(gtime_t time, interrupt_t type) {
    // jitter time by up to 10ms
    gtime_t real_time = time + 
	(gtime_t)(drand48()>=0.5?-drand48():drand48())*10;

    SystemEvent *ev = new SystemEvent(real_time, EV_TIMER);
    ev->timer_type = type;
    ev->sid = god->getCurrServer()->sid;
    god->addEvent(ev);
}


void SimProtocolLayer::log_in(uint32 size) {
    WindowPair *p = windows[god->getCurrServer()->sid];
    ASSERT(p != NULL);

    //	cout << "sid: " << god->getCurrServer()->sid << " time: " << 
    //	getTime() << " size: " << size << endl;

    Measurement last = p->inbound.back();
    if (getTime() - last.time < BUCKET_SIZE) {
	p->inbound.back().size += size;
    } else {
	Measurement m;
	m.time = getTime();
	m.size = size;
	p->inbound.push_back(m);
    }

#ifdef INLINE_LOAD_CHECK
    checkLightlyLoaded(god->getCurrServer());
#endif
}

void SimProtocolLayer::log_out(uint32 size) {
    WindowPair *p = windows[god->getCurrServer()->sid];
    ASSERT(p != NULL);

    Measurement last = p->outbound.back();
    if (getTime() - last.time < BUCKET_SIZE) {
	p->outbound.back().size += size;
    } else {
	Measurement m;
	m.time = getTime();
	m.size = size;
	p->outbound.push_back(m);
    }

#ifdef INLINE_LOAD_CHECK
    checkLightlyLoaded(god->getCurrServer());
#endif
}

void SimProtocolLayer::checkLightlyLoaded() {
#ifndef INLINE_LOAD_CHECK
    checkLightlyLoaded(god->getCurrServer());
#endif
}

void SimProtocolLayer::checkLightlyLoaded(Server *serv) {
    bwidth_t load_in = inboundBwidth(serv->sid);
    bwidth_t load_out = outboundBwidth(serv->sid);
    bool light_in = load_in <= serv->getParams()->lowwater_in;
    bool light_out = load_out <= serv->getParams()->lowwater_out;

    // TODO: better heuristic for picking whether it is "light"?
    // one of these could be true while the other is false...
    if (light_in && light_out) {
	LoadInfo info;
	info.sid = serv->sid;
	info.tohighwater_in = 
	    serv->getParams()->highwater_in - load_in;
	info.tohighwater_out = 
	    serv->getParams()->highwater_out - load_out;
	lightlyLoaded[serv->sid] = info;
    } else {
	LoadMapIter p = lightlyLoaded.find(serv->sid);
	if (p != lightlyLoaded.end()) {
	    lightlyLoaded.erase(p);
	}
    }

#ifdef DEBUG
    /*
      debugStream() << "sid=" << serv->sid << " in=" <<
      load_in << " out=" << load_out << 
      " num_clients=" << getNumClients(serv->sid) <<
      " client_in=" << clientIn(getNumClients(serv->sid)) <<
      " client_out=" << clientOut(getNumClients(serv->sid)) << endl;
    */
#endif
}

bwidth_t SimProtocolLayer::inboundBwidth() {
    return inboundBwidth(god->getCurrServer()->sid);
}

bwidth_t SimProtocolLayer::inboundBwidth(SID sid) {
    WindowPair *p = windows[sid];
    ASSERT(p != NULL);

    while ( p->inbound.size() > 0 && 
	    p->inbound.front().time + windowSize < getTime() ) {
	p->inbound.pop_front();
    }

    double used = 0;
    for (WindowIter i = p->inbound.begin(); i != p->inbound.end(); i++) {
	used += i->size;
    }

    //cerr << "IN:" << sid << ": buckets=" << p->inbound.size() << " used=" << used << " windowSize=" << windowSize << endl;

    return (bwidth_t)(used/windowSize) + 
	(bwidth_t)clientIn(getNumClients(sid));
    // treat client bandwidth as constant rate
}

bwidth_t SimProtocolLayer::outboundBwidth() {
    return outboundBwidth(god->getCurrServer()->sid);
}

bwidth_t SimProtocolLayer::outboundBwidth(SID sid) {
    WindowPair *p = windows[sid];
    ASSERT(p != NULL);

    while ( p->outbound.size() > 0 && 
	    p->outbound.front().time + windowSize < getTime() ) {
	p->outbound.pop_front();
    }

    double used = 0;
    for (WindowIter i = p->outbound.begin(); i != p->outbound.end(); i++) {
	used += i->size;
    }

    return (bwidth_t)(used/windowSize) + 
	(bwidth_t)clientOut(getNumClients(sid));
    // treat client bandwidth as constant rate
}

LoadMap *SimProtocolLayer::getLightlyLoaded() {
    return &lightlyLoaded;
}

ostream& SimProtocolLayer::debugStream() {
    return logger->getDebugStream();
}

int SimProtocolLayer::getNumClients(SID sid) {
    return clients[sid];
}

int SimProtocolLayer::getNumClients() {
    return getNumClients(god->getCurrServer()->sid);
}

void SimProtocolLayer::clientConnect(GObject *o) {
    SID sid = god->getCurrServer()->sid;
    int num = ++(clients[sid]);
    logger->logClientLoad( sid, clientIn(num), clientOut(num) );
    checkLightlyLoaded(god->getCurrServer());
}

void SimProtocolLayer::clientDisconnect(GObject *o) {
    SID sid = god->getCurrServer()->sid;
    int num = --(clients[sid]);
    logger->logClientLoad( sid, clientIn(num), clientOut(num) );
    checkLightlyLoaded(god->getCurrServer());
}

void SimProtocolLayer::addServer(Server *s) {
    windows[s->sid] = new WindowPair();
    clients[s->sid] = 0;
    logger->logClientLoad( s->sid, 0, 0 );
    checkLightlyLoaded(s);
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
