////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////
/* -*- Mode:c++; c-basic-offset:4; tab-width:4; indent-tabs-mode:t -*- */

/**************************************************************************
  God.h

begin           : Nov 8, 2003
version         : $Id: God.h 2382 2005-11-03 22:54:59Z ashu $
copyright       : (C) 2003      Jeff Pang        ( jeffpang@cs.cmu.edu )
(C) 2003      Justin Weisz     (  jweisz@cs.cmu.edu  )

***************************************************************************/

#ifndef __GOD_H__
#define __GOD_H__

#include <queue>
#include <fstream>
#include <pthread.h>
#include <framework2/common.h>
#include <framework2/Server.h>
#include <framework2/GApplication.h>
#include "WEvent.h"
#include "TraceEvent.h"
#include "SystemEvent.h"
#include "WorkloadApp.h"

    class SimProtocolLayer;

typedef hash_map<GUID, SID, hash_GUID, equal_GUID> InitMap;
typedef InitMap::iterator InitMapIter;
typedef hash_map<GUID, Server *, hash_GUID, equal_GUID> ObjServerMap; 
typedef ObjServerMap::iterator ObjServerMapIter;
typedef hash_map<GUID, GUIDSet, hash_GUID, equal_GUID>  InterestedMap;
typedef InterestedMap::iterator InterestedMapIter;
typedef hash_map<GUID, GUIDSet, hash_GUID, equal_GUID>  InterestsMap;
typedef InterestsMap::iterator InterestsMapIter;

// if the prio queue exceeds this length...
#define MAX_QUEUE_SIZE 50000
// sleep for this long before continuing processing
#define QUEUE_WAIT     10

typedef priority_queue<WEvent *, vector<WEvent *>, 
		       EventTimeComparator > EventQueue;

typedef enum { INIT_RANDOM, INIT_FILE, INIT_SMART } init_t;

typedef struct {
    const char *file;         // name of file to load
    uint32 timeMultiplier;    // scale trace time by this much
    pthread_mutex_t *lock;    // lock on wait, eventQueue, finished
    pthread_cond_t  *proceed; // tells consumer to proceed
    EventQueue *eventQueue;   // event queue to fill
    gtime_t maxTime;          // time of last TraceEvent added to queue
    bool finished;            // loader is finished
} LoaderArgs;

/**
 * State of the entire world (for the simulator)
 */
class God {
 private:

    void loadTrace(const char *file);
    void loadInit(const char *file);
    Server *pickInitialServer(init_t method, GUID guid);

    WEvent *nextEvent();
    bool moreEvents();

    void updateInterests(GUID guid, GUIDVec& interests);
    void createObject(GUID guid, Server *server);
    void destroyObject(GUID guid);

 protected:

    /**
     * map<guid of object, server it is on>
     */
    ObjServerMap  objServerMap;
    /**
     * map<guid of object, set<guid of objects it is interested in> >
     */
    InterestsMap interestsMap;
    /**
     * map<guid of object, set<guid of objects interested in it> >
     */
    InterestedMap interestedMap;
    /**
     * How to decide where to create new initial objects
     */
    init_t initMethod;
    /**
     * map<guid of object, sid to create it on initially>
     */
    InitMap initMap;
    /**
     * Time simulation began
     */
    gtime_t startTime;
    /**
     * Current time in simulation
     */
    gtime_t currTime;
    /**
     * Current event being processed in the simulation
     */
    WEvent *currEvent;
    /**
     * Current acting node
     */
    Server *currServer;
    /**
     * The protocol layer used for sim
     */
    SimProtocolLayer *proto;

    /**
     * All the servers in the world; map<SID, server>
     */
    ServerMap servers;

    /**
     * File trace is loaded from
     */
    const char *tracefile;
    const char *initialfile;
    uint32 timeMultiplier;
    /**
     * The queue of events
     */
    EventQueue eventQueue;
    // lock on queue
    pthread_mutex_t queueLock;
    // condition on all trace events before a system event being loaded
    pthread_cond_t  eventCondVar;
    // loader thread
    pthread_t loaderThread;
    LoaderArgs loaderArgs;

 public:

    God(gtime_t startTime, uint32 timeMultiplier, const char *tracefile,
	init_t initMethod, const char *initialfile);
    virtual ~God();

    void    registerProto(SimProtocolLayer *proto);
    gtime_t getStartTime();
    gtime_t getCurrTime();
    void    setCurrTime(gtime_t time);
    WEvent *getCurrEvent();
    void    setCurrEvent(WEvent *ev);
    Server *getCurrServer();
    void    setCurrServer(Server *serv);

    Server *locateObj(GObject *obj);
    Server *locateObj(GUID guid);
    void objMoved(GUID guid, SID sid);
    GUIDSet& getInterested(GObject *obj);
    GUIDSet& getInterested(GUID guid);

    Server *getServer(SID sid);
    void addServer(Server *server);

    void run(gtime_t stopTime);
    void addEvent(SystemEvent *ev);

    ////////// DEBUGGING //////////

    /**
     * Warning! this will erase everything from the event queue!
     */
    void dumpEventQueue(ostream& out);
    void printEventQueueTop(ostream& out);
};

#endif
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
