////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////
/* -*- Mode:c++; c-basic-offset:4; tab-width:4; indent-tabs-mode:t -*- */

/**************************************************************************
  God.h

begin           : Nov 8, 2003
version         : $Id: God.cpp 2382 2005-11-03 22:54:59Z ashu $
copyright       : (C) 2003      Jeff Pang        ( jeffpang@cs.cmu.edu )
(C) 2003      Justin Weisz     (  jweisz@cs.cmu.edu  )

***************************************************************************/

#include "God.h"
#include "SimProtocolLayer.h"
#include <framework2/Benchmark.h>
#include "sim_shared.h"

    // define this to load the entire trace before we start going
    //#define SYNCH_TRACE_LOAD 1

#define READ(var) { \
    in >> (var); \
	if (!in) Debug::Debug::die ("error reading '%s' at %s:%d", #var, file, lineno); \
}

#define LOCK(lock) { \
    int ret = pthread_mutex_lock( lock ); \
	ASSERT(ret == 0); \
}

#define UNLOCK(lock) { \
    int ret = pthread_mutex_unlock( lock ); \
	ASSERT(ret == 0); \
}

static void *traceLoader(void *arg);

/**
 * TraceFile Format:
 *
 * <time> interest <guid> <N> <guid_1> ... <guid_N>
 *
 * <time> mod <origin> <target> <N> <key_1> <type_1> <val_1> ... \ 
 *                                  <key_N> <type_N> <val_N> <size>
 *
 * <time> sub <guid> <N> <M1> <key_11> <op_11> <type_11> <val_11> ... \
 *                            <key_1M1> <op_1M1> <val_1M1> ...        \
 *                       <MN> <key_N1> <op_N1> <type_N1> <val_N1> ... \
 *                            <key_NMN> <op_NMN> <val_NMN>
 *
 * <time> create <creator> <createe> <N> <key_1> <type_1> <val_1> ... \ 
 *                                       <key_N> <type_N> <val_N>
 * <time> destroy <guid>
 */
void God::loadTrace(const char *file) {
    if ( pthread_mutex_init(&queueLock, NULL) != 0) {
	Debug::die("can't create pthread mutex");
    }
    loaderArgs.file = file;
    loaderArgs.timeMultiplier = timeMultiplier;
    loaderArgs.lock = &queueLock;
    loaderArgs.proceed = &eventCondVar;
    loaderArgs.eventQueue = &eventQueue;
    loaderArgs.maxTime = 0;
    loaderArgs.finished = false;

    int ret = pthread_create(&loaderThread, NULL, traceLoader, &loaderArgs);
    ASSERT(ret == 0);

#ifdef SYNCH_TRACE_LOAD
    // wait for loading to be complete
    void *rs;
    ret = pthread_join(loaderThread, &rs);
    ASSERT(ret == 0);
#else
    // let it load a little before starting
    usleep(500000);
#endif
}

void *traceLoader(void *arg) {
    LoaderArgs *args = (LoaderArgs *)arg;
    const char *file = args->file;

    ifstream in(file);

    if (!in) {
	Debug::die("can't open workload file '%s'", file);
    }

    int lineno = 0;
    gtime_t time;
    char type_name[LINEBUF_SIZE];
    int  type;

    while (!in.eof()) {
	lineno++;
	//cerr << lineno << endl;

	if (in.peek() == EOF) break;

	READ(time);

	// rescale trace time to what we want
	time = time * args->timeMultiplier;

	READ(type_name);
	if (! strcmp(type_name, "interest") ) {
	    type = EV_INTEREST;
	} else if (! strcmp(type_name, "mod") ) {
	    type = EV_MOD;
	} else if (! strcmp(type_name, "sub") ) {
	    type = EV_SUB;
	} else if (! strcmp(type_name, "create") ) {
	    type = EV_CREATE;
	} else if (! strcmp(type_name, "destroy") ) {
	    type = EV_DESTROY;
	} else {
	    type = -1;
	}

	TraceEvent *ev = new TraceEvent(time, (tev_t)type);
	ev->lineno = lineno;

	switch(type) {
	case EV_INTEREST: {
	    READ(ev->origin);
	    int sz;
	    READ(sz);
	    for (int i=0; i<sz; i++) {
		GUID interest;
		READ(interest);
		ev->interests.push_back(interest);
	    }
	    break;
	}
	case EV_MOD: {
	    READ(ev->origin);
	    READ(ev->target);
	    int sz;
	    READ(sz);
	    for (int i=0; i<sz; i++) {
		attrkey_t key;
		Value val;
		READ(key);
		READ(val);
		ev->attributes[key] = val;
	    }
	    size_t size;
	    READ(ev->size);
	    break;
	}
	case EV_SUB: {
	    READ(ev->origin);
	    int sz;
	    READ(sz);
	    for (int i=0; i<sz; i++) {
		Interest *sub = new Interest();
		int num;
		READ(num);
		for (int j=0; j<num; j++) {
		    attrkey_t key;
		    OperatorType op;
		    Value val;
		    READ(key);
		    READ(op);
		    READ(val);
		    sub->AddConstraint(key, op, val);
		}
		ev->subs.insert(sub);
	    }
	    break;
	}
	case EV_CREATE: {
	    READ(ev->origin);
	    READ(ev->target);
	    int sz;
	    READ(sz);
	    for (int i=0; i<sz; i++) {
		attrkey_t key;
		Value val;
		READ(key);
		READ(val);
		ev->attributes[key] = val;
	    }
	    break;
	}
	case EV_DESTROY: {
	    READ(ev->origin);
	    break;
	}
	default:
	    Debug::die("unknown trace event type: %d\n", type);
	}

	in.ignore(LINEBUF_SIZE, '\n');

	LOCK(args->lock);

	WEvent *prev = args->eventQueue->empty()?NULL:args->eventQueue->top();
	args->eventQueue->push(ev);
	args->maxTime = time;

	UNLOCK(args->lock);

	// If the next event in the queue was a SystemEvent, then the consumer
	// is waiting to make sure that all TraceEvents that are before the
	// SystemEvent in time are loaded, so we must signal them to proceed
	// when this condition is fulfilled...
	// WARNING: this assumes the trace events are in chronological order!
	if (prev != NULL && prev->cls == WEvent::SYSTEM_EVENT && 
	    time >= prev->time) {
	    int ret = pthread_cond_signal( args->proceed );
	    ASSERT(ret == 0);
	}

	if (args->eventQueue->size() > MAX_QUEUE_SIZE) {
	    sleep(QUEUE_WAIT);
	}

    }

    in.close();

    LOCK(args->lock);

    args->finished = true;

    UNLOCK(args->lock);

    int ret = pthread_cond_signal( args->proceed );
    ASSERT(ret == 0);

    return NULL;
}

///////////////////////////////////////////////////////////////////////////////

void God::loadInit(const char *file) {
    ifstream in(tracefile);

    if (!in) {
	Debug::die("can't open workload file '%s'", tracefile);
    }

    int  lineno = 0;

    while (!in.eof()) {
	lineno++;

	if (in.peek() == EOF) break;
	GUID guid;
	SID  sid;
	READ(guid);
	READ(sid);
	in.ignore(LINEBUF_SIZE, '\n');

	initMap[guid] = sid;
    }

    in.close();
}

Server *God::pickInitialServer(init_t method, GUID guid) {
    ASSERT(servers.size() > 0);

    switch(method) {
    case INIT_FILE: {
	// predefined starting locations
	InitMapIter iter = initMap.find(guid);
	ASSERT(iter != initMap.end());
	ServerMapIter sp = servers.find(iter->second);
	ASSERT(sp != servers.end());
	ASSERT(sp->second != NULL);
	return sp->second;
    }
    case INIT_RANDOM: {
	// random starting locations
	int index = (int)( drand48()*(double)servers.size() );
	ServerMapIter iter = servers.begin();
	while (index-- > 0)
	    iter++;
	ASSERT(iter->second != NULL);
	return iter->second;
    }
    case INIT_SMART: {
	// create on server with minimum number of clients
	int min = 999999999;
	Server *argmin = NULL;
	Server *curr   = getCurrServer();
	for (ServerMapIter i = servers.begin(); i != servers.end(); i++) {
	    Server *s = i->second;
	    setCurrServer(s);
	    if (s->getNumClients() < min) {
		min = s->getNumClients();
		argmin = s;
	    }
	    setCurrServer(curr);
	}
	ASSERT(argmin != NULL);
	return argmin;
    }
    default:
	Debug::die("unknown init method: %d", method);
    }
    return NULL;
}

///////////////////////////////////////////////////////////////////////////////

God::God(gtime_t startTime, uint32 timeMultiplier, const char *tracefile, 
	 init_t initMethod, const char *initialfile) : 
    startTime(startTime), currTime(startTime), timeMultiplier(timeMultiplier),
    tracefile(tracefile), initMethod(initMethod), initialfile(initialfile), 
    currEvent(NULL), currServer(NULL) {
}

God::~God() {
    LOCK(&queueLock);
    while (! eventQueue.empty() ) {
	WEvent  *ev = eventQueue.top();
	eventQueue.pop();
	delete( ev );
    }
    UNLOCK(&queueLock);
}

void God::registerProto(SimProtocolLayer *proto) {
    this->proto = proto;
}

gtime_t God::getStartTime() { return startTime; }

gtime_t God::getCurrTime() { return currTime; }

void God::setCurrTime(gtime_t time) { currTime = time; }

WEvent *God::getCurrEvent() { return currEvent; };

void God::setCurrEvent(WEvent *ev) { currEvent = ev; }

Server *God::getCurrServer() { return currServer; }

void God::setCurrServer(Server *serv) { currServer = serv; }

Server *God::locateObj(GObject *obj) {
    return locateObj(obj->guid);
}

Server *God::locateObj(GUID guid) {
    Server *ret = objServerMap[guid];
    ASSERT(ret != NULL);
    GUID_DEBUG_MSG(guid, "locating %d at %d",
		   guid.localOID, ret->sid.addr);
    return ret;
}

void God::objMoved(GUID guid, SID sid) {
    ServerMapIter iter = servers.find(sid);
    ASSERT(iter != servers.end());
    ASSERT(objServerMap[guid] != iter->second);
    GUID_DEBUG_MSG(guid, "moved guid %d from %d to %d",
		   guid.localOID, locateObj(guid)->sid.addr, sid.addr);

    objServerMap[guid] = iter->second;
}

GUIDSet& God::getInterested(GObject *obj) {
    return getInterested(obj->guid);
}

GUIDSet& God::getInterested(GUID guid) {
    return interestedMap[guid];
}

Server *God::getServer(SID sid) {
    Server *ret = servers[sid];
    ASSERT(ret != NULL);
    return ret;
}

void God::addServer(Server *server) {
    servers[server->sid] = server;
    proto->addServer(server);
}

void God::updateInterests(GUID guid, GUIDVec& interests) {
    // remove old interests
    InterestsMapIter old = interestsMap.find(guid);
    ASSERT(old != interestsMap.end());
    for (GUIDSetIter i = old->second.begin(); i != old->second.end(); i++) {
	InterestedMapIter iter = interestedMap.find(*i);
	// this object was destroyed in the in-term, so ignore it...
	if (iter == interestedMap.end()) {
	    continue;
	}
	iter->second.erase(guid);
    }
    old->second.clear();

    GUID_DEBUG_MSG(guid, "updating interests to (these are interested):");

    // add new interests
    old->second.insert<GUIDVecIter>(interests.begin(), interests.end());
    for (GUIDVecIter i = interests.begin(); i != interests.end(); i++) {
	InterestedMapIter iter = interestedMap.find(*i);
	ASSERTM(iter != interestedMap.end(), 
		"guid %d is interested in non-existant object %d",
		guid.localOID, i->localOID);
	GUID_DEBUG_RUN(guid, cerr << *i << endl);

	iter->second.insert(guid);
    }

}

void God::createObject(GUID guid, Server *server) {
    GUIDSet empty;
    ASSERT(interestsMap.find(guid) == interestsMap.end());
    ASSERT(interestedMap.find(guid) == interestedMap.end());

    interestsMap.insert(InterestsMap::value_type(guid, empty));
    interestedMap.insert(InterestedMap::value_type(guid, empty));
    objServerMap[guid] = server;

    GUID_DEBUG_MSG(guid, "created object %d on server %d", 
		   guid.localOID, server->sid.addr);
}

void God::destroyObject(GUID guid) {
    InterestsMapIter old1 = interestsMap.find(guid);
    ASSERT(old1 != interestsMap.end());
    for (GUIDSetIter i = old1->second.begin(); 
	 i != old1->second.end(); i++) {
	InterestedMapIter i2 = interestedMap.find(*i);
	ASSERT(i2 != interestedMap.end());
	i2->second.erase(guid);
    }
    interestsMap.erase(old1);

    InterestedMapIter old2 = interestedMap.find(guid);
    ASSERT(old2 != interestedMap.end());
    for (GUIDSetIter i = old2->second.begin(); 
	 i != old2->second.end(); i++) {
	InterestsMapIter i2 = interestsMap.find(*i);
	ASSERT(i2 != interestsMap.end());
	i2->second.erase(guid);
    }
    interestedMap.erase(old2);

    ObjServerMapIter iter = objServerMap.find(guid);
    ASSERT(iter != objServerMap.end());
    objServerMap.erase(iter);
}

WEvent *God::nextEvent() {
    WEvent *ev;
    gtime_t maxTime;

    LOCK(&queueLock);

    ev = eventQueue.top();
    if (!loaderArgs.finished && ev->cls == WEvent::SYSTEM_EVENT &&
	ev->time > loaderArgs.maxTime) {
	int ret = pthread_cond_wait( &eventCondVar, &queueLock );
	ASSERT(ret == 0);

	// NOTE: once the condition is fulfilled, it can not be made false
	// again because we assume all events proceed forward in time;
	// hence there is no need to recheck the condition
	ev = eventQueue.top();
	ASSERT(ev->time <= loaderArgs.maxTime);
    }

    eventQueue.pop();

    UNLOCK(&queueLock);

    return ev;
}

bool God::moreEvents() {
    LOCK(&queueLock);

    while (eventQueue.empty() && !loaderArgs.finished) {
	UNLOCK(&queueLock);
	// Could use a condition variable here, but we probably want to avoid
	// getting into a lock-step situation where consumer/producer run
	// alternately for waiting for single events. Just let loading happen
	// for a while if the consumer has caught up (we know loading is
	// relatively fast)
	usleep(100000);
	LOCK(&queueLock);
    }

    bool ret = !eventQueue.empty();

    UNLOCK(&queueLock);

    return ret;
}

void God::run(gtime_t stopTime) {

    // load the required files...
    loadTrace(tracefile);
    if (initMethod == INIT_FILE) loadInit(initialfile);

    // start the simulation...
    debug("starting simulation...");

    for (ServerMapIter i = servers.begin(); i != servers.end(); i++) {
	WorkloadApp *app = dynamic_cast<WorkloadApp *>(i->second->getApp());
	setCurrServer( i->second );
	app->initWorld();
    }

    int iter = 0;

    cout << "sim time:       ";
    cout.flush();

    do {

	START(BENCH_ITERATION);
	START(BENCH_PQUEUE);

	WEvent  *ev = nextEvent();
	setCurrTime  ( ev->time );
	setCurrEvent ( ev );

	STOP(BENCH_PQUEUE);

	if (iter++ % 1000 == 0) { 
	    char buf[10];
	    sprintf(buf, "%6llu", ev->time);
	    cout << "\b\b\b\b\b\b" << buf;
	    cout.flush();
	    //cout << "."; cout.flush();
	}

	Server *server;
	if (ev->cls == WEvent::TRACE_EVENT) {
	    TraceEvent *tev = dynamic_cast<TraceEvent *>(ev);
	    //debug("processing TraceEvent: ");
	    //DEBUG_RUN(cerr << "  " << tev << endl);

	    GUID_DEBUG_MSG(tev->origin, "TraceEvent[orirgin]: ");
	    GUID_DEBUG_RUN(tev->origin, cerr << tev << endl);
	    GUID_DEBUG_MSG(tev->target, "TraceEvent[target]: ");
	    GUID_DEBUG_RUN(tev->target, cerr << tev << endl);

	    if ( tev->type == EV_INTEREST ) {

		ASSERTM(objServerMap.find(tev->origin) != 
			objServerMap.end(),
			"origin of interest does not exist");

		START(BENCH_INTEREST);

		updateInterests(tev->origin, tev->interests);

		STOP(BENCH_INTEREST);

	    } else {
		switch(tev->type) {
		case EV_CREATE: {
		    ASSERTM(tev->origin == GUID_NONE ||
			    objServerMap.find(tev->origin) != 
			    objServerMap.end(),
			    "creator does not exist");
		    ASSERTM(objServerMap.find(tev->target) == 
			    objServerMap.end(),
			    "createe already exists!");
		    START(BENCH_CREATE);
		    break;
		}
		case EV_DESTROY: {
		    ASSERTM(objServerMap.find(tev->origin) != 
			    objServerMap.end(),
			    "destroyed object does not exist");
		    START(BENCH_DESTROY);
		    break;
		}
		case EV_MOD: {
#ifdef DEBUG
		    ASSERTM(objServerMap.find(tev->origin) != 
			    objServerMap.end(),
			    "origin of mod does not exist");
		    ASSERTM(objServerMap.find(tev->target) != 
			    objServerMap.end(),
			    "target of mod does not exist");
		    InterestsMapIter iter = interestsMap.find(tev->origin);
		    ASSERT(iter != interestsMap.end());
		    ASSERTM(tev->origin == tev->target ||
			    iter->second.find(tev->target) != 
			    iter->second.end(),
			    "origin modifying object it is not interested in");
		    InterestedMapIter iter2 = interestedMap.find(tev->target);
		    ASSERT(iter2 != interestedMap.end());
		    ASSERT(tev->origin == tev->target ||
			   iter2->second.find(tev->origin) !=
			   iter2->second.end());
		    START(BENCH_MOD);
#endif
		    break;
		}
		case EV_SUB: {
		    ASSERTM(objServerMap.find(tev->origin) != 
			    objServerMap.end(),
			    "origin of sub does not exist");
		    START(BENCH_SUB);
		    break;
		}
		default:
		    Debug::die("can't get here");
		}

		if (tev->type == EV_CREATE &&
		    tev->origin == GUID_NONE) {
		    init_t method;
		    // load balance player objects
		    // place others randomly for even distribution
		    Value type = tev->attributes[TYPE];
		    ASSERT(type.m_Type != ATTR_INVALID);
		    if ( *(type.m_Sval) == PLAYER )
			method = initMethod;
		    else
			method = INIT_RANDOM;
		    server = pickInitialServer( method, tev->target );
		    GUID_DEBUG_MSG(tev->target, "executing target is on %d",
				   server->sid.addr);
		} else {
		    server = locateObj( tev->origin );
		    GUID_DEBUG_MSG(tev->origin, "executing origin is on %d",
				   server->sid.addr);
		}

		if ( tev->type == EV_CREATE ) {
		    createObject(tev->target, server);
		} else if ( tev->type == EV_DESTROY ) {
		    destroyObject(tev->origin);
		}
		setCurrServer( server );
		WorkloadApp *app = 
		    dynamic_cast<WorkloadApp *>(server->getApp());
		app->handleTraceEvent(tev);

		switch(tev->type) {
		case EV_CREATE:
		    STOP(BENCH_CREATE);
		    break;
		case EV_DESTROY:
		    STOP(BENCH_DESTROY);
		    break;
		case EV_MOD:
		    STOP(BENCH_MOD);
		    break;
		case EV_SUB:
		    STOP(BENCH_SUB);
		    break;
		default:
		    Debug::die("can't get here");
		}
	    }
	    delete( tev );
	} else {
	    SystemEvent *sev = dynamic_cast<SystemEvent *>(ev);
	    //debug("processing SystemEvent: ");
	    //DEBUG_RUN(cerr << "  " << sev << endl);

	    server = getServer( sev->sid );
	    setCurrServer( server );

	    switch(sev->type) {
	    case EV_TIMER: {
		// FIXME: this should go to the protocol layer and upcall
		// to the Server, not down from the app

		switch(sev->timer_type) {
		case TIMER_INTEREST:
		    START(BENCH_TIMER_INTEREST);
		    break;
		case TIMER_MIGRATE:
		    START(BENCH_TIMER_MIGRATE);
		    break;
		default:
		    Debug::die("can't get here");
		}

		WorkloadApp *app = 
		    dynamic_cast<WorkloadApp *>(server->getApp());
		app->handleSystemEvent(sev);

		switch(sev->timer_type) {
		case TIMER_INTEREST:
		    STOP(BENCH_TIMER_INTEREST);
		    break;
		case TIMER_MIGRATE:
		    STOP(BENCH_TIMER_MIGRATE);
		    break;
		default:
		    Debug::die("can't get here");
		}

		break;
	    }
	    case EV_MIGRATE: {

		GUID_DEBUG_MSG(sev->obj->guid, "SystemEvent[migrate]: ");
		GUID_DEBUG_RUN(sev->obj->guid, cerr << sev << endl);

		START(BENCH_MIGRATE);

		proto->handleMigrate(sev);

		STOP(BENCH_MIGRATE);

		break;
	    }
	    default:
		Debug::die("unknown system event type %d", sev->type);
	    }

	    delete( sev );
	}

	STOP(BENCH_ITERATION);

    } while (moreEvents() && getCurrTime() <= stopTime);
    cout << endl;

    debug("ending simulation...");

    Benchmark::print();

    return;
}

void God::addEvent(SystemEvent *ev) {
    if (ev->type == EV_MIGRATE) {
	GUID_DEBUG_MSG(ev->obj->guid, "inserting migrate event:");
	GUID_DEBUG_RUN(ev->obj->guid, cerr << ev << endl);
    }

    LOCK(&queueLock);
    eventQueue.push(ev);
    UNLOCK(&queueLock);
}

///////////////////////////////////////////////////////////////////////////////

void God::dumpEventQueue(ostream& out) {
    LOCK(&queueLock);
    while (!eventQueue.empty()) {
	printEventQueueTop(out);
	out << "\n";
	eventQueue.pop();
    }
    UNLOCK(&queueLock);
}

// TODO: this function not thread safe by itself!
void God::printEventQueueTop(ostream& out) {
    WEvent  *ev = eventQueue.top();
    if (ev->cls == WEvent::TRACE_EVENT) {
	out << dynamic_cast<TraceEvent *>(ev);
    } else {
	out << dynamic_cast<SystemEvent *>(ev);
    }
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
