////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

using namespace std;

#include <boost/random.hpp>
#include <fstream>
#include <iostream>
#include <sstream>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <Colyseus.h>
#include <gameapi/common.h>

real32 gMapX, gMapY, gMapHeight;
real32 gGoalProb;
real32 gWayPointRadius;
uint32 gNumBuildings;
real32 gBuildingSizeMean, gBuildingSizeStd;
char   gOutputPrefix[255];

static OptionType g_Options[] =
    {
	{ 'x', "xmax", OPT_FLT, 
	  "size of X size of the map",
	  &gMapX, "1300", NULL },
	{ 'y', "ymax", OPT_FLT, 
	  "size of Y size of the map (if not given, equal to X size)",
	  &gMapY, "-1", NULL },

	{ 'h', "height", OPT_FLT, 
	  "height of world", 
	  &gMapHeight, "16", NULL },
	{ 'p', "goalprob", OPT_FLT, 
	  "prob of a waypoint being a goal (unif. dist.)", 
	  &gGoalProb, "0.05", NULL },
	{ 'r', "radius", OPT_FLT, 
	  "waypoint radius", 
	  &gWayPointRadius, "32", NULL },
	{ 'n', "buildings", OPT_INT, 
	  "number of buildings on the map", 
	  &gNumBuildings, "8", NULL }, // 2 per player
	{ 'm', "building-mean", OPT_FLT, 
	  "mean building width (normally dist.)", 
	  &gBuildingSizeMean, "650", NULL }, // ~ AoiSize in quake2
	{ 't', "building-std", OPT_FLT, 
	  "stddev building width (normally dist.)", 
	  &gBuildingSizeStd, "400", NULL }, // ~ AoiSize std in quake2
	{ 'o', "output", OPT_STR, 
	  "output prefix", 
	  gOutputPrefix, "map", NULL },
	{ 0, 0, 0, 0, 0, 0, 0 }
    };

int main(int argc, char **argv)
{
    argc = ProcessOptions(g_Options, argc, argv, false);
    DBG_INIT(&g_LocalSID);

    if (gMapY < 0) 
	gMapY = gMapX;

    boost::mt19937 rng;
    boost::normal_distribution<> buildingSizeDist(gBuildingSizeMean,
						  gBuildingSizeStd);
    boost::variate_generator<boost::mt19937, 
	boost::normal_distribution<> > buildingSizeGen(rng, buildingSizeDist);

    if ( system( (string("mkdir -p ") + string(gOutputPrefix)).c_str() ) )
	Debug::die("can't create map directory");

    string dims_file = string(gOutputPrefix) + "/dims";
    ofstream dims( dims_file.c_str() );
    dims << gMapX << " " << gMapY << " " << gMapHeight << endl;
    dims.close();

    string buildings_file = string(gOutputPrefix) + "/buildings";
    ofstream buildings( buildings_file.c_str() );
    if (!buildings) {
	Debug::die("can't create buildings file");
    }

    BBox world(Vec3(0,0,-gMapHeight/2),
	       Vec3(gMapX,gMapY,gMapHeight/2));

    for (uint32 i=0; i<gNumBuildings; i++) {
	// where the building is centered
	Vec3 center( drand48()*gMapX,
		     drand48()*gMapY,
		     0 );

	// width and height
	real32 width = buildingSizeGen();
	// some artificial caps
	if (width < 1) 
	    width = 1;
	if (width > gBuildingSizeMean + 3*gBuildingSizeStd)
	    width = gBuildingSizeMean + 3*gBuildingSizeStd;
	real32 height = buildingSizeGen();
	// some artificial caps
	if (height < 1) 
	    height = 1;
	if (height > gBuildingSizeMean + 3*gBuildingSizeStd)
	    height = gBuildingSizeMean + 3*gBuildingSizeStd;

	Vec3 b[4];
	b[0] = Vec3(-width/2, -height/2, 0);
	b[1] = Vec3(-width/2, height/2, 0);
	b[2] = Vec3(width/2, height/2, 0);
	b[3] = Vec3(width/2, -height/2, 0);

	real32 rotate = M_PI*drand48();
	for (uint32 i=0; i<4; i++) {
	    // rotate the building a random amount and translate to center
	    b[i] = b[i].RotateAroundVector(Vec3(0,0,1), rotate) + center;
	}

	// TODO: prevent building overlap??

	for (int i=0; i<4; i++)
	    buildings << b[i][0] << " " << b[i][1] << endl;
    }

    // last box frames the map
    buildings << world.min[0] << " " << world.min[1] << endl;
    buildings << world.min[0] << " " << world.max[1] << endl;
    buildings << world.max[0] << " " << world.max[1] << endl;
    buildings << world.max[0] << " " << world.min[1] << endl;

    buildings.close();

    stringstream cmd;
    cmd << "./voronoi " << buildings_file << " | ./waypoint_filter.pl"
	<< " -x " << gMapX << " -y " << gMapY << " -r " << gWayPointRadius
	<< " -g " << gGoalProb << " > " << gOutputPrefix << "/waypoints";

    system( cmd.str().c_str() );

    return 0;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
