////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////
#include <Mercury.h>
#include <om/OMMessage.h>
#include <om/GameAdaptor.h>
#include <om/Manager.h>
#include <om/ReplicaConnection.h>

static float  loss_rate;
static uint32 send_interval;
static uint32 delta_size;
static uint32 nobjects;

static IPEndPoint me;
static IPEndPoint to;

OptionType opts[] = {
    {'/', "loss_rate", OPT_FLT, "",
     &loss_rate, "0.01", NULL},
    {'/', "send_interval", OPT_INT, "",
     &send_interval, "100", NULL},
    {'/', "delta_size", OPT_INT, "",
     &delta_size, "100", NULL},
    {'/', "nobjects", OPT_INT, "",
     &nobjects, "20", NULL},

    {0, 0, 0, 0, 0, 0, 0}
};

///////////////////////////////////////////////////////////////////////////////

static DeltaMask init_mask(1);

struct TestObject : public GObject
{
    TestObject(guid_t guid) : GObject(guid) {}
    TestObject(GObjectInfoIface *info) : GObject(info) {}
    virtual ~TestObject() {}

    virtual const DeltaMask& GetInitDeltaMask() {
	return init_mask;
    }


    virtual bool IsNew() { return false; }
    virtual void FillEvents(EventList *reg,
			    EventList *unreg,
			    EventSet  *curr) {
	return;
    }
    virtual uint32 InterestTime(OMEvent *ev, InterestMap *curr) {
	return 0;
    }
    virtual uint32 InterestTime(GObject *obj, InterestMap *curr) {
	return 0;
    }
    virtual void PackUpdate(Packet *buffer, const DeltaMask& mask) {
	if (mask.IsSet(0))
	    for (uint32 i=0; i<delta_size; i++)
		buffer->WriteByte(0);
    }
    virtual void UnpackUpdate(Packet *buffer, const DeltaMask& mask, 
			      SIDMap *unresolved) {
	if (mask.IsSet(0))
	    for (uint32 i=0; i<delta_size; i++)
		buffer->ReadByte();
    }

    void SetDirty() {
	SetDeltaMaskField(0);
    }
};

class TestObjectStore : public ObjectStore
{
    friend class TestGame;
private:
    map<GUID, GObject *, less_GUID> m;
    map<GUID, GObject *, less_GUID>::iterator it;
public:
    virtual void Begin() {
	it = m.begin();
    }

    virtual GObject *Next() {
	if (it == m.end())
	    return NULL;
	GObject *ret = it->second;
	it++;
	return ret;
    }

    virtual GObject *Find(guid_t guid) {
	map<GUID, GObject *, less_GUID>::iterator ptr = m.find(guid);
	if (ptr != m.end())
	    return ptr->second;
	return NULL;
    }

protected:
    virtual void _ManagerAdd(GObject *obj) {
	m[obj->GetGUID()] = obj;
    }

    virtual void _ManagerRemove(guid_t guid) {
	m.erase(guid);
    }

    virtual void _ApplicationAdd(GObject *obj) {
	m[obj->GetGUID()] = obj;
    }
    virtual void _ApplicationRemove(guid_t guid) {
	m.erase(guid);
    }
};

struct TestGame : public GameAdaptor
{
    TestObjectStore *s, *p;

    TestGame() {
	s = new TestObjectStore();
	p = new TestObjectStore();
    }

    virtual uint32 GetSendInterval() {
	return send_interval;
    }

    virtual ObjectStore *GetObjectStore() { return s; }
    virtual ObjectStore *GetPendingStore() { return p; }

    virtual void FillInterests(InterestList *reg,
			       InterestList *unreg,
			       GUIDMap *reg_assoc,
			       InterestMap *curr) {}

    virtual GObject *Construct(GObjectInfoIface *info,
			       Packet *pkt, 
			       const DeltaMask& mask,
			       SIDMap *unresolved) {
	TestObject *obj = new TestObject(info);
	obj->UnpackUpdate(pkt, mask, unresolved);
	return obj;
    }

    virtual void Destroy(GUID guid) {
	GObject *obj;
	if ((obj = s->Find(guid)) != NULL)
	    delete obj;
	if ((obj = p->Find(guid)) != NULL)
	    delete obj;

	s->_ApplicationRemove(guid);
	p->_ApplicationRemove(guid);
    }
};

class TestManager : public Manager
{
protected:
    ReplicaConnectionSender *s;
    ReplicaConnectionReceiver *r;
public:
    TestManager() : Manager() {
	m_Game = new TestGame();
	m_PubSubRouter = WANMercuryNode::GetInstance(g_Preferences.port);
	m_DirectRouter = DirectRouter::GetInstance(me, this);
	s = new ReplicaConnectionSender(to, this);
	r = new ReplicaConnectionReceiver(to, 0, 0, this);

	m_UniqueGUID.m_IP       = g_LocalSID.GetIP();
	m_UniqueGUID.m_Port     = g_LocalSID.GetPort();

	Manager::m_Instance = this;
    }

    void AddObj() {
	GetObjectStore()->ApplicationAdd(new TestObject(CreateGUID()));
    }

    void SendMsg(MsgApp *msg)
    {
	if (drand48() < loss_rate) {
	    // drop!
	    DBG << "dropped: " << msg << endl;
	    return;
	}
	DBG << "sent: " << msg << endl;
	m_DirectRouter->SendMsg(&to, msg);
    }

    void DoWork() {
	static bool isnew = false;

	// mark all objects dirty and register all to be sent
	ObjectStore *st = GetObjectStore();
	TestObject *o;
	st->Begin();
	while ((o = dynamic_cast<TestObject *>(st->Next())) != NULL) {
	    s->RegisterGUID(o->GetGUID(), false, isnew);
	    isnew = false;
	    o->SetDirty();
	}

	{
	    //DBG << s << endl;
	    MsgAppUpdate *msg = s->CreateUpdate();
	    if (msg != NULL) {
		SendMsg(msg);
	    }
	}

	m_DirectRouter->DoWork(10);

	MsgApp *msg;

	while ((msg = m_DirectRouter->ReadMsg()) != NULL) {
	    if (msg->GetType() == MSG_APP_UPDATE) {
		r->HandleUpdate(dynamic_cast<MsgAppUpdate *>(msg));

		MsgAppUpdateAck ack = MsgAppUpdateAck();
		ack.epoch    = r->GetEpoch();
		ack.ackSeqno = r->GetLastSeqNoAck();

		// xxx debug
		OS::GetCurrentTime(&ack.timestamp);
		SendMsg(&ack);

		delete msg;
	    } else if (msg->GetType() == MSG_APP_UPDATE_ACK) {
		s->HandleUpdateAck(dynamic_cast<MsgAppUpdateAck *>(msg));
		delete msg;
	    } else {
		WARN << "unknown msg type: " << msg << endl;
		ASSERT(0);
	    }
	}
    }
};

int main(int argc, char **argv)
{
    InitializeColyseus(&argc, argv, opts);

    if (argc != 3) {
	WARN << "usage: repconntest <this_addr> <other_addr>" << endl;
	return 1;
    }

    me = IPEndPoint(argv[1]);
    to = IPEndPoint(argv[2]);
    g_LocalSID = me;

    TestManager *m = new TestManager();

    for (uint32 i=0; i<nobjects; i++)
	m->AddObj();

    sleep(5);

    INFO << "starting!" << endl;

    TimeVal now, next;
    OS::GetCurrentTime(&next);
    while (1) {
	OS::GetCurrentTime(&now);
	PERIODIC2(5000, now, {
	    Benchmark::print();
	});

	//INFO << " now=" << now << " next=" << next << endl;
	if (now < next) {
	    RealNet::DoWork(next - now);
	    OS::GetCurrentTime(&now);
	    if (now < next)
		usleep((next - now)*USEC_IN_MSEC);
	} else {
	    RealNet::DoWork(0);
	}
	next = next + send_interval;
	if (now > next) {
	    next = now; // too far off, reset
	}

	m->DoWork();
    }

    return 0;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
