////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

#include <om/common.h>
#include <util/debug.h>
#include "TestValue.h"
#include <mercury/Packet.h>
#include <cmath>

#define EPSILON 0.000001

#define EXCEPTION(s) { WARN << s << endl; ASSERT(0); }
#define MercWarning(s,...) { WARN << s << endl; }

//
// TODO: string comparisons are screwed up! GUID and SID comparisons are not implemented yet.
//

TestValue::TestValue(uint32 i)
{
    m_Type = ATTR_UINT;
    m_Ival = i;
}

TestValue::TestValue(char c)
{
    m_Type = ATTR_CHAR;
    m_Cval = c;
}

TestValue::TestValue(float f)
{
    m_Type = ATTR_FLOAT;
    m_Fval = f;
}

TestValue::TestValue(string s)
{
    m_Type = ATTR_STRING;
    m_Sval = new string(s);
}

TestValue::TestValue(char *s)
{
    m_Type = ATTR_STRING;
    m_Sval = new string(s);
}

TestValue::TestValue(guid_t g)
{
    m_Type = ATTR_GUID;
    m_GUIDval = new GUID(g);
}

TestValue::TestValue(sid_t s)
{
    m_Type = ATTR_SID;
    m_SIDval = new SID(s);
}

TestValue::TestValue(guid_t guid, sid_t primary_loc)
{
    m_Type = ATTR_IDBUNDLE;
    m_IDval.guid = new GUID(guid);
    m_IDval.sid  = new SID(primary_loc);
}

TestValue::TestValue(size_t len, void *data)
{
    m_Type = ATTR_BLOB;
    m_Bval.len = len;
    m_Bval.data = malloc(len);
    ASSERT(m_Bval.data);
    memcpy(m_Bval.data, data, len);
}

TestValue::TestValue(const TestValue & oval)
{
    m_Type = ATTR_UINT;
    m_Sval = 0;
    //	ASSERT(oval.m_Type != ATTR_LAZYBLOB);
    _AssignMembers(oval);
}

TestValue::TestValue(Packet * pkt)
{
    //	MercDebug(1, "Constructing value out of a packet!\n");
    //	pkt->DumpBuffer(pkt->m_BufPosition, 10);
    //	MercDebug(1, "\n");

    m_Type = (AttrType) pkt->ReadByte();

    DB(50) << "constructing value of type : " << g_AttrStrings[m_Type] << endl;
    switch (m_Type) {
    case ATTR_UINT:
	m_Ival = pkt->ReadInt();
	break;
    case ATTR_CHAR:
	m_Cval = (char) pkt->ReadByte();
	break;
    case ATTR_FLOAT:
	m_Fval = pkt->ReadFloat();
	break;
    case ATTR_STRING:
	m_Sval = new string();
	pkt->ReadString(*m_Sval);
	break;
    case ATTR_GUID:
	m_GUIDval = new GUID(pkt);
	break;
    case ATTR_SID:
	m_SIDval = new SID(pkt);
	break;
    case ATTR_IDBUNDLE:
	m_IDval.guid = new GUID(pkt);
	m_IDval.sid  = new SID(pkt);
	break;
    case ATTR_BLOB:
	m_Bval.len = pkt->ReadInt();
	m_Bval.data = malloc(m_Bval.len);
	ASSERT(m_Bval.data);
	pkt->ReadBuffer((byte *) m_Bval.data, m_Bval.len);
	break;
    default:
	//F_EXCEPTION("TestValue: badly serialized value... %d \n", m_Type);
	ASSERT(false);
	break;
    }
}

void TestValue::Serialize(Packet * pkt)
{
    pkt->WriteByte(m_Type);

    switch (m_Type) {
    case ATTR_UINT:
	pkt->WriteInt(m_Ival);
	break;
    case ATTR_CHAR:
	pkt->WriteByte((byte) m_Cval);
	break;
    case ATTR_FLOAT:
	pkt->WriteFloat(m_Fval);
	break;
    case ATTR_STRING:
	pkt->WriteString(*m_Sval);
	break;
    case ATTR_GUID:
	m_GUIDval->Serialize(pkt);
	break;
    case ATTR_SID:
	m_SIDval->Serialize(pkt);
	break;
    case ATTR_IDBUNDLE:
	m_IDval.guid->Serialize(pkt);
	m_IDval.sid->Serialize(pkt);
	break;
    case ATTR_BLOB:
	//fprintf(stderr, "writing blob to buffer, length is %d\n", m_Bval.len);
	pkt->WriteInt(m_Bval.len);
	pkt->WriteBuffer((byte *) m_Bval.data, m_Bval.len);
	break;
    default:
	WARN << "TestValue::Serialize(): my state is corrupt! :-(( \n";
	break;
    }
    //MercDebug(1, "TestValue serialized form: ");
    //pkt->DumpBuffer(pkt->m_BufPosition - GetLength(), GetLength());
    //MercDebug(1, "\n");
}

// copying by value for TestValue is REALLY BAD. Because of this blob re-allocing.
void TestValue::_AssignMembers(const TestValue & oval)
{
    TestValue & tval = (TestValue &) oval;

    if (m_Type == ATTR_STRING && m_Sval)
	delete m_Sval;
    else if (m_Type == ATTR_BLOB && m_Bval.data)
	free(m_Bval.data);

    m_Type = tval.m_Type;

    switch (m_Type) {
    case ATTR_UINT:
	m_Ival = tval.m_Ival;
	break;
    case ATTR_CHAR:
	m_Cval = tval.m_Cval;
	break;
    case ATTR_FLOAT:
	m_Fval = tval.m_Fval;
	break;
    case ATTR_GUID:
	m_GUIDval = new GUID(*(tval.m_GUIDval));
	break;
    case ATTR_SID:
	m_SIDval = new SID(*(tval.m_SIDval));
	break;
    case ATTR_IDBUNDLE:
	m_IDval.guid = new GUID(*(tval.m_IDval.guid));
	m_IDval.sid  = new SID(*(tval.m_IDval.sid));
	break;
    case ATTR_STRING:
	m_Sval = new string(*(tval.m_Sval));
	break;
    case ATTR_BLOB:
	{
	    m_Bval.len = oval.m_Bval.len;
	    m_Bval.data = malloc(oval.m_Bval.len);
	    ASSERT(m_Bval.data);
	    memcpy(m_Bval.data, oval.m_Bval.data, oval.m_Bval.len);
	    break;
	}
    default:
	break;
    }
}

/* magic numbers? sizeof would make more sense. --mukesh */
uint32 TestValue::GetLength()
{
    uint32 length = 1;

    switch (m_Type) {
    case ATTR_UINT:
	length += 4;
	break;
    case ATTR_CHAR:
	length += 1;
	break;
    case ATTR_FLOAT:
	length += 4;
	break;
    case ATTR_STRING:
	/* # chars + '\0' + length? --mukesh */
	// Jeff: no null byte ever written
	length += m_Sval->length() /*+ 1*/ + 4;
	break;
    case ATTR_GUID:
	length += m_GUIDval->GetLength();
	break;
    case ATTR_SID:
	length += m_SIDval->GetLength();
	break;
    case ATTR_IDBUNDLE:
	length += m_IDval.guid->GetLength() + m_IDval.sid->GetLength();
	break;
    case ATTR_BLOB:
	length += sizeof(m_Bval.len);
	length += m_Bval.len;
	break;
    default:
	WARN << "TestValue::GetLength() my state is corrupt! :-(( \n";
	break;
    }
    return length;
}

TestValue::~TestValue()
{
    if (m_Type == ATTR_STRING && m_Sval != NULL)
	delete m_Sval;
    else if (m_Type == ATTR_BLOB && m_Bval.data != NULL) {
	m_Bval.len = 0;
	free(m_Bval.data);
    }
    else if (m_Type == ATTR_GUID && m_GUIDval != NULL)
	delete m_GUIDval;
    else if (m_Type == ATTR_SID && m_SIDval != NULL)
	delete m_SIDval;
    else if (m_Type == ATTR_IDBUNDLE) {
	if (m_IDval.guid != NULL) {
	    delete m_IDval.guid;
	}
	if (m_IDval.sid != NULL) {
	    delete m_IDval.sid;
	}
    }
}

TestValue & TestValue::operator=(const TestValue & rhs)
{
    m_Type = ATTR_UINT;			/* initialize ourselves. in fact, REWRITE THIS WHOLE CLASS SOMEDAY */

    m_Sval = 0;					/* BUG? (mem leak if we held a string) --mukesh */

    if (this != &rhs) {
	//		ASSERT(rhs.m_Type != ATTR_LAZYBLOB);
	_AssignMembers(rhs);
    }
    return *this;
}

#ifndef MIN
#define MIN(a, b)  ((a) < (b) ? (a) : (b))
#endif

/* assume all are of the same type */
float TestValue::ComputeDistance(TestValue & x, TestValue & y, TestValue & absmin,
				 TestValue & absmax)
{
    float tot_range = 0.0; 
    float x_fl = 0.0, y_fl = 0.0;

    switch (x.m_Type) {
    case ATTR_CHAR:
	x_fl = (float) x.m_Cval;
	y_fl = (float) y.m_Cval;
	tot_range = (float) absmax.m_Cval - (float) absmin.m_Cval;
	break;
    case ATTR_UINT:
	x_fl = (float) x.m_Ival;
	y_fl = (float) y.m_Ival;
	tot_range = (float) absmax.m_Ival - (float) absmin.m_Ival;
	break;
    case ATTR_FLOAT:
	x_fl = (float) x.m_Fval;
	y_fl = (float) y.m_Fval;
	tot_range = (float) absmax.m_Fval - (float) absmin.m_Fval;
	break;
    case ATTR_GUID:
	EXCEPTION("GUID type not supported right now!");
	break;
    case ATTR_SID:
	EXCEPTION("SID type not supported right now!");
	break;
    case ATTR_IDBUNDLE:
	EXCEPTION("ID Bundle type not supported right now!");
	break;
    case ATTR_STRING:
	EXCEPTION("strings type not supported right now!");
	break;
    case ATTR_BLOB:
	EXCEPTION("distance between blobs? you must be kidding!");
	break;
    default:
	EXCEPTION("unknown type: " << x.m_Type);
    }

    float t = fabs(x_fl - y_fl);
    return MIN(t, tot_range - t);
}

int TestValue::operator<(TestValue & rhs)
{
    int comp = 0;

    if (m_Type != rhs.m_Type) {
	WARN << 
	    "cannot compare values with different types (%s < %s)" <<
	    g_AttrStrings[m_Type] << g_AttrStrings[rhs.m_Type] << endl;
	WARN << "errors may result because of this\n";
	return 0;
    }

    switch (m_Type) {
    case ATTR_CHAR:
	comp = m_Cval < rhs.m_Cval;
	break;

    case ATTR_UINT:
	comp = m_Ival < rhs.m_Ival;
	break;

    case ATTR_FLOAT:
	comp = m_Fval < rhs.m_Fval;
	break;

    case ATTR_STRING:
	comp = m_Sval < rhs.m_Sval;
	break;

    case ATTR_GUID:
	EXCEPTION("GUID type not supported right now!");
	break;

    case ATTR_SID:
	EXCEPTION("SID type not supported right now!");
	break;

    case ATTR_IDBUNDLE:
	EXCEPTION("ID Bundle type not supported right now!");
	break;

    case ATTR_BLOB:
	EXCEPTION("can't compare blobs!");
	break;

    default:
	WARN << "value operator<: unknown type encountered (%d)" <<
	    m_Type << endl;
	comp = 0;
	break;
    }

    return comp;
}


int TestValue::operator>(TestValue & rhs)
{
    int comp = 0;

    if (m_Type != rhs.m_Type) {
	MercWarning
	    ("cannot compare values with different types (%s > %s)\n",
	     g_AttrStrings[m_Type], g_AttrStrings[rhs.m_Type]);
	MercWarning("errors may result because of this\n");
	return 0;
    }

    switch (m_Type) {
    case ATTR_CHAR:
	comp = m_Cval > rhs.m_Cval;
	break;

    case ATTR_UINT:
	comp = m_Ival > rhs.m_Ival;
	break;

    case ATTR_FLOAT:
	comp = m_Fval > rhs.m_Fval;
	break;

    case ATTR_STRING:
	comp = m_Sval > rhs.m_Sval;
	break;

    case ATTR_GUID:
	EXCEPTION("GUID type not supported right now!");
	break;

    case ATTR_SID:
	EXCEPTION("SID type not supported right now!");
	break;

    case ATTR_IDBUNDLE:
	EXCEPTION("ID Bundle type not supported right now!");
	break;

    case ATTR_BLOB:
	EXCEPTION("can't compare blobs!");
	break;

    default:
	MercWarning("value operator>: unknown type encountered (%d)\n",
		    m_Type);
	comp = 0;
	break;
    }

    return comp;
}


int TestValue::operator==(TestValue & rhs)
{
    int comp = 0;

    if (m_Type != rhs.m_Type) {
	MercWarning
	    ("cannot compare values with different types (%s == %s)\n",
	     g_AttrStrings[m_Type], g_AttrStrings[rhs.m_Type]);
	MercWarning("errors may result because of this\n");
	return 0;
    }

    switch (m_Type) {
    case ATTR_CHAR:
	comp = m_Cval == rhs.m_Cval;
	break;

    case ATTR_UINT:
	comp = m_Ival == rhs.m_Ival;
	break;

    case ATTR_FLOAT:
	comp = (fabs(m_Fval - rhs.m_Fval) < EPSILON);
	break;

    case ATTR_STRING:
	comp = (STR_EQ((*m_Sval), (*(rhs.m_Sval))));
	break;

    case ATTR_GUID:
	EXCEPTION("GUID type not supported right now!");
	break;

    case ATTR_SID:
	EXCEPTION("SID type not supported right now!");
	break;

    case ATTR_IDBUNDLE:
	EXCEPTION("ID Bundle type not supported right now!");
	break;

    case ATTR_BLOB:
	EXCEPTION("can't compare blobs!");
	break;

    default:
	MercWarning("value operator==: unknown type encountered (%d)\n",
		    m_Type);
	comp = 0;
	break;
    }

    return comp;
}


int TestValue::operator>=(TestValue & rhs)
{
    int comp = 0;

    if (m_Type != rhs.m_Type) {
	MercWarning
	    ("cannot compare values with different types (%s >= %s)\n",
	     g_AttrStrings[m_Type], g_AttrStrings[rhs.m_Type]);
	MercWarning("errors may result because of this\n");
	return 0;
    }

    switch (m_Type) {
    case ATTR_CHAR:
	comp = m_Cval >= rhs.m_Cval;
	break;

    case ATTR_UINT:
	comp = m_Ival >= rhs.m_Ival;
	break;

    case ATTR_FLOAT:
	comp = m_Fval >= rhs.m_Fval;
	break;

    case ATTR_STRING:
	comp = m_Sval >= rhs.m_Sval;
	break;

    case ATTR_GUID:
	EXCEPTION("GUID type not supported right now!");
	break;

    case ATTR_SID:
	EXCEPTION("SID type not supported right now!");
	break;

    case ATTR_IDBUNDLE:
	EXCEPTION("ID Bundle type not supported right now!");
	break;

    case ATTR_BLOB:
	EXCEPTION("can't compare blobs!");
	break;

    default:
	MercWarning("value operator>=: unknown type encountered (%d)\n",
		    m_Type);
	comp = 0;
	break;
    }

    return comp;
}


int TestValue::operator<=(TestValue & rhs)
{
    int comp = 0;

    if (m_Type != rhs.m_Type) {
	MercWarning
	    ("cannot compare values with different types (%s <= %s)\n",
	     g_AttrStrings[m_Type], g_AttrStrings[rhs.m_Type]);
	MercWarning("errors may result because of this\n");
	return 0;
    }

    switch (m_Type) {
    case ATTR_CHAR:
	comp = m_Cval <= rhs.m_Cval;
	break;

    case ATTR_UINT:
	comp = m_Ival <= rhs.m_Ival;
	break;

    case ATTR_FLOAT:
	comp = m_Fval <= rhs.m_Fval;
	break;

    case ATTR_STRING:
	comp = m_Sval <= rhs.m_Sval;
	break;

    case ATTR_GUID:
	EXCEPTION("GUID type not supported right now!");
	break;

    case ATTR_SID:
	EXCEPTION("SID type not supported right now!");
	break;

    case ATTR_IDBUNDLE:
	EXCEPTION("ID Bundle type not supported right now!");
	break;

    case ATTR_BLOB:
	EXCEPTION("can't compare blobs!");
	break;

    default:
	MercWarning("value operator<=: unknown type encountered (%d)\n",
		    m_Type);
	comp = 0;
	break;
    }

    return comp;
}


double TestValue::CalcAbsDistance(TestValue * a, TestValue * b)
{
    double ret = 0.0;

    if (a->m_Type != b->m_Type) {
	MercWarning
	    ("cannot compare values with different types (%s vs. %s)\n",
	     g_AttrStrings[a->m_Type], g_AttrStrings[b->m_Type]);
	MercWarning("errors may result because of this\n");
	EXCEPTION
	    ("TestValue::calc_abs_distance: cannot compute the distance between two TestValues of differing types\n");
    }

    switch (a->m_Type) {
    case ATTR_CHAR:
	ret = (double) ((int) (a->m_Cval) - (int) (b->m_Cval));
	break;

    case ATTR_UINT:
	ret = (double) (a->m_Ival - b->m_Ival);
	break;

    case ATTR_FLOAT:
	ret = (double) (a->m_Fval - b->m_Fval);
	break;

    case ATTR_STRING:
	{
	    const char *a_str = (a->m_Sval)->c_str();
	    const char *b_str = (b->m_Sval)->c_str();
	    double a_val = 0;
	    double b_val = 0;
	    int a_sum = 0;
	    int b_sum = 0;
	    size_t i;

	    for (i = 0; i < a->m_Sval->size(); i++) {
		a_sum += (int) (a_str[i]);
	    }

	    for (i = 0; i < b->m_Sval->size(); i++) {
		b_sum += (int) (b_str[i]);
	    }

	    for (i = 0; i < a->m_Sval->size(); i++) {
		a_val +=
		    pow(2.0,
			(double) a->m_Sval->size() -
			i) * (int) (a_str[i]) / a_sum;
	    }

	    for (i = 0; i < b->m_Sval->size(); i++) {
		b_val +=
		    pow(2.0,
			(double) b->m_Sval->size() -
			i) * (int) (b_str[i]) / b_sum;
	    }

	    ret = a_val - b_val;
	}
	break;

    case ATTR_GUID:
	EXCEPTION("GUID type not supported right now!");
	break;

    case ATTR_SID:
	EXCEPTION("SID type not supported right now!");
	break;

    case ATTR_IDBUNDLE:
	EXCEPTION("ID Bundle type not supported right now!");
	break;

    case ATTR_BLOB:
	EXCEPTION("can't compare blobs!");
	break;

    default:
	MercWarning("value operator<: unknown type encountered (%d)\n",
		    a->m_Type);
	break;
    }

    ret = (ret < 0) ? -ret : ret;

    return ret;
}


void TestValue::Print(FILE * stream)
{
    //    fprintf(stream, "m_Type=%d ATTR_BLOB=%d [%d]\n", m_Type, ATTR_BLOB, (m_Type <= ATTR_BLOB));
    fprintf(stream, "(%s)", g_AttrStrings[m_Type]);

    switch (m_Type) {
    case ATTR_CHAR:
	fprintf(stream, " %c", m_Cval);
	break;
    case ATTR_UINT:
	fprintf(stream, " %d", m_Ival);
	break;
    case ATTR_FLOAT:
	fprintf(stream, " %f", m_Fval);
	break;
    case ATTR_STRING:
	fprintf(stream, " %s", m_Sval->c_str());
	break;
    case ATTR_GUID:
	fprintf(stream, " %s", m_GUIDval->ToString());
	break;
    case ATTR_SID:
	fprintf(stream, " %s", m_SIDval->ToString());
	break;
    case ATTR_IDBUNDLE:
	fprintf(stream, " %s,%s", m_IDval.guid->ToString(),
		m_IDval.sid->ToString());
	break;
    case ATTR_BLOB:
	fprintf(stream, " [BLOB]");
	break;
    default:
	break;
    }
}

istream& operator>>(istream& in, TestValue& val) {
    char type[LINEBUF_SIZE];
    in >> type;
    if (! strcmp(type, "int") ) {
	uint32 v;
	in >> v;
	val = TestValue(v);
    } else if (! strcmp(type, "real") ) {
	float v;
	in >> v;
	val = TestValue(v);
    } else if (! strcmp(type, "string") ) {
	string v;
	in >> v;
	val = TestValue(v);
    } else {
	WARN << "unknown attribute type: " << type << endl;
	in.setstate( ios::failbit );
    }
    return in;
}

ostream& operator<<(ostream& out, TestValue& val) {
    switch(val.GetType()) {
    case ATTR_UINT:
	out << "int:" << val.m_Ival;
	break;
    case ATTR_CHAR:
	out << "char:" << val.m_Cval;
	break;
    case ATTR_FLOAT:
	out << merc_va("real:%.3f", val.m_Fval);
	break;
    case ATTR_STRING:
	out << "string:" << *val.m_Sval;
	break;
    case ATTR_GUID:
	out << "guid:" << *val.m_GUIDval;
	break;
    case ATTR_SID:
	out << "sid:" << *val.m_SIDval;
	break;
    case ATTR_IDBUNDLE:
	out << "id_bundle:" << *val.m_IDval.guid << "," << *val.m_IDval.sid;
	break;
    case ATTR_BLOB:
	out << "blob:size=" << val.m_Bval.len;
	break;
    case 0:
	out << "NO_VALUE";
	break;
    default:
	Debug::die("unknown value type: %d", val.GetType());
    }
    return out;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
