# Scripting tools.
#
# To use:
#
# require "TestLib.pl";
#
# See usage details below.

use strict;
use IPC::Open2;
use IO::Socket::INET;
use IO::Select;
use Fcntl;
use POSIX "sys_wait_h";

###############################################################################
#
# Test Options
#

# Run the TestApps with within gdb or valgrind. doesn't make sense unless
# XTERM is enabled.
# (gdb, valgrind, valgrindmem '')
our $MODE     = "";
# Instead of running output silently to a log, run each TestApp in an xterm
our $XTERM    = 0;
# Be verbose?
our $VERBOSE  = 1;

###############################################################################

our $PROG = "./TestApp";
our $BOOTSTRAP = "../../build/bootstrap";

our $MERCPORT_BASE = 20000;
our $TESTPORT_BASE = 60000;
our $DIRPORT_INC   = 5000;

# did the test succeed?
our $SUCCESS;

# nodeid => { nodeid     => $,
#             pid        => $,
#             out        => $,
#             in         => $,
#             killed     => $,
#             sid        => $,
#           }

our %PID_TO_NODE;
our $KILLED_ALL;
our %KILLED_PIDS;

$SIG{INT} = $SIG{TERM} = $SIG{QUIT} = sub {
    fail("Received SIG $_[0]; cleaning up", 1);
};
$SIG{PIPE} = 'IGNORE';
$SIG{CHLD} = \&REAPER;

sub REAPER {
    my $pid;
    while (($pid = waitpid(-1, &WNOHANG)) > 0) {
	my $early_exit = 0;
        my $exit_code = $? >> 8;
	my $sigval = sigval($? &255);

	if ($KILLED_ALL || $KILLED_PIDS{$pid}) {
	    # ok, we killed it on purpose
	    return;
	}

	if (defined $PID_TO_NODE{$pid}) {
		error("node $PID_TO_NODE{$pid} exited prematurely ...");
		$early_exit = 1;
		# something crashed -- no point in continuing with test
	} else {
	    error("unknown child exited!");
	}

	if ($sigval eq "SIGSEGV" || $sigval eq "SIGBUS") {
	    error("(crashed! pid=$pid exit_val=$exit_code, signal=$sigval)");
	} else {
	    error("(exited pid=$pid exit_val=$exit_code, signal=$sigval)");
	}

	if ($early_exit) {
	    fail();
	}
    }
    $SIG{CHLD} = \&REAPER; # install *after* calling waitpid
}

sub sigval {
    my $n = shift;

    if ($n == 1) { return "SIGHUP" }
    elsif ($n == 2)  { return "SIGINT" }
    elsif ($n == 3)  { return "SIGQUIT" }
    elsif ($n == 4)  { return "SIGILL" }
    elsif ($n == 6)  { return "SIGABRT" }
    elsif ($n == 8)  { return "SIGFPE" }
    elsif ($n == 9)  { return "SIGKILL" }
    elsif ($n == 11) { return "SIGSEGV" }
    elsif ($n == 13) { return "SIGPIPE" }
    elsif ($n == 14) { return "SIGALRM" }
    elsif ($n == 15) { return "SIGTERM" }
    elsif ($n == 17) { return "SIGCHLD" }
    elsif ($n == 18) { return "SIGCONT" }
    elsif ($n == 19) { return "SIGSTOP" }
    elsif ($n == 20) { return "SIGTSTP" }
    elsif ($n == 21) { return "SIGTTIN" }
    elsif ($n == 22) { return "SIGTTOU" }
    elsif ($n == 7)  { return "SIGBUS" }
    else             { return $n; }
}

system ("rm -f TEST.*.LOG bootstrap.log MercEvents.*");
system ("killall -9 bootstrap >/dev/null 2>&1");
system ("killall -9 TestApp >/dev/null 2>&1");
our $CONF = "/tmp/conf.$$";

sub max($$) {
    my ($a, $b) = @_;
    return $a > $b ? $a : $b;
}

sub min($$) {
    my ($a, $b) = @_;
    return $a < $b ? $a : $b;
}

###############################################################################

# start_bootstrap(@conf)
#
# @conf = (attr_name => [ type, min_value, max_value ])
#
# Description: start the bootstrap server with the specified configuration.
# attr_name is the attribute name, type is one of {int, float, etc.}.
sub start_bootstrap(%)
{
    fail("no attributes!") if !@_;
    my %attributes = @_;
    open(CFG, ">$CONF") || fail("can't open $CONF");
    foreach my $k (keys %attributes) {
	my @args = @{$attributes{$k}};
	print CFG "$k\t" . join("\t", @args) . "\n";
    }
    #print CFG "ToHighWaterMark\tfloat 0 1024000.0\n";
    close(CFG);

    system("$BOOTSTRAP --schema $CONF -v 6 2>bootstrap.log 1>&2 &") && fail("couldn't start bootstrap");

    vinfo("started bootstrap server with conf:\n" . `cat /$CONF`);
    Sleep(0.25);

    #system("cat $CONF");
}

sub xterm($$$)
{
    my $cmd   = shift;
    my $log   = shift;
    my $title = shift;

    my $pid = fork();
    if ($pid == 0) {
	system("xterm -T \"$title\" -e bash -c \"$cmd 2>&1 | tee $log\"");
	exit 0;
    } elsif ($pid) {
	# ok
    } else {
	fail("fork error: $!");
    }

    return ($pid, undef, undef);
}

sub run($$) {
    my $cmd   = shift;
    my $log   = shift;

    my($in, $out);
    my $pid = open2($in, $out, "$cmd 2> $log");
    
    return ($pid, $in, $out);
}

# $noderef = start_node($nodeid, $config, @args)
#
# nodeid - a unique number for the node
# config - config file for Manager in format { KEY => VAL }
# args   - arguments to start the node with (port is predefined)
#
# Description: start up a node running TestApp. Note, you MUST call the
# appropriate sequence of "Send()" "Recv()" on the nodes already in
# Mercury in order for this node to join properly before calling
# any commands on it. Use start_nodes(...) to do this automatically.
sub start_node($@) {
    my ($nodeid, $conf, @args, $node);
    
    $nodeid     = shift;
    $conf       = shift;
    @args       = @_;

    my $conffile = "/tmp/TEST.$$.CONF";

    open(CONF, ">$conffile") || fail("can't open conf file: $conffile");
    foreach my $k (keys %$conf) {
	print CONF "$k " . $conf->{$k} . "\n";
    }
    close(CONF);
    
    @args = (@args, "-C", "-R", "-c", $conffile);
    
    unshift @args, '-p', $MERCPORT_BASE+$nodeid; # HACK
    unshift @args, '--network', '--testport', $TESTPORT_BASE+$nodeid;

    my $log = "TEST.$$.NODE${nodeid}.LOG";

    vinfo("starting node $nodeid (@args) => $log");
    
    $node = {};

    $node->{nodeid} = $nodeid;
    $node->{sid} = "127.0.0.1:" . ($MERCPORT_BASE+$nodeid+$DIRPORT_INC); # HACK

    my $cmd;
    if ($MODE eq 'gdb') {
	my $temp = "/tmp/TestLibGDBArgs";
	open(T, ">$temp") || die $!;
	print T "handle SIGUSR2 nostop\n";
	print T "handle SIGUSR2 noprint\n";
	print T "r @args\n";
	close(T);
	$cmd = "gdb -x $temp $PROG";
    } elsif ($MODE eq 'valgrind') {
	$cmd = "valgrind --num-callers=5 $PROG @args";
    } elsif ($MODE eq 'valgrindmem') {
	$cmd = "valgrind --leak-check=yes $PROG @args";
    } else {
	$cmd = "$PROG @args";
    }

    my($pid, $in, $out);
    if ($XTERM) {
	($pid, $in, $out) = xterm($cmd, $log, "Node $nodeid ($node->{sid})");
    } else {
	($pid, $in, $out) = run($cmd, $log);
    }
    my $socket;
    my $maxtries = 20;
    while (!defined $socket) {
	$socket = IO::Socket::INET->new(PeerAddr => 'localhost',
					PeerPort => $TESTPORT_BASE+$nodeid,
					Proto    => 'tcp',
					Type     => SOCK_STREAM,
					Timeout  => 10);
	Sleep(0.1);
	if (--$maxtries <= 0) {
	    fail("Can't connect to Test Port " . 
		 ($TESTPORT_BASE+$nodeid) . "for node $nodeid");
	}
    }

    $node->{pid} = $pid;
    $node->{in} = $socket;
    $node->{out} = $socket;
    
    if ($node->{pid} <= 0 || !kill (0, $node->{pid}) ) {
	fail("Couldn't start node $nodeid!");
    }
    $PID_TO_NODE{$node->{pid}} = $nodeid;

    return $node;
}

# %nodes = start_nodes($number, @args)
#
# Description: start several nodes with the specified arguments 
# (port predefined). The return value is a hash of all the node references
# indexed by node id (1, 2, ..., $number).
sub start_nodes($@) {
    my %nodes;
    my $num = shift @_;

    for (my $i=1; $i<=$num; $i++) {
	$nodes{$i} = start_node($i, @_);

	# idle the nodes for a while to process the join messages
	# (we need this because Mercury is single threaded now)
	for (my $j=0; $j<3; $j++) {
	    for (my $n=1; $n < $i; $n++) {
		merc_idle($nodes{$n}, 3);
	    }
	}
    }

    return %nodes;
}

# stop_node($noderef)
#
# Description: kill the node.
sub stop_node($) {
    my $node = shift;
    fail("bad node") if !$node || !$node->{out};

    vinfo("stopping node " . $node->{nodeid});

    $KILLED_PIDS{ $node->{pid} } = 1;
    kill 15, $node->{pid};
}

# stop_all()
#
# Description: kill everything. This is kind of hacky.
sub stop_all() {
    vinfo("stopping everything"); 
    
    $KILLED_ALL = 1;
    system("killall bootstrap >/dev/null 2>&1");
    system("killall TestApp >/dev/null 2>&1");
    system("killall gdb >/dev/null 2>&1");
    unlink $CONF;
}

###############################################################################

# Send($noderef)
#
# Description: Calls Send() on the node. REMEMBER THAT EVERYTHING IS
# SINGLE THREADED NOW! YOU MUST CALL THIS OR r() IN ORDER FOR MECURY TO
# MAKE PROGRESS (e.g., process join messages etc. after start_node()).
sub Send($) {
    #vinfo($_[0]->{nodeid} . ": send"); 

    my $resp = cmd($_[0], "send");
}

# Recv($noderef)
#
# Description: Calls Recv() on the node.
sub Recv($) {
    #vinfo($_[0]->{nodeid} . ": recv"); 

    cmd($_[0], "recv");
}

# Merc($noderef)
#
# Description: Calls Merc() on the node.
sub Merc($) {
    #vinfo($_[0]->{nodeid} . ": recv"); 

    cmd($_[0], "merc");
}

# idle($noderef, $loops)
#
# Description: Calls Send(), Recv() $loops times on the node.
sub idle($$) {
    my $node  = shift;
    my $loops = shift;
    if (!$loops) { $loops = 5 }

    #vinfo($node->{nodeid} . ": idling for $loops cycles"); 

    while ($loops-- > 0) {
	Send($node);
	Recv($node);
    }
}

# idle_all(\%noderefs, $loops)
#
# Description: Calls Send(), Recv() $loops times on each node in the hash.
# Calls it once on each node in order and repeats until finished.
sub idle_all($$) {
    my $nodes = shift;
    my $loops = shift;
    if (!$loops) { $loops = 5 }

    vinfo("idling all for $loops cycles"); 

    while ($loops-- > 0) {
	# HACK: if we are in the process of joining, then we need
	# to process each guy's join messages in turn. only the first
	# guy to join actually is able to handle messages so far.
	# The other guys will be blocked waiting
	foreach my $k (sort keys %$nodes) {
	    #info("idling $k $loops...");
	    Send($nodes->{$k});
	    Recv($nodes->{$k});
	    #Sleep(0.2);
	}
    }
}

# merc_idle($noderef, $loops)
#
# Description: Calls Merc() $loops times on the node.
sub merc_idle($$) {
    my $node  = shift;
    my $loops = shift;
    if (!$loops) { $loops = 5 }

    #vinfo($node->{nodeid} . ": idling for $loops cycles"); 

    while ($loops-- > 0) {
	Merc($node);
	#Sleep(0.1);
    }
}

# merc_idle_all(\%noderefs, $loops)
#
# Description: Calls Merc() $loops times on each node in the hash.
# Calls it once on each node in order and repeats until finished.
sub merc_idle_all($$) {
    my $nodes = shift;
    my $loops = shift;
    if (!$loops) { $loops = 5 }

    vinfo("merc_idling all for $loops cycles"); 

    while ($loops-- > 0) {
	# HACK: if we are in the process of joining, then we need
	# to process each guy's join messages in turn. only the first
	# guy to join actually is able to handle messages so far.
	# The other guys will be blocked waiting
	foreach my $k (sort keys %$nodes) {
	    Merc($nodes->{$k});
	}
    }
}

our $UNIQUE_ID = 0;

# $guid = add($noderef)
#
# Description: add a new object to the node. Returns the guid of the object.
sub add($) {
    vinfo($_[0]->{nodeid} . ": add"); 

    my $resp = cmd($_[0], "add", "object$UNIQUE_ID");
    $UNIQUE_ID++;

    my $guid;
    if ($resp =~ /guid=([^ \t\n]+)/) {
	$guid = $1;
    } else {
	error("add error: $resp");
    }

    return $guid;
}

# remove($noderef, $guid)
#
# Description: remove the guid from the node.
sub remove($$) {
    vinfo($_[0]->{nodeid} . ": remove " . $_[1]); 

    my $resp = cmd($_[0], "remove", $_[1]);
    if ($resp =~ /OK/) {
	return;
    } else {
	error("set error: $resp");
    }
}

# $set($noderef, $guid, $key, $val)
#
# key = "key"
# val = "type:val"
#
# Description: Set the attribute key to the value val in the object guid
# on the node.
sub set($$$$) {
    my ($node, $guid, $key, $val) = @_;

    vinfo($node->{nodeid} . ": set $guid $key $val");

    my $resp = cmd($node, "set", $guid, $key, $val);
    if ($resp =~ /OK/) {
	return;
    } else {
	error("set error: $resp");
    }
}

# setin($noderef, $guid, @interest)
#
# @interest = ([key, op, val], ...)
#
# Description: Set the interest of the object guid on the node to the
# interest array. Each element of the interest is, for example,
# [ "x", ">", "float:10" ]
sub setin($@) {
    my $node = shift;
    my $guid = shift;
    my @in   = @_;

    my @array;
    foreach my $c (@in) {
	@array = (@array, @$c);
    }

    vinfo($node->{nodeid} . ": setin $guid @array"); 
    
    my $resp = cmd($node, "setin", $guid, @array);
    if ($resp =~ /OK/) {
	return;
    } else {
	error("setin error: $resp");
	return;
    }
}

# setin($noderef, $guid, $fixed, $delta)
#
# Description: Set the cost of an object.
sub setcost($$$$) {
    my $node  = shift;
    my $guid  = shift;
    my $fixed = shift;
    my $delta = shift;

    vinfo($node->{nodeid} . ": setcost $guid $fixed $delta"); 
    
    my $resp = cmd($node, "setcost", $guid, $fixed, $delta);
    if ($resp =~ /OK/) {
	return;
    } else {
	error("setcost error: $resp");
	return;
    }
}

# migrate($noderef, $guid, $sid)
#
# Description: migrate an object from the node to the node identified by sid.
# You can obtain SIDs from the sid field in objects or $noderef->{sid}.
sub migrate($$$) {
    my $node = shift;
    my $guid = shift;
    my $sid  = shift;

    vinfo($node->{nodeid} . ": migrate $guid $sid"); 

    my $resp = cmd($node, "migrate", $guid, $sid);
    if ($resp =~ /OK/) {
	return;
    } else {
	error("migrate error: $resp");
	return;
    }
}

# $obj = show($noderef, $guid)
#
# Description: retrieve the contents of the object identified by guid on
# the node. The return value is an object hash containing:
#
# {
#   guid      => $guid
#   sid       => $sid
#   replica   => bool
#   dirty     => bool
#   migrating => bool
#   attrs     => { key => [ type, val ] }
#   subs      => { key => [ op, type, val ] }
# }
sub show($$) {
    my $node = shift;
    my $guid = shift;

    vinfo($node->{nodeid} . ": show $guid"); 

    my $resp = cmd($node, "show", $guid);
    if ($resp =~ /OK: (.*)/) {
	return parse_obj($1);
    } else {
	error("show error: $resp");
	return undef;
    }
}

# %ret = store($noderef)
#
# Description: retrieve all the objects from the node in a hash indexed
# by the object guids. The format of object refernces are defined above in
# show(...)
sub store($) {
    my %ret;

    vinfo($_[0]->{nodeid} . ": store"); 

    my $resp = cmd($_[0], "store");
    if (!$resp || $resp =~ /OK/) {
	my @lines = split(/\n/, $resp);
	foreach my $l (@lines) {
	    if ($l =~ /OK: (.*)/) {
		my $obj = parse_obj($1);
		if ($obj) {
		    $ret{$obj->{guid}} = $obj;
		}
	    }
	}
	return %ret;
    } else {
	error("store error: $resp");
	return undef;
    }
}

# $str = benchmark($noderef)
# 
# Description: dump the benchmark numbers on $noderef
sub benchmark($) {
    vinfo($_[0]->{nodeid} . ": benchmark");

    my $resp = cmd($_[0], "benchmark");
    return $resp;
}

# $str = connections($noderef)
# 
# Description: dump the replica connections
sub connections($) {
    vinfo($_[0]->{nodeid} . ": conn");

    my $resp = cmd($_[0], "conn");
    return $resp;
}

# dump_graph($noderef)
# 
# Description: Show the interest graph on a node using graphvis
sub dump_graph($) {
    vinfo($_[0]->{nodeid} . ": graphvis");

    my $resp = cmd($_[0], "graphvis");
    if ($resp !~ /OK/) {
	error("graphvis error: $resp");
    }
}

# @graph = graph($noderef)
# 
# Description: dump the graph on a node to text in the format
# @graph = ( [src, dst, link_time], ... )
sub graph($) {
    vinfo($_[0]->{nodeid} . ": graph");

    my $resp = cmd($_[0], "graph");
    my @lines = split(/\n/, $resp);
    foreach my $line (@lines) {
	my @ent = split(/,/, $line);
	$line = \@ent;
    }
    return @lines;
}

# %graph = graph_by_tgt(@graph)
#
# Description: index the entries by target.
sub graph_by_tgt(@) {
    my @graph = @_;

    my %ret;
    foreach my $ent (@graph) {
	$ret{$ent->[1]} = {} if ! exists $ret{$ent->[1]};
	$ret{$ent->[1]}->{$ent->[0]} = $ent->[2];
    }
    return %ret;
}

# %graph = graph_by_src(@graph)
#
# Description: index the entries by source.
sub graph_by_src(@) {
    my @graph = @_;

    my %ret;
    foreach my $ent (@graph) {
	$ret{$ent->[0]} = {} if ! exists $ret{$ent->[0]};
	$ret{$ent->[0]}->{$ent->[1]} = $ent->[2];
    }
    return %ret;
}

# $str = graph_to_str(@graph)
#
# Description: turns a graph data structure into a printable string
# in the format "src->tgt (link_time)\n"
sub graph_to_str(@) {
    my @graph = @_;
    my $str = '';
    foreach my $ent (@graph) {
	$str .= $ent->[0] . "->" . $ent->[1] . " (" . $ent->[2] . ")\n";
    }
    return $str;
}

# $string = obj_to_str($obj)
#
# Description: Convert an object reference into a human readable string.
sub obj_to_str($) {
    my $obj = shift;

    if (! $obj) { return ""; }

    my $ret = '(GObject';

    $ret .= " guid=" . $obj->{guid};
    $ret .= " sid=" . $obj->{sid};
    $ret .= " status=" . ($obj->{replica}?"R":"P");
    $ret .= " dirty=" . ($obj->{dirty}?"Y":"N");
    $ret .= " migrating=" . ($obj->{migrating}?"Y":"N");
    $ret .= " attributes=[";
    my $attrs = $obj->{attrs};
    my @keys = keys %$attrs;
    for (my $i=0; $i<@keys; $i++) {
	my $k = $keys[$i];
	$ret .= "," if $i != 0;
	my @v = @{$attrs->{$k}};
	$ret .= "$k:$v[0]:$v[1]";
    }
    $ret .= "]";
    $ret .= " sub=[";
    my $subs = $obj->{subs};
    @keys = @$subs;
    for (my $i=0; $i<@keys; $i++) {
	my $k = $keys[$i];
	$ret .= "," if $i != 0;
	my @v = @$k;
	$ret .= "$v[0]:$v[1]:$v[2]:$v[3]";
    }
    $ret .= "]";
    $ret .= ")";

    return $ret;
}

# obj_eq(obj1, obj2, cmp_status)
#
# Description: Returns true if two objects are "equal". We ignore dirty
# status and the object's subscription in the comparison (to make this
# useful for comparing a replica against the expected primary). If
# cmp_status is true, we compare the replica status.
sub obj_eq($$@) {
    my ($obj1, $obj2, $cmp_status) = @_;

    if (!$obj1 || !$obj2) {
	return 0;
    }

    if ($obj1->{guid} ne $obj2->{guid}) {
	return 0;
    }

    if ($obj1->{sid} ne $obj2->{sid}) {
	return 0;
    }

    if ($cmp_status && $obj1->{replica} ne $obj2->{replica}) {
	return 0;
    }

    # ignore dirty status

    my $attrs1 = $obj1->{attrs};
    my $attrs2 = $obj2->{attrs};
    my @keys1 = keys %$attrs1;
    my @keys2 = keys %$attrs2;
    if (@keys1 != @keys2) {
	return 0;
    }
    for (my $i=0; $i<@keys1; $i++) {
	my $k = $keys1[$i];
	my $v1 = $attrs1->{$k};
	my $v2 = $attrs2->{$k};

	if (!$v1 || !$v2) {
	    return 0;
	}

	if ($v1->[0] ne $v2->[0] || $v1->[1] ne $v2->[1]) {
	    return 0;
	}
    }

    # ignore subs

    return 1;
}

###############################################################################

sub cmd($@) {
    my $node = shift;
    fail("bad node") if !$node || !$node->{out};

    my $out = $node->{out};
    my $cmd = join(" ", @_) . "\n";
    print $out $cmd;
    my $line;
    my $ret;

    my $in = $node->{in};
    $line = read_line($in);
    if ($line && $line !~ /==BEGIN_RESP/) {
	error("response to @_ didn't begin with ==BEGIN_RESP: $line");
    }
    while($line = read_line($in)) {
	last if $line =~ /==END_RESP/;

	$ret .= $line;
    }

    return $ret;
}

###############################################################################

# Returns an object of the form:
# {
#   guid => $
#   sid  => $
#   status => $
#   dirty => $
#   attrs => { key => [ type, val ] }
#   subs  => { key => [ op, type, val ] }
# }
sub parse_obj($) {
    my $str = shift;
    my $obj = {};

    if ($str =~ /\(GObject guid=([^ ]+) sid=([^ ]+) status=(P|R) dirty=(Y|N) migrating=(Y|N) attributes=\[([^\]]*)\] sub=Interest .*guid=[^ \t\n]+ .*cons=\[([^\]]*)\] .*tags=\[[^\]]*\].*\)/) {
	info("objstr: $str");
	$obj->{guid}      = $1;
	$obj->{sid}       = $2;
	$obj->{replica}   = $3 eq 'R';
	$obj->{dirty}     = $4 eq 'Y';
	$obj->{migrating} = $5 eq 'Y';
	my $attrs = $6;
	my $subs  = $7;

	$obj->{attrs} = {};
	for my $a (split(/,/, $attrs)) {
	    my ($k, $t, @v) = split(/:/, $a);
	    my $v = join(":", @v);
	    $obj->{attrs}->{$k} = [$t, $v];
	}

	$obj->{subs} = [];
	foreach my $c (split(/,/, $subs)) {
	    $c =~ s/\(|\)//g;
	    my ($k, $o, $t, @v) = split(/:/, $c);
	    my $v = join(":", @v);
	    push @{$obj->{subs}}, [$k, $o, $t, $v];
	}

	return $obj;
    } else {
	error("bad object string: $str");
	return undef;
    }
}

###############################################################################

# Report an error
# Params:
# $0: message
# $1: file handle to print to (optional, default=STDERR)
sub error($) {
    my $fh = *STDERR;
    if (defined $_[1]) {
	$fh = $_[1];
    }

    $SUCCESS = 0;

    print $fh (_time() . " [EE] $_[0]\n");
}

# Report information
# Params:
# $0: message
# $1: file handle to print to (optional, default=STDERR)
sub info($) {
    my $fh = *STDERR;
    if (defined $_[1]) {
	$fh = $_[1];
    }
    print $fh (_time() . " [II] $_[0]\n");
}

sub vinfo($) {
    info($_[0]) if $VERBOSE;
}

our $FAILED = 0;

# Reports failure to run and complete test
# Params:
# $0: message
# $1: file handle to print to (optional, default=STDOUT)
sub fail(@) {
    my $fh = *STDOUT;
    $_[0] = "FAILED" if ! defined $_[0];
    print $fh (_time() . " [FF] $_[0]\n");
    if (!$_[1]) {
	print $fh (_time() . " [FF] Sleeping to allow you to debug\n");
	sleep(3000000);
    }
    # don't recursively fail again!
    if ($FAILED == 1) {
	return;
    }
    $FAILED = 1;
    stop_all();
    info("UNKNOWN");
    exit(255);
}

# true if no errors reported
sub success() {
    return $SUCCESS;
}

sub _time() {
    return time();
}

# Read line with timeout
# $0: the file descriptor to read from
# $1: timeout in sec (optional, default=1)
sub read_line($$) {
    my $fh = shift;
    my $timeout = shift;
    $timeout = 30 if (!defined $timeout);
    my $res = undef;

    eval {
	local $SIG{ALRM} = sub { die "alarm clock restart" };
	alarm $timeout;
	eval {
	    $res = <$fh>;
	    $res =~ s/\r\n/\n/g;
	};
	alarm 0;

	if ($@ && $@ =~ /alarm clock restart/) {
	    die "timeout failure";
	}
    };
    alarm 0;

    if ($@ && $@ =~ /timeout failure/) {
	error("Timed out waiting for response");
	return undef;
    }
    return $res;
}

sub Sleep($) {
    select undef, undef, undef, $_[0];
}

###############################################################################

1;
