#!/usr/bin/perl
#
# Execute a command on the terminal interface of a bunch of nodes
#

use strict;
use Travertine;
use Getopt::Std;
use IO::File;
use IO::Socket::INET;
use vars qw($opt_l $opt_t $opt_h $opt_p $opt_b $opt_o);

$Travertine::DEBUG_SHOW_STACKTRACE = 0;

our $MAX_PARALLEL = 10;

getopts("l:t:p:hob:");

if (!$ARGV[0] || $opt_h) {
    print STDERR "usage: TermExec.pl [options] <exp_file> <cmd>\n\n";
    print STDERR "       -l rex   only hosts with login matching regexp\n";
    print STDERR "       -t rex   only hosts with title matching regexp\n";
    print STDERR "       -p pass  terminal password if required\n";
    print STDERR "       -b port  base port which vservers start from\n";
    print STDERR "       -o       to stdout in clean format\n";
    print STDERR "       -h       this help message\n";
    exit 0;
}

our $INPUT = *STDIN;
our $LOGIN_REX = defined($opt_l) ? $opt_l : undef;
our $TITLE_REX = defined($opt_t) ? $opt_t : undef;
our $PASSWORD  = defined($opt_p) ? $opt_p : undef;
our $BASEPORT  = defined($opt_b) ? $opt_b : 30000;
our $STDOUT    = $opt_o;

if ($ARGV[0] ne '-') {
    $INPUT = new IO::File("<$ARGV[0]");
    shift @ARGV;
    tdie "can't open $ARGV[0]: $!" if !$INPUT;
}

our $CMD = join(' ', @ARGV);

our @tuples;

while (<$INPUT>) {
    chomp $_;
    my ($login, $title, $ser) = ($_ =~ /^([^\t]+)\t'([^\t]+)'\t([^\t]+)$/);
    tdie "bad line: $_" if !defined $login or !defined $title or !$ser;

    next if ($LOGIN_REX && $login !~ $LOGIN_REX);
    next if ($TITLE_REX && $title !~ $TITLE_REX);

    push @tuples, [ $login, $title ];
}

if (@tuples == 0) {
    tdie "no matching logins/titles";
}

tinfo "## executing commands ...";

my @res = ParallelExec3($MAX_PARALLEL, sub {
    my $login = shift;
    my $title = shift;

    my ($user, $host, $iface) = ($login =~ /(.*)\@(.*):(.*)/);
    tdie "bad login: $login" if !$host;

    my $port;
    if ($title =~ /bootstrap/) {
	return undef;
    } elsif ($title =~ /master/) {
	$port = $BASEPORT;
    } elsif ($title =~ /.*:(\d+)/) {
	$port = $BASEPORT + $1;
    } else {
	tdie "bad title: $title";
    }

    my $sock = IO::Socket::INET->new(PeerAddr => $host,
				     PeerPort => $port,
				     Proto    => 'tcp',
				     Type     => SOCK_STREAM,
				     Timeout  => 20);

    if (!$sock) {
	twarn "can't open socket to $host:$port (title: $title)";
	return undef;
    }

    if ($PASSWORD) {
	print $sock "$PASSWORD\n";
    }
    # read off the message banner;
    my $banner = read_line($sock);

    if ($banner !~ /Colyseus Terminal/) {
	chomp $banner;
	$banner =~ s/^Password: //;
	$sock->close();
	twarn "$host:$port (title: $title) closed socket, it said: $banner";
	return undef;
    }

    print $sock "$CMD\n";
    my $line;
    my $resp;
    while ($line = read_line($sock)) {
	next if $line =~ /^(Password: )?==BEGIN_RESP/;
	last if $line =~ /^==END_RESP/;
	$resp .= $line;
    }
    if ($line !~ /^==END_RESP/) {
	twarn "$host:$port (title: $title) resp did not end with ==END_RESP:\n"
	    . thighlight($resp);
    } else {
	$resp .= "";
    }

    close($sock);

    return $resp;
    
}, @tuples);

tinfo "## displaying results ...";

for (my $i=0; $i<@tuples; $i++) {
    my ($login, $title) = @{$tuples[$i]};
    my $resp = @res[$i];
    my $good = defined $resp;

    next if $title =~ /bootstrap/;
    if (!defined $resp) {
	$resp = 'NO RESPONSE';
    } else {
	chomp $resp;
	$resp = thighlight($resp) if !$STDOUT;
    }

    my $str = "$login\t\"$title\"\n$resp";

    if ($STDOUT) {
	print "$str\n";
    } else {
	if ($good) {
	    tinfo $str;
	} else {
	    twarn $str;
	}
    }
}

tinfo "## done.";

###############################################################################

sub read_line($$) {
    my $fh = shift;
    my $timeout = shift;
    $timeout = 30 if (!defined $timeout);
    my $res = undef;

    eval {
        local $SIG{ALRM} = sub { die "alarm clock restart" };
        alarm $timeout;
        eval {
            $res = <$fh>;
            $res =~ s/\r\n/\n/g;
        };
        alarm 0;

        if ($@ && $@ =~ /alarm clock restart/) {
            die "timeout failure";
        }
    };
    alarm 0;

    if ($@ && $@ =~ /timeout failure/) {
        twarn "timed out waiting for response";
        return undef;
    }
    return $res;
}
