#!/usr/bin/perl
#
# This script executes ColyseusQuake on a remote machine which has the
# image generated by MakeImage.pl installed.
#
# The first host will be used as the master host and as the bootstrap host.
#

use strict;
use Net::hostent;
use Socket;
use Getopt::Std;
use Travertine;
use vars qw($opt_B $opt_o $opt_x $opt_g $opt_M $opt_t $opt_l $opt_v $opt_b 
	    $opt_m $opt_e $opt_k $opt_S $opt_W $opt_c $opt_p $opt_i $opt_D
	    $opt_E $opt_V $opt_G $opt_L $opt_d $opt_a $opt_A $opt_F $opt_N
	    $opt_U $opt_z $opt_P $opt_T);

## Config Options

our $MODE       = '';
our $VERBOSITY  = -1;
our $DBG_FILES  = "";

## File locations (relative to $TOPDIR/id-source)

our $TOPDIR         = "$ENV{HOME}/id-source";
our $LIBRARY_PATH   = "../Merc:.";
our $BOOTSTRAP_PROG = "../Merc/build/bootstrap";
our $QUAKE_PROG     = "./quake2";
our $QUAKE_SCHEMA   = "../Merc/build/schema_quake.cfg";

###############################################################################

our $MERC_MASTER_PORT  = 19999;
our $QUAKE_MASTER_PORT = 39999;

our @MERC_PORTS   = (20000, 20001, 20002, 20003, 20004, 20005, 20006, 20007,
		     20008, 20009, 20010, 20011, 20012, 20013, 20014, 20015,
		     20016, 20017, 20018, 20019, 20020, 20021, 20022, 20013,);
#    SIDPort      = MERC_PORT + 5000
#    TerminalPort = MERC_PORT + 10000
our @QUAKE_PORTS  = (40000, 40001, 40002, 40003, 40004, 40005, 40006, 40007,
		     40008, 40009, 40010, 40011, 40012, 40013, 40014, 40015,
		     40016, 40017, 40018, 40019, 40020, 40021, 40022, 40013,);

###############################################################################

getopts("BxgM:t:l:b:v:m:e:kS:cpi:DE:V:G:Lda:AFNSUz:P:T:");

if (@ARGV == 0) {
    print STDERR "usage: Run.pl [options] [user1\@]master[:iface] [user2\@]host2[:iface] ...\n\n";
    print STDERR "  You can optionally specify an internal interface after the hostname that\n";
    print STDERR "  will be used as the hostname/IP in the application (needed on emulab).\n";
    print STDERR "  The experiment will start and run *detached* -- that is, the script\n";
    print STDERR "  does not wait until the experiment finishes. Instead it outputs\n";
    print STDERR "  about the experiment to a filehandle (default stdout) so that\n";
    print STDERR "  you can 're-attach' to it later. The output format is:\n\n";
    print STDERR "    login<tab>'node name'<tab>serialized_remote_daemon_ref\n\n";
    print STDERR "       -B       run as bg processes instead of remote daemons (can't attach)\n";
    print STDERR "       -o       output experiment info to file (default stdout)\n";
    print STDERR "       -x       invoke xterm (one per daemon)\n";
    print STDERR "       -g       invoke gterm (one tab per daemon)\n";
    print STDERR "       -k       just kill all relevant processes on servers\n";
    print STDERR "       -t dir   remote pubsub top directory\n";
    print STDERR "       -l dir   directory to output logs to\n";
    print STDERR "       -v num   number of virtual servers per machine\n";
    print STDERR "       -b num   number of bots on each server\n";
    print STDERR "       -M num   number of monsters on each server\n";
    print STDERR "       -m map   map name\n";
    print STDERR "       -e num   how long to execute (msec)\n";
    print STDERR "       -A       no artificial latency\n";
    print STDERR "       -a file  latency file (assumed to be in Merc/topologies/)\n";
    print STDERR "       -S dim   do striping along this dimension\n";
    print STDERR "       -c       DISABLE caching. repeat: DISABLE caching (default is enabled)\n";
    print STDERR "       -p       DISABLE predictive pubs/subs\n";
    print STDERR "       -F       don't fight monsters (normal DM)\n";
    print STDERR "       -i type  how to partition items\n";
    print STDERR "       -D       use DHT style pubs/subs\n";
    print STDERR "       -N       disable measurement (logging)\n";
    print STDERR "       -z IP    run with visualizer directed at IP\n";
    print STDERR "       -E mode  run mode {normal, gdb, valgrind, valgrindmem}\n";
    print STDERR "       -V num   debug verbosity\n";
    print STDERR "       -G files debug files\n";
    print STDERR "       -P num   max parallel ssh connections on startup\n";
    print STDERR "       -L       clean + create remote logdir\n";
    print STDERR "       -S       run directory creation with sudo\n";
    print STDERR "       -d       use --deltas\n";
    print STDERR "\n";
    exit 1;
}

our $RUN_AS_BG       = defined $opt_B;
our $TERM_PASSWD     = $opt_T || "";
our $LOG_DIR         = $opt_l || "/tmp";
our $VIRTUAL_SERVERS = $opt_v || 1;
our $NUMBOTS         = defined $opt_b ? $opt_b : 1;
our $NUMMONSTERS     = defined $opt_M ? $opt_M : 1;
our $BOTS_ON_MASTER  = 1;
our $MAP             = $opt_m || "big_map";
our $NSERVERS        = scalar(@ARGV) * $VIRTUAL_SERVERS;
our $TIMELIMIT       = $opt_e || 1000000000;
our $STRIPE_DIM      = $opt_S || "";
our $ENABLE_CACHE    = 1; $ENABLE_CACHE = 0 if (defined $opt_c);
our $ENABLE_PREDICTION = 1; $ENABLE_PREDICTION = 0 if (defined $opt_p);
our $ITEM_PARTITIONING = defined $opt_i ? $opt_i : 'b';
our $DHT = defined $opt_D;
our $CREATE_LOGDIR = defined $opt_L;
our $USE_SUDO = defined $opt_U;
our $VISUALIZER = $opt_z;
our $DELTAS = defined $opt_d;
our $USE_LATENCY  = !defined $opt_A;
our $LATENCY_FILE = defined $opt_a ? $opt_a : "n100.v6.lat";
our $NO_FIGHT_MONSTERS = defined $opt_F;
our $DO_MEASUREMENT = !defined $opt_N;
our $MAXPARALLEL = $opt_P || scalar(@ARGV);

if ($opt_t) {
    $TOPDIR = "$opt_t/id-source";
}

our $OUTPUT = *STDOUT;
if ($opt_o) {
    $OUTPUT = new IO::File(">$opt_o");
    tdie "can't open $opt_o: $!" if !$OUTPUT;
}

our $INVOKE_XTERM = 0;
our $INVOKE_GTERM = 0;

if ($opt_x) {
    $INVOKE_XTERM = 1;    
}
if ($opt_g) {
    $INVOKE_GTERM = 1;
    $INVOKE_XTERM = 0;
}

if ($STRIPE_DIM eq "") {
    $QUAKE_SCHEMA = "../Merc/build/schema_quake_onehub.cfg";
}
else {
    $QUAKE_SCHEMA = "../Merc/build/schema_quake_stripe.cfg";
}

our $KILL            = $opt_k;

if ($opt_E) {
    $MODE = $opt_E;
}
if ($opt_V) {
    $VERBOSITY = $opt_V;
}
if ($opt_G) {
    $DBG_FILES = $opt_G;
}

###############################################################################

sub Run
{
    my ($ssh, $func, $args, $log, $title) = @_;

    my %opts = ( -daemon => 1, 
		 -log => $log,
		 -title => $title );
    if ($RUN_AS_BG) {
	%opts = ( -background => 1,
		  -log => $log );
    }

    my $ref = ExecRemoteFunc($ssh, $func, $args, %opts);
    if (!$ref) {
	return undef;
    }

    if ($INVOKE_XTERM) {
	if ($RUN_AS_BG) {
	    twarn "can't attach to a bg func";
	} else {
	    $ref->attachToXterm();
	}
    }

    return $ref;
}

sub RunBootstrap
{
    my ($topdir, $libpath, $prog, $args)  = @_;
    
    chdir($topdir);
    $ENV{LD_LIBRARY_PATH} .= ":$libpath";
    psystem("$prog $args");
    if ($? == -1) {
	twarn "failed to execute: $!";
    }
    elsif ($? & 127) {
	twarn "child died with signal %d, %s coredump",
	($? & 127),  ($? & 128) ? "with ":  " without ";
    }
    else {
	twarn sprintf "child exited with value %d", $? >> 8;
    }
}

sub RunQuake
{
    my ($topdir, $libpath, $prog, $args, $mode, $log_dir, $if, $port) = @_;

    chdir($topdir);
    $ENV{LD_LIBRARY_PATH} .= ":$libpath";

    my $cmd;
    if ($mode eq 'gdb') {
	my $temp = "/tmp/RunLocalTest.master.$$";
	open(T, ">$temp") || die $!;
	print T "handle SIGUSR2 nostop\n";
	print T "exec-file ./quake2\n";
	print T "r $args\n";
	close(T);
	$cmd = "gdb -x $temp ./quake2";
    } elsif ($mode eq 'valgrind') {
	$cmd = "valgrind --tool=addrcheck ./quake2 $args";
    } elsif ($mode eq 'valgrindmem') {
	$cmd = "valgrind --tool=addrcheck --leak-check=yes ./quake2 $args";
    } else {
	$cmd = "$prog $args";
    }

    psystem("netstat -u -s > $log_dir/Netstat.$if:$port.out");
    psystem($cmd);
    if ($? == -1) {
	twarn "failed to execute: $!";
    }
    elsif ($? & 127) {
	twarn "child died with signal %d, %s coredump",
	($? & 127),  ($? & 128) ? "with ":  " without ";
    }
    else {
	tinfo sprintf "child exited with value %d", $? >> 8;
    }
    psystem("netstat -u -s >> $log_dir/Netstat.$if:$port.out");
}

sub StartBootstrap
{
    my ($login, $ssh) = @_;
    my ($user, $host, $iface) = SplitLogin($login);

    tinfo "** Starting Bootstrap on: $iface:15000";
    return Run($ssh, \&RunBootstrap, 
	       [ $TOPDIR, $LIBRARY_PATH, $BOOTSTRAP_PROG,
		 " -v $VERBOSITY "         .
		 "--schema $QUAKE_SCHEMA " . 
		 "--hostname $iface --histograms --buckets 35 " .
		 "--nservers $NSERVERS" ],
	       "$LOG_DIR/OutputLog.bootstrap.out",
	       "bootstrap ($host)");
}

sub StartQuake
{
    my ($login, $ssh, $mercPort, $quakePort, $args) = @_;
    my ($user, $host, $iface) = SplitLogin($login);

    my $seed = rand(42000);
    
    $args .= " --hostname $iface --port $mercPort +set port $quakePort";
    $args .= " --randseed $seed ";

    tinfo "* Starting Quake on: $iface";
    tinfo "*      merc-port   : $mercPort";
    tinfo "*      direct-port : " . ($mercPort+5000);
    tinfo "*      term-port   : " . ($mercPort+10000);
    tinfo "*      quake-port  : $quakePort";

    return Run($ssh, \&RunQuake,
	       [$TOPDIR, $LIBRARY_PATH, $QUAKE_PROG, $args, $MODE, 
		$LOG_DIR, $iface, $mercPort],
	       "$LOG_DIR/OutputLog.$iface:".($mercPort+5000).".out",
	       "$host:$mercPort");
}

sub CleanServer
{
    my ($ssh) = shift @_;

    ExecRemoteFunc($ssh, sub {
	my $logdir  = shift;
	my $usesudo = shift;
	
	if ($logdir) {
	    psystem("rm -f $logdir/*.log");
	    psystem("rm -f $logdir/*.out");
	    psystem(($usesudo?"sudo ":"") . "mkdir -p $logdir");
	    psystem(($usesudo?"sudo ":"") . "chmod 777 $logdir");
	}
	psystem("killall screen >/dev/null 2>&1");
	psystem("killall -9 bootstrap >/dev/null 2>&1");
	psystem("killall -9 quake2 >/dev/null 2>&1");
	psystem("killall -9 gdb >/dev/null 2>&1");
	psystem("killall -9 valgrind >/dev/null 2>&1");
    }, [$CREATE_LOGDIR ? $LOG_DIR : undef, $USE_SUDO], 
		   -print => 1, -timeout => 300 );
}

sub ResolveIP($)
{
    my $host = shift;

    if ($host !~ /\d+\.\d+\.\d+\.\d+/) {
	my $hent = gethost($host);
	die "bad hostname: $host -- $?" if
	    !defined $hent || @{$hent->addr_list} < 1;
	$host = inet_ntoa($hent->addr_list->[0]);
    }

    return $host;
}

sub SplitLogin($)
{
    my $login = shift;

    my ($user, $host) = ($login =~ /^([^\@]+)\@(.+)$/);
    die "bad login: $login" if !$user || !$host;

    # local interface required for emulab
    my $iface = $host;
    if ($host =~ /:/) {
	($host, $iface) = ($host =~ /([^:]+):(.+)/);
	die "bad host-pair: $host" if !$iface || !$host;
    }

    #$host  = ResolveIP($host);
    #$iface = ResolveIP($iface);

    return ($user, $host, $iface);
}

sub ToHost($)
{
    my ($user, $host, $iface) = SplitLogin(shift);
    return $iface;
}

# XXX
# ASHWIN: Dont call this unless you want a new latency file generated every time!
sub GenerateAndInstallLatFile($$) {
    my $nservers = shift;
    my $vservers = shift;
     
    my $filename = "n$nservers.v$vservers.lat";
    psystem("perl ../../Merc/scripts/P2PSimGraphToEmulabLats.pl -n $nservers -v $vservers -x node -b node0:15000 ../../Merc/topologies/p2psim-kingdataset-29062004.data >/tmp/$filename");
    psystem("rsync -v -e ssh -azb /tmp/$filename $ENV{USER}\@users.emulab.net:$TOPDIR/topologies/");
}

###############################################################################

# ( name@host[:iface] strings )
our @logins = 
    map { if ($_ !~ /\@/) { $_ = "$ENV{USER}\@$_" } else { $_ } } @ARGV;

# open ssh connections
tinfo "** Opening ssh connections...";
our @ssh = ParallelExec2(sub {
    my $login = shift;
    my ($user, $host, $iface) = SplitLogin($login);
    my $ssh = Travertine::SSHTunnel->new($user, $host);
    tdie "can't open connection to $login" if !$ssh;
    return $ssh;
}, @logins);

# ( [ login, ssh, num_to_start ] )
our @servers;
for (my $i=0; $i<@logins; $i++) {
    $servers[$i] = [ $logins[$i], $ssh[$i], $VIRTUAL_SERVERS ];
}

# cleanup old processes
tinfo "** Cleaning up old processes"; 
ParallelExec3($MAXPARALLEL, \&CleanServer, @ssh);

if ($KILL) { 
    exit(0); 
}

my $master = shift @servers;
my ($mUser, $mHost, $mIface) = SplitLogin($master->[0]);

--$master->[2];
unshift @servers, $master;

###############################################################################
# Arguments

my $cachesize = int(2 * log($NSERVERS)/log(2));
my $maxttl    = $NSERVERS + 10;
my $latency_file = "../Merc/topologies/$LATENCY_FILE";

my $cache_args = "";
if ($ENABLE_CACHE) {
    $cache_args = "--cache --cachesize $cachesize";
}

# by default, prediction is enabled (if you don't pass any args to quake)
my $pred_args = "";
if (!$ENABLE_PREDICTION) {
    $pred_args = " --subttl_player 100 --subttl_monster 100 --subttl_missile 100 " .
     " --pubttl_player 100 --pubttl_monster 100 --pubttl_missile 100" .
     " --subpred_player 0 --subpred_monster 0 --subpred_missile 0" .
     " --pubpred_player 0 --pubpred_monster 0 --pubpred_missile 0";
}

my $dht_args = "";
if ($DHT) {
    $dht_args = " --dht --dht_buckets 100 --dht_hub r";
    $STRIPE_DIM = '';
}

my $delta_args = "";
if ($DELTAS) {
    $delta_args = " --deltas";
}

my $monster_args = " --fightmonsters +set sv_playerhealth 1000 +set sv_monsterhealth 100 +set sv_scalehealthitem 10.0 ";
if ($NO_FIGHT_MONSTERS) {
    $monster_args = " +set sv_playerhealth 500 +set sv_scalehealthitem 5.0 ";
}

my $latency_args = "";
if ($USE_LATENCY) {
    $latency_args = " --latency --latency-file $latency_file ";
}

my $measurement_args = "";
if ($DO_MEASUREMENT) {
    $measurement_args = " --measurement --log-binary --log-dir '$LOG_DIR' " .
	"--record-obj-deltas --record-obj-interests ";
}

my $visualizer_args = "";
if ($VISUALIZER) {
    $visualizer_args = " --visualizer-ip '$VISUALIZER' ";
}

my $common_args = "--waitjoin $cache_args --verbosity $VERBOSITY " .
    "-D '$DBG_FILES' --timelimit $TIMELIMIT --nservers $NSERVERS " .
    "--bootstrap $mIface:15000 --maxttl $maxttl " . 
    "--pubtriggers --puboverwrite --item-partition $ITEM_PARTITIONING " . 
    "--spawn_type t --stripe_dim '$STRIPE_DIM' --term-passwd '$TERM_PASSWD'" .
    " $measurement_args $latency_args $monster_args $pred_args $dht_args $delta_args $visualizer_args " .
    " +set dedicated 1 +cheats 1 +exec server.cfg +set maxentities 2048 +set maxclients 64 " . 
    " +map $MAP --bbox-file bbox/$MAP.bbox";
    
###############################################################################

my @refs;
for (my $i=0; $i<@servers; $i++) {
    $refs[$i] = [];
}

# start bootstrap server
tinfo "** Staring bootstrap server...";
my $bootstrap = StartBootstrap($master->[0], $master->[1]);
tdie "couldn't start bootstrap" if !$bootstrap;
push @{$refs[0]}, [ $bootstrap, "bootstrap (" . ToHost($master->[0]) . ")" ];

while (1) {
    # wait for it to startup 
    sleep 1;
    my $tail = $bootstrap->tailLog(10);
    if ($tail =~ /read schema file successfully/) {
	last;
    } else {
	tinfo "bootstrap not ready yet... (log):\n" . thighlight($tail);
	if (! $bootstrap->isAlive()) {
	    tdie "bootstrap is dead!";
	}
    } 
}

# start master server
tinfo "** Starting master server...";
my $master_args = "$common_args --master";
if ($BOTS_ON_MASTER) {
    $master_args .= " --nbots $NUMBOTS --nmonsters $NUMMONSTERS ";
}

my $mserver = StartQuake($master->[0], $master->[1],
			 $MERC_PORTS[0], $QUAKE_PORTS[0], $master_args);
tdie "couldn't start master" if !$mserver;
push @{$refs[0]}, [ $mserver, "master (" . ToHost($master->[0]) . ")" ];

while (1) {
    # wait for it to startup
    sleep 1;
    my $head = $mserver->headLog(100);
    # XXX fixme: print something out so we know we're inited
    if ($head =~ /Current option values/) {
	last;
    } else {
	tinfo "master not ready yet... (log):\n" . thighlight($head);
	if (! $mserver->isAlive()) {
	    tdie "master is dead!";
	}
    }
}

# start the slave servers
tinfo "** Starting slave servers...";
my @srefs = ParallelExec3($MAXPARALLEL, sub {
    my $server = shift;

    my $slave_args = "$common_args --master-ip $mIface " .
	"--nbots $NUMBOTS --nmonsters $NUMMONSTERS";
    my $index      = 1;

    my @refs = ();

    while ($server->[2]-- > 0) {
	my $sserver = StartQuake($server->[0], $server->[1],
				 $MERC_PORTS[$index], $QUAKE_PORTS[$index], 
				 $slave_args);
	tdie "couldn't start slave $server->[0] \#$index" if !$sserver;
	push @refs, [ $sserver, ToHost($server->[0]) . ":$index" ];

	$index++;
    }

    return \@refs;
}, map { $_ = [ $_ ]; } @servers);

for (my $i=0; $i<@srefs; $i++) {
    foreach my $ref (@{$srefs[$i]}) {
	push @{$refs[$i]}, $ref;
    }
}

# invoke terminal if requested
if ($INVOKE_GTERM) {

    if ($RUN_AS_BG) {
	twarn "can't attach to bg funcs";
    } else {
	# gather all daemon references
	my @allrefs;
	my @alltitles;
	for (my $i=0; $i<@refs; $i++) {
	    foreach my $ref (@{$refs[$i]}) {
		push @allrefs, $ref->[0];
		push @alltitles, $ref->[1];
	    }
	}

	Travertine::RemoteDaemon::AttachToGterm(\@allrefs, 
						-titles => \@alltitles);
    }
}


# everything started!
tinfo "** Everything started!";

# print out the exeriment for later attachment (if required)
for (my $i=0; $i<@refs; $i++) {
    my $login = $logins[$i];
    foreach my $ref (@{$refs[$i]}) {
	my $title = $ref->[1];
	my $str   = $ref->[0]->serialize;

	print $OUTPUT "$login\t'$title'\t$str\n";
    }
}
