#!/usr/bin/perl
#
# $Id: BTInstall.pl 2460 2005-11-10 02:10:53Z jeffpang $
#    
##########################################################################
#
# If gethostname gives a stupid localhost.localdomain on your machine,
# be sure to provide the -H option...  
#
# Installs Colyseus and Quake on remote nodes. Modification of the Install.pl 
# script. This one pushes the software using BitTorrent. 
#
# The installation proceeds as follows:
#
# (1) A local copy of the 'distribution' is made in $tmp by
#     sym-linking files from $LOCAL_TOPDIR. A .torrent file is 
#     made from this. 
# (2) The tracker and seed-node are started locally. 
# (3) A pre-install procedure is run on all the remote hosts. The 
#     remote nodes should have the bittorrent rpm installed.
#     The .torrent file as well as one additional python script 
#     is rsync'ed to all the nodes. 
# (4) The btpull.py script is started at each remote host, which pulls the
#     software using bittorrent.
# (5) A post-install procedure is run on all the remote hosts.
# (6) [Optional] The local copy of the distribution is removed.

use strict;
use Getopt::Std;
use vars qw($opt_t $opt_l $opt_c $opt_p $opt_v $opt_H $opt_T $opt_k);
use Travertine;

###############################################################################
# Options

# We have to execute this in addition to the normal application _PRE function

our $pre = sub {
    # ps behaves badly sometimes, so have to resort to killall :(
    psystem("killall -9 python >/dev/null 2>&1");
};

our $APP_CONF = shift @ARGV;
Usage() if (!$APP_CONF);

our @APP_INST_DIST;
our $APP_INST_PRE;
our $APP_INST_POST;

tdie "bad conf file: '$APP_CONF'" if !-f $APP_CONF;
no strict;
require "$APP_CONF";
use strict;

if (!defined @APP_INST_DIST) { 
    tdie "variable \@APP_INST_DIST not defined; nothing to install?";
}
if (!defined $APP_INST_PRE) { 
    tdie "subroutine \$APP_INST_PRE not defined";
}
if (!defined $APP_INST_POST) {
    tdie "subroutine \$APP_INST_POST not defined";
}

###############################################################################

getopts("t:l:cp:vH:T:k");

if (@ARGV == 0) {
    Usage();
}

sub Usage() {
    print STDERR "usage: BTInstall.pl AppConf.pl [options] [user1\@]host1 [user2\@]host2 ...\n\n";
    print STDERR "       -l       local top directory (pubsub dir)\n";
    print STDERR "       -t dir   install in this directory\n";
    print STDERR "       -T dir   temp directory at the remote host (def: /tmp)\n";
    print STDERR "       -c       cleanup local tmp dist after finished\n";
    print STDERR "       -p       max parallel executions in flight (def: 10)\n";
    print STDERR "       -H       hostname to use for ourselves\n";
    print STDERR "       -v       verbose output\n";
    print STDERR "       -k       just kill all remote bt instances\n";
    print STDERR "\n";
    exit 1;
}

our $TOPDIR;
our $LOCAL_TOPDIR = $ENV{HOME};
our $CLEAN = defined $opt_c;
our $MAXPARALLEL = $opt_p || 15;
our $VERBOSE =  defined $opt_v;

### these are BTInstall specific
our $HOSTNAME = defined $opt_H ? $opt_H : `hostname`;
chomp $HOSTNAME;
# pick a random tracker port so that old instances are unlikely to contact us
# xxx really should pass all the hosts to btsource so it knows exactly which
#     ones it should wait for and ignore the rest.
our $TRACKER_PORT = 55000 + int(rand(10000));
our $TRACKER_URL = "http://$HOSTNAME:$TRACKER_PORT/announce";
our $DST_TMP_DIR = defined $opt_T ? $opt_T : "/tmp";

if ($opt_t) {
    $TOPDIR = $opt_t;
} else {
    tdie "need -t topdir!";
}

if ($opt_l) {
    $LOCAL_TOPDIR = $opt_l;
}

## Temp directory where we build the rsync dist locally
our $tmp = `basename $TOPDIR`;
chomp $tmp;
$tmp = "/tmp/$tmp";

our @logins = @ARGV;
@logins = map { if ($_ !~ /\@/) { $_="$ENV{USER}\@$_" } else { $_ } } @logins;

###############################################################################
tinfo "* checking if bittorrent is installed";
my $ret = psystem "python -c 'from BitTorrent.track import track;' 2>/dev/null ";
if ($ret) {     
    twarn "--BitTorrent modules not found! Install the bittorrent rpm or \nset PYTHONPATH to the location of the BitTorrent/ module--";
    exit 1;
}

if ($opt_k) {
    tinfo "* killing remote bt clients";
    do_all($pre);
    exit 0;
}

tinfo "* making current dist";
make_dist();

tinfo "* running pre-install";
# save duplicate ssh connections;
my $pre2 = Travertine::CombineFuncs($APP_INST_PRE, $pre); 
do_all($pre2);

tinfo "* starting tracker";
my $trpid = start_tracker_seed(scalar @logins);

tinfo "* performing sync";
# This function pushes the relevant bittorrent scripts to the node
# along with the .torrent file. This function runs locally so we 
# can rely on global variables
#
our $sync = sub {
    my $login = shift;

    # Assume this is running in the directory containing btpull.py
    my $ret = psystem("rsync -rtLz -e ssh btpull.py $tmp.torrent $login:$DST_TMP_DIR/");
    twarn "rsync to $login failed (see above messages)" if $ret;
};

do_sync();

tinfo "* starting bittorrent processes on the remote hosts";
sub bt_start {
    my ($tmpdir, $watchdog, $torrent, $destdir) = @_;
    my $pid = 0;
    
    $destdir = `dirname $destdir`;
    chomp $destdir;
     # XXX TODO: just do the rsync here. saves ssh connections.
    exec "$tmpdir/btpull.py $watchdog --minport 55000 --maxport 55009 --max_uploads 15 --responsefile $torrent --saveas $destdir";
}

my @rbrefs = ParallelExec3($MAXPARALLEL, sub {
    my $login = shift;
    my ($user, $mach) = split(/\@/, $login);

    my $ssh = Travertine::SSHTunnel->new($user, $mach);
    if (!$ssh) {
	twarn "can't open sshTunnel to $user\@$mach";
	return undef;
    }

    tinfo "$mach started." if $VERBOSE;
    my $torrent = `basename $tmp`; chomp $torrent;
    $torrent = "$DST_TMP_DIR/$torrent.torrent";

    my @args = ($DST_TMP_DIR, $HOSTNAME, $torrent, $TOPDIR);
    return ExecRemoteFunc($ssh, \&bt_start, \@args, -background => 1, -log => "$DST_TMP_DIR/btinstall.log");
}, @logins);

tinfo "* waiting for the nodes to finish downloading";
waitpid($trpid, 0); # wait for the tracker process to end!

=start
# try to use only one ssh connection here... 

# now kill the background btpull.py processes
tinfo "* stopping remote bittorrent processes";
ParallelExec3($MAXPARALLEL, sub {
    my $r = shift;
    $r->stop(); 
}, @rbrefs);
=cut

tinfo "* running post-install";
my $post = sub {
    psystem("killall -9 python >/dev/null 2>&1");
};

my $post2 = Travertine::CombineFuncs($APP_INST_POST, $post); 
do_all($post2);

if ($CLEAN) {
    tinfo "* cleaning up tmp dist";
    psystem("rm -rf $tmp");
}

###############################################################################
sub start_tracker_seed {
    my $ndownloaders = shift;

    my $cmd = "./btsource.py -p $TRACKER_PORT -T $tmp -n $ndownloaders";

    my $pid = fork();
    if ($pid > 0) {
	return $pid;
    }
    else {
	 # psystem "xterm -T \"Source Node\" -e bash -c \"$cmd\""; 
	psystem "$cmd";
	exit();
    }
}

sub make_dist {
    my $ret = 0;

    psystem "rm -rf $tmp";
    foreach my $glob (@APP_INST_DIST) {
	my $dirname = $glob;
	$dirname =~ s/[^\/]*$//;
	$ret ||= psystem "mkdir -p $tmp/$dirname";
	$ret ||= psystem "cp -sHLrf $LOCAL_TOPDIR/$glob $tmp/$dirname";
	
	tdie "failed: $!" if $ret;
    }
    psystem "btmakemetafile.py $tmp $TRACKER_URL --target $tmp.torrent";
}

sub do_all {
    my $func   = shift;
    return if !$func;

    ParallelExec3($MAXPARALLEL, sub {
	my $login = shift;
	my ($user, $mach) = split(/\@/, $login);
	tinfo "$mach started." if $VERBOSE;
	rsystem($user, $mach, $func, $TOPDIR, $user);
	tinfo "$mach completed." if $VERBOSE;
    }, @logins);
}


sub do_sync {
    tdie "no sync function defined" if !$sync;
    ParallelExec3($MAXPARALLEL, sub {
	my $login = shift;
	my ($user, $mach) = split(/\@/, $login);
	tinfo "$mach started." if $VERBOSE;
	my $ret = &{$sync}($login);
	tinfo "$mach completed." if $VERBOSE;
	return $ret;
    }, @logins);
}
