////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////
/* -*- Mode:C++; c-basic-offset:4; tab-width:4; indent-tabs-mode:nil -*- */

/**************************************************************************
  Terminal.h

  Class representing a publication.

begin           : Nov 6, 2002
copyright       : (C) 2002-2003 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2002-2003 Justin Weisz       ( jweisz@cs.cmu.edu )

***************************************************************************/


#ifndef __TERMINAL__H
#define __TERMINAL__H

#include <sstream>
#include <om/common.h>
#include <util/ServerSocket.h>
#include <util/Thread.h>
#include <util/CondVar.h>

#define THREAD_POOL_SIZE 5
#define MAX_MSG_SIZE     512

#define RESP_OK               "OK\n"
#define RESP_NONE             "NONE\n"
#define RESP_INVALID_COMMAND  "INVALID COMMAND\n"
#define RESP_INVALID_ARGS     "INVALID ARGUMENTS\n"
#define RESP_NOT_ENABLED      "PROTO NOT ENABLED\n"
#define RESP_TOO_MANY_CLIENTS "TOO MANY CLIENTS CONNECTED\n"

    /**
     * Handler for a particular command given to the terminal.
     *
     * When a user present the command 'cmd' or 'abbrev' then the
     * 'handler' function is executed. The arguments to the command are in
     * the null terminated list of strings 'args' and the result of the
     * command should be printed on the output stream. The convention is
     * to prefix the output with OK if it was successful and ERROR if it
     * failed. 'usage' should display the arguments the command takes
     * (e.g., in "<arg1> <arg2>" format") and 'help' should describe what
     * the command does.
     */
    typedef struct {
	const char *cmd;
	const char *abbrev;
	void (*handler)(ostream& out, char **args);
	const char *usage;
	const char *help;
    } InputHandler;

typedef vector<InputHandler *> InputHandlers;

class Manager;
class TerminalWorker;

/**
 * Remote execution terminal used to inspect the contents and status
 * of the Manager at runtime. Enabled in <code>Manager.h</code>
 */
class Terminal : public Thread
{
    ServerSocket m_LocalSock;
    TerminalWorker *pool[THREAD_POOL_SIZE];

 public:
    Terminal(uint16 port, Manager *manager);
    virtual ~Terminal();

    void Run();
};

/**
 * Helper thread to Terminal.
 */
class TerminalWorker : public Thread
{
    ServerSocket m_Sock;
    bool m_Finished;
    CondVar m_Sleep;

    void _Run();

 public:

    static void handleVersion(ostream& out, char **args);
    static void handleHelp(ostream& out, char **args);
    static void handleMigrate(ostream& out, char **args);
    static void handleShow(ostream& out, char **args);
    static void handleStore(ostream& out, char **args);
    static void handlePending(ostream& out, char **args);
    static void handleGraph(ostream& out, char **args);
    static void handleGraphVis(ostream& out, char **args);
    static void handleLoad(ostream& out, char **args);
    static void handleLoadMap(ostream& out, char **args);
    static void handleObjectInfo(ostream& out, char **args);
    static void handleBenchmark(ostream& out, char **args);
    static void handleRotate(ostream& out, char **args);
    static void handleReplicaConn(ostream& out, char **args);
    static void handleDelGUIDs(ostream& out, char **args);
    static void handleSubList(ostream& out, char **args);
    static void handlePubList(ostream& out, char **args);
    static void handleQuit(ostream& out, char **args);

    //static int parse_val(ostream& out, Value *val, char *str);
    static guid_t parse_guid(char *str);
    static sid_t parse_sid(char *str);
    static GObject *find_obj(char *name);

    TerminalWorker(ServerSocket& sock);
    virtual ~TerminalWorker();

    void Restart(ServerSocket& sock);
    void Run();
    bool Finished() { return m_Finished; }
};

#endif // __EVENT__H
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
