////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************
  ReplicaConnection.h

begin           : Nov 6, 2002
copyright       : (C) 2002-2004 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2003-2004 Jeff Pang      ( jeffpang@cs.cmu.edu   )

***************************************************************************/

#ifndef __REPLICA_CONNECTION_H__
#define __REPLICA_CONNECTION_H__

#include <list>
#include <mercury/common.h>
#include <mercury/Message.h>
#include <om/common.h>
#include <om/GObject.h>
#include <om/DeltaInfo.h>
#include <om/OMLogs.h>

    class Manager;
class GInterest;
struct MsgAppUpdate;
struct MsgAppUpdateAck;

///////////////////////////////////////////////////////////////////////////////
///// Remote Replica Connection Maintenance

// After DELTA_WINDOW_SIZE outstanding packets, we just assume the remote 
// replicas are so horribly out of date that we just reset the window and 
// resend full replicas with a complete delta mask. (max is 255)
#define DELTA_WINDOW_SIZE 100 // = 10sec at 10fps

typedef list<MsgAppUpdate *>  DeltaWindow;
typedef DeltaWindow::iterator DeltaWindowIter;

class RepConnnectionSender;

struct RepConnSendStats {
    static const uint32 MEASUREMENT_INTERVAL = 5000;

    SID sid;
    TimeVal intervalStart;
    uint16 sent;
    uint16 async;
    uint16 lost;
    uint32 window;
    uint32 samples;

    RepConnSendStats(const SID& sid) : 
	sid(sid), sent(0), async(0), lost(0), window(0), samples(0) {
	OS::GetCurrentTime(&intervalStart);
	// synch to interval
	uint64 msec = ((uint64)intervalStart.tv_sec)*MSEC_IN_SEC+
	    ((uint64)intervalStart.tv_usec)/USEC_IN_MSEC;
	msec = (msec/MEASUREMENT_INTERVAL)*MEASUREMENT_INTERVAL;
	intervalStart.tv_sec = msec/MSEC_IN_SEC;
	intervalStart.tv_usec = (msec%MSEC_IN_SEC)*USEC_IN_MSEC;
    }

    void Record(const TimeVal& now) {
	if (now - intervalStart >= MEASUREMENT_INTERVAL) {
	    RepConnSendDroppedEntry e(sid, sent, async, lost,
				  samples > 0 ? window/(float)samples : 0);
	    LOG(RepConnSendDroppedLog, e);
	    
	    intervalStart = intervalStart + MEASUREMENT_INTERVAL;
	    sent = async = lost = window = samples = 0;
	}
    }
};

struct RepConnRecvStats {
    static const uint32 MEASUREMENT_INTERVAL = 5000;

    SID sid;
    TimeVal intervalStart;
    uint16 recv;
    uint16 missing;
    uint16 old;

    uint16 async;
    uint16 async_missing;
    uint16 async_old;

    RepConnRecvStats(const SID& sid) : 
	sid(sid), recv(0), missing(0), old(0),
	 async(0), async_missing(0), async_old(0) {
	OS::GetCurrentTime(&intervalStart);
	// synch to interval
	uint64 msec = ((uint64)intervalStart.tv_sec)*MSEC_IN_SEC+
	    ((uint64)intervalStart.tv_usec)/USEC_IN_MSEC;
	msec = (msec/MEASUREMENT_INTERVAL)*MEASUREMENT_INTERVAL;
	intervalStart.tv_sec = msec/MSEC_IN_SEC;
	intervalStart.tv_usec = (msec%MSEC_IN_SEC)*USEC_IN_MSEC;
    }

    void Record(const TimeVal& now) {
	if (now - intervalStart >= MEASUREMENT_INTERVAL) {
	    RepConnRecvDroppedEntry e(sid, recv, missing, old,
				      async, async_missing, async_old);
	    LOG(RepConnRecvDroppedLog, e);
	    
	    intervalStart = intervalStart + MEASUREMENT_INTERVAL;
	    recv = missing = old = async = async_missing = async_old = 0;
	}
    }
};

/**
 * Sender side state for delta-connections.
 */
class ReplicaConnectionSender {
 private:
    friend class Manager;
    friend ostream& operator<<(ostream& out, ReplicaConnectionSender *conn);

    Manager *m_Manager;

 public: // testing
    SID    m_SID;                // sid of remote node
    uint32 m_LastSeqNoAcked;     // last delta packet ack'd by receiver
    uint32 m_LastSeqNoDropped;   // receiver dropped packet, 0 if none
    uint32 m_LastSeqNoSent;      // last delta packet sent to receiver
    byte   m_Epoch;              // seqno epoch

    DeltaWindow   m_Outstanding; // outstanding packets (not yet ack'd)
    DeltaMap      m_Deltas;      // replicas on this connection + pending masks

    TimeVal m_LastSendTime;
    RepConnSendStats m_Stats;

    void ResetConnection();
 public:

    ReplicaConnectionSender(SID sid, Manager *manager) : 
	m_Manager(manager), m_SID(sid), 
	m_LastSeqNoAcked(0), m_LastSeqNoDropped(0), 
	m_LastSeqNoSent(0), m_Epoch(0), m_LastSendTime(TIME_NONE),
	m_Stats(RepConnSendStats(sid)) {}
    virtual ~ReplicaConnectionSender() {}

    inline SID GetSID() { return m_SID; }

    MsgAppUpdate *CreateUpdate(bool onlyNew = false);
    void HandleUpdateAck(MsgAppUpdateAck *msg);

    void RegisterGUID(GUID guid, bool isDelete=false, bool isNew=false,
		      bool isProactive = false,
		      GInterest *reg = NULL /* MEASUREMENT ONLY */);
    void UnRegisterGUID(GUID guid);
    DeltaMap *GetDeltaMap() { return &m_Deltas; }
    void GarbageCollectRegistrations();
};

/**
 * Receiver side state for delta-connections.
 */
class ReplicaConnectionReceiver {
 private:
    friend class Manager;
    friend ostream& operator<<(ostream& out, ReplicaConnectionReceiver *conn);

    Manager *m_Manager;

 public: // testing
    SID     m_SID;
    byte    m_Epoch;
    uint32  m_LastSeqNoAck; // last delta packet recv'd to ack by us
    TimeVal m_LastRecvTime; // last time we got a non-empty update

    RepConnRecvStats m_Stats;

    void ResetConnection(byte epoch, uint32 updateSeqno);
 public:

    ReplicaConnectionReceiver(SID sid, byte initEpoch, 
			      uint32 initSeqno, Manager *manager) : 
	m_Manager(manager), m_SID(sid), m_Epoch(initEpoch),
	m_LastSeqNoAck(initSeqno), m_LastRecvTime(TIME_NONE),
	m_Stats(RepConnRecvStats(sid)){}
    virtual ~ReplicaConnectionReceiver() {}

    inline SID GetSID() { return m_SID; }
    inline byte   GetEpoch() { return m_Epoch; }
    inline uint32 GetLastSeqNoAck() { return m_LastSeqNoAck; }
    void HandleUpdate(MsgAppUpdate *msg);
};

ostream& operator<<(ostream& out, ReplicaConnectionSender *conn);
ostream& operator<<(ostream& out, ReplicaConnectionReceiver *conn);

typedef map<SID, ReplicaConnectionSender *, less_SID> 
		   OutgoingReplicaConnectionMap;
typedef OutgoingReplicaConnectionMap::iterator
OutgoingReplicaConnectionMapIter;

typedef map<SID, ReplicaConnectionReceiver *, less_SID> 
		   IncomingReplicaConnectionMap;
typedef IncomingReplicaConnectionMap::iterator
IncomingReplicaConnectionMapIter;

#endif
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
