////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************
  PubSubManager.h

begin           : Nov 6, 2002
copyright       : (C) 2002-2004 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2003-2004 Jeff Pang      ( jeffpang@cs.cmu.edu   )

***************************************************************************/

#ifndef __PUBSUB_MANAGER_H__
#define __PUBSUB_MANAGER_H__

#include <wan-env/WANMercuryNode.h>
#include <mercury/Message.h>
#include <util/DataChunker.h>
#include <util/TimedStruct.h>
#include <om/common.h>
#include <om/OMEvent.h>
#include <om/OMInterest.h>

    typedef map<GUID, EventSet *, less_GUID> EventMMap;
typedef EventMMap::iterator EventMMapIter;

typedef multimap<GUID, GUID, less_GUID> InterestIDMap;
typedef InterestIDMap::iterator InterestIDMapIter;

///////////////////////////////////////////////////////////////////////////////

class Manager;
class OMApplication;

/**
 * The PubSubManager manages the pubs/subs sent by objects owned on this
 * node and ensures the correct TTL's, etc. are respected.
 */
class PubSubManager {
 private:

    WANMercuryNode *m_Router;
    Manager     *m_Manager;
    OMApplication *m_App;

    // The current event of our objects
    EventMMap m_MyEvents;
    // The current interest of our objects
    InterestMap m_MyInterests;

    // A map mapping Interest GUIDs => set<object GUIDs> -- this is needed
    // to figure out which objects the tags actually correspond to
    InterestIDMap m_InterestIDMap;

    // The events to register in the next cycle
    EventList m_EventRegQueue;
    // The events to unregister in the next cycle
    EventList m_EventUnRegQueue;
    // The interests to register in the next cycle
    InterestList m_InterestRegQueue;
    // The interests to unregister in the next cycle
    InterestList m_InterestUnRegQueue;

    // Events we registered that matched interests we own
    EventMap m_LocalMatches;

    // Events we have received but have not yet expired
    // each frame we look at these again to see if we get any new matches
    EventMap  m_EventCache;
    EventList m_EventCacheIter;

    void MatchLocalItems();
    void ExpireStaleItems();

 public:

    PubSubManager(Manager *manager, WANMercuryNode *router);
    virtual ~PubSubManager();

    // This auto unregisters the old pub under the same guid if any
    void RegisterEvent(OMEvent *pub);	
    void UnregisterEvent(OMEvent *pub);

    void RegisterInterest(OMInterest *in);
    void UnregisterInterest(OMInterest *in);
    // map from: ObjectGUID -> InterestGUID
    void ChangeInterestAssoc(GUIDMap *assoc);

    EventSet *GetCurrEvents(const GUID& guid);
    InterestMap *GetCurrInterests() { return &m_MyInterests; }
    InterestIDMap *GetInterestIDMap() { return &m_InterestIDMap; }

    OMEvent *GetNextLocalMatch();

    void AddCachedEvent(OMEvent *ev);
    // must be explicitly re-added to the cache if desired
    void BeginCachedEventIter();
    OMEvent *GetNextCachedEvent();

    void Send(uint32 timeout);
    void Maintenance(uint32 timeout);

    void DumpInterests(ostream& out);
    void DumpEvents(ostream& out);
};

#endif
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
