////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////
#include <mercury/Scheduler.h>
#include <mercury/Message.h>         // for stupid MsgPublication
#include <mercury/Interest.h>
#include <mercury/Event.h>
#include <fstream>
#include <mercury/Hub.h>
#include <om/OMPubsubStore.h>

static const bool USE_RTREE = true;
static const int  RTREE_DIMENSION = 2; // XXX: hack. this really has to go to the app (testgame),
// but we dont have time right now.


void EvWrapper::GetExtent (Rect<Value> *extent) {
    Event *ev = pmsg->GetEvent ();
    ASSERT (extent->dim == RTREE_DIMENSION);
    ASSERT (ev->GetNumConstraints () >= extent->dim);

    for (int i = 0; i < extent->dim; i++) {
	Constraint *cst = ev->GetConstraint (i);
	extent->min[i] = cst->GetMin ();
	extent->max[i] = cst->GetMax ();
    }
}

void InWrapper::GetExtent (Rect<Value> *extent) {
    ASSERT (extent->dim == RTREE_DIMENSION);
    ASSERT (in->GetNumConstraints () >= extent->dim);

    for (int i = 0; i < extent->dim; i++) {
	Constraint *cst = in->GetConstraint (i);
	extent->min[i] = cst->GetMin ();
	extent->max[i] = cst->GetMax ();
    }
}

OMPubsubStore::OMPubsubStore () :
    m_PubTree (RTree<EvWrapper, Value> (RTREE_DIMENSION)),
    m_SubTree (RTree<InWrapper, Value> (RTREE_DIMENSION)) 
{

}

void OMPubsubStore::StoreSub (Interest *in)
{
    _InsertSub (in->Clone ());
}

OMPubsubStore::IntLstIter OMPubsubStore::_DeleteSub (OMPubsubStore::IntLstIter& iter)
{
    InWrapper *iwr = *iter;
    Interest *in = iwr->in;

    if (USE_RTREE) {
	bool erased = m_SubTree.Erase (iwr);
	ASSERT (erased);
    }

    delete in;
    delete iwr;
    return m_SubList.erase (iter);
}

void OMPubsubStore::_InsertSub (Interest *in)
{
    InWrapper *iwr = new InWrapper (in);

    m_SubList.push_back (iwr);
    if (USE_RTREE) {
	m_SubTree.Insert (iwr);
	m_SubTree.CheckInvariants ();
    }
}

void OMPubsubStore::_DeletePub (PubMsgMapIter& giter)
{
    EvWrapper *ewr = giter->second;
    MsgPublication *prevMsg = ewr->pmsg;

    if (USE_RTREE) {
	bool erased = m_PubTree.Erase (ewr);
	ASSERT (erased);
    }

    delete prevMsg;
    delete ewr;
    prevMsg = NULL;
    m_TriggerMap.erase (giter);
}

void OMPubsubStore::_InsertPub (MsgPublication *pmsg)
{
    EvWrapper *ewr = new EvWrapper (pmsg);
    m_TriggerMap.insert (PubMsgMap::value_type (pmsg->GetEvent (), ewr));

    if (USE_RTREE) {
	m_PubTree.Insert (ewr);
	m_PubTree.CheckInvariants ();
    }
}

void OMPubsubStore::StoreTrigger (MsgPublication *pmsg)
{
    // store the publication and associate an expiry time...

    Event *ev = pmsg->GetEvent ();
    PubMsgMapIter giter = m_TriggerMap.end ();     // set it to some bad value

    // TTL = 0 ==> this pub is not meant to be stored!
    if (ev->GetLifeTime () == 0)
	return;

    NOTE(Mercury::TRIGGERMAP_COUNT, m_TriggerMap.size ());

    giter = m_TriggerMap.find (ev);   // this channels "ev->LessThan(otherev)"

    if (giter != m_TriggerMap.end ())
    {
	// we found a previous publication
	bool overwrite = ev->OverwriteEvent (giter->second->pmsg->GetEvent ());
	if (!overwrite) 
	    return;
    }
    // we can either overwrite or insert the new trigger now.

    if (giter != m_TriggerMap.end()) {
	_DeletePub (giter);

    }

    _InsertPub (pmsg->Clone ());
}

static void _sub_pred (list<Interest *> *pmatch, InWrapper *iwr) {
    pmatch->push_back (iwr->in);
}

void OMPubsubStore::GetOverlapSubs (MsgPublication *pmsg, list<Interest *> *pmatch)
{
    NOTE(SUBLIST_COUNT, m_SubList.size());

    if (USE_RTREE) {
	EvWrapper t (pmsg);
	return m_SubTree.GetOverlaps (&t, wrap (_sub_pred, pmatch));
    }

    Event *pub = pmsg->GetEvent ();
    for (IntLstIter it = m_SubList.begin (); it != m_SubList.end (); ++it) {
	Interest *in = (*it)->in;
	if (in->Overlaps (pub))
	    pmatch->push_back (in);
    }
}

static void _trigger_pred (list<MsgPublication *> *pmatch, EvWrapper *ewr) {
    pmatch->push_back (ewr->pmsg);
}

void OMPubsubStore::GetOverlapTriggers (Interest *in, list<MsgPublication *> *pmatch)
{
    if (USE_RTREE) {
	InWrapper t (in);
	return m_PubTree.GetOverlaps (&t, wrap (_trigger_pred, pmatch));
    }

    for (PubMsgMapIter it = m_TriggerMap.begin (); it != m_TriggerMap.end (); ++it) {
	MsgPublication *pmsg = it->second->pmsg;	
	Event *pub = pmsg->GetEvent ();

	if (in->Overlaps (pub))
	    pmatch->push_back (pmsg);
    }
}

void OMPubsubStore::Clear ()
{
    for (IntLstIter it = m_SubList.begin (); it != m_SubList.end (); /* ++it */) {
	it = _DeleteSub (it);
    }
    m_SubList.clear ();

    for (PubMsgMapIter giter = m_TriggerMap.begin (); giter != m_TriggerMap.end (); /* ++giter */) {
	PubMsgMapIter oit (giter);
	++oit;

	_DeletePub (giter);
	giter = oit;
    }   
}

// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
