////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////
#include <om/OMEvent.h>
#include <om/OMInterest.h>
#include <om/OMLoadManager.h>

const string ATTR_LOAD               = string("Load");
const string ATTR_TO_LOW_WATER_MARK  = string("ToLowWaterMark");
const string ATTR_TO_HIGH_WATER_MARK = string("ToHighWaterMark");

EventType PUB_OM;

void OM_RegisterEventTypes () {
    PUB_OM = REGISTER_TYPE (Event, OMEvent);
    PUB_OM_LOAD = REGISTER_TYPE (Event, OMLoadEvent);
}

OMEvent::OMEvent() :
    MercuryEvent(),  TimedStruct((uint32)0),
    m_GUID(GUID_NONE), m_SID(SID_NONE), m_SequenceNumber(0), m_Update(NULL)
{
}

OMEvent::OMEvent(guid_t target, sid_t loc) :
    MercuryEvent(), TimedStruct((uint32)0), 
    m_GUID(target), m_SID(loc), m_SequenceNumber(0), m_Update(NULL)
{
}

OMEvent::OMEvent(const OMEvent& other) :
    MercuryEvent(other), TimedStruct(other),
    m_GUID(other.m_GUID), m_SID(other.m_SID), 
    m_SequenceNumber(other.m_SequenceNumber), 
    m_Tags (other.m_Tags),
    m_Update(other.m_Update ? new GUpdate(*other.m_Update) : NULL)
{
}

OMEvent::OMEvent(Packet * pkt) :
    MercuryEvent(pkt), TimedStruct((uint32)0), m_SequenceNumber(0)
{
    m_GUID = GUID(pkt);
    m_SID  = SID(pkt);

    uint32 numTags = pkt->ReadInt();
    for (uint32 i = 0; i < numTags; i++) {
	GUID tag(pkt);
	m_Tags.insert(tag);
    }

    bool fullUpdate = pkt->ReadByte();
    if (fullUpdate) {
	m_Update = new GUpdate(pkt);
    }
    else {
	m_Update = NULL;
    }

    m_SequenceNumber = pkt->ReadByte();
}


void OMEvent::Serialize(Packet * pkt)
{
    MercuryEvent::Serialize(pkt);

    // XXX TODO: Optimize m_IsPointEvent

    m_GUID.Serialize(pkt);
    m_SID.Serialize(pkt);

    uint32 numTags = m_Tags.size();
    pkt->WriteInt(numTags);
    for (Tags::iterator it = m_Tags.begin(); it != m_Tags.end(); it++) {
	GUID guid = *it;
	guid.Serialize(pkt);
    }

    if (m_Update) {
	pkt->WriteByte(1);
	m_Update->Serialize(pkt);
    } else {
	pkt->WriteByte(0);
    }

    pkt->WriteByte(m_SequenceNumber);	
}

uint32 OMEvent::GetLength()
{
    uint32 length = MercuryEvent::GetLength();

    // XXX TODO: Optimize m_IsPointEvent

    length += m_GUID.GetLength() + m_SID.GetLength();

    length += 4;   // numTags
    for (Tags::iterator it = m_Tags.begin(); it != m_Tags.end(); it++) {
	GUID guid = *it;
	length += guid.GetLength();
    }

    length += 1; // isUpdate
    if (m_Update) {
	length += m_Update->GetLength();
    }

    length += 1;    // m_SequenceNumber

    return length;
}

bool OMEvent::IsPointEvent()
{
    // XXX TODO: cache this info! don't recompute every time!
    for (int i = 0, len = m_Constraints.size(); i < len; i++)
    {
	if (m_Constraints[i].GetMin() != m_Constraints[i].GetMax()) {
	    return false;
	}
    }
    return true;
}

void OMEvent::AddTuple(const Tuple& t)
{
    Constraint c(t.GetAttrIndex(), t.GetValue(), t.GetValue());
    AddConstraint(c);
}

const Value *OMEvent::GetValue(int attr)
{
    Constraint *c = GetConstraintByAttr(attr);
    ASSERT( c->GetMin() == c->GetMax() );
    return &c->GetMin();
}

void OMEvent::AddTag(GUID tag) 
{ 
    m_Tags.insert(tag); 
}

void OMEvent::AddTags(OMInterest *in)
{
    Tags o = in->GetTags();
    for (Tags::iterator it = o.begin(); it != o.end(); it++) {
	m_Tags.insert(*it);
    }
}

Tags& OMEvent::GetTags() 
{ 
    return m_Tags; 
}

bool OMEvent::LessThan (const Event *oe) const 
{
    // Jeff: what the heck was the second clause for?
    if (Event::LessThan(oe)) // || Event::LessThan(oe))
	return Event::LessThan(oe);
    ASSERT(oe->GetType() == PUB_OM);
    return less_GUID()(m_GUID, ((OMEvent *) oe)->GetGUID());
}

bool OMEvent::OverwriteEvent(Event *old) const 
{
    if (old->GetType() != PUB_OM)
	return false;

    OMEvent *o = (OMEvent *) old;
    bool ret = GetGUID() != o->GetGUID() || IsNewer(o);
    return ret;
}

void OMEvent::Print(FILE * stream)
{
    MercuryEvent::Print(stream);
}

ostream& OMEvent::Print (ostream& out) const 
{    
    MercuryEvent::Print (out);

    OMEvent *evt = (OMEvent *) this;

    out << " sequencenum=" << (int) evt->GetSeqno();    
    out << " guid=" << evt->GetGUID() << " sid=" << evt->GetSID(); 
    out << " tags=[";
    Tags tags = evt->GetTags();
    for (Tags::iterator it = tags.begin(); it != tags.end(); it++) {
	if (it != tags.begin()) out << ",";
	out << *it;
    }
    out << "]";
    out << " lifetime=" << evt->GetLifeTime()
	<< " deathtime=" << evt->GetDeathTime() << ")";
    return out;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
