////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////
/* -*- Mode:c++; c-basic-offset:4; tab-width:4; indent-tabs-mode:t -*- */

/**************************************************************************
  Policy.cpp

begin           : Oct 16, 2003

copyright       : (C) 2003      Jeff Pang        ( jeffpang@cs.cmu.edu )
(C) 2003      Justin Weisz     (  jweisz@cs.cmu.edu  )

***************************************************************************/

#include <algorithm>
#include <fstream>
#include <om/MigrationPolicy.h>
#include <om/Manager.h>
#include <om/GameAdaptor.h>
#include <om/GObject.h>
#include <util/Benchmark.h>

    ostream& operator <<(ostream& out, MigrateMap& mm) {
    out << "[";
    for (MigrateMapIter i = mm.begin(); i != mm.end(); i++) {
	if (i != mm.begin()) out << ",";
	out << i->first << "->" << i->second;
    }
    out << "]";
    return out;
  }

//////////////////////////////////////////////////////////////////////////////

void StaticPlacementPolicy::Migrate(MigrateMap& toMigrate,
				    GObject *locChange) {
    return;
}

//////////////////////////////////////////////////////////////////////////////

void RandomPlacementPolicy:: Migrate(MigrateMap& toMigrate,
				     GObject *locChange) {
    GObject *obj;

    if (m_ServersList.size() == 0) {
	DB(0) << "no servers to pick from!" << endl;
	return;
    }

    ObjectStore *store = m_Manager->GetObjectStore();
    store->Begin();
    while ((obj = store->Next()) != NULL) {
	if (!obj->IsReplica() && obj->IsMigratable() && 
	    drand48() > 0.9) {
	    SID dest = m_ServersList[ (int)(m_ServersList.size()*drand48()) ];

	    toMigrate[obj->GetGUID()] = dest;
	}
    }
}

//////////////////////////////////////////////////////////////////////////////

/*

static const string ATTR_NODE = string("node");

#define READ(var) { \
in >> (var); \
if (!in) Debug::die ("error reading '%s' at %s:%d", #var, file, lineno); \
}

void RegionPlacementPolicy::loadRegionMap(const char *file) {
ifstream in(file);

if (!in) {
Debug::die("can't open region file '%s'", file);
}

int  lineno = 0;

while (!in.eof()) {
lineno++;

if (in.peek() == EOF) break;
uint32 node;
SID  sid;
READ(node);
READ(sid);
in.ignore(LINEBUF_SIZE, '\n');

ASSERTM(regionMap.find(node) == regionMap.end(),
"region %d mapped twice in region map file %s",
node, file);

regionMap[node] = sid;
}

in.close();
}

*/

RegionPlacementPolicy::RegionPlacementPolicy()
{
}

// for now assume we just randomly assign an even number of regions to
// each server (region modulo numServers)
SID RegionPlacementPolicy::getResponsibleServer(uint32 region) {
    RegionMapIter iter = ManagerParams::REGION_MAP.find(region);
    ASSERT(iter != ManagerParams::REGION_MAP.end());
    return iter->second;
}

void RegionPlacementPolicy::MigrateFreq(MigrateMap& toMigrate) {

    ObjectStore *store = m_Manager->GetObjectStore();

    GObject *obj;
    store->Begin();
    while((obj = store->Next()) != NULL) {
	if (!obj->IsReplica() && obj->IsMigratable()) {
	    SID sid = getResponsibleServer( obj->GetRegion() );

	    if (sid != m_Manager->GetSID()) {
		// migrate it to that server!
		toMigrate[ obj->GetGUID() ] = sid;
		/*
		  INFO << obj->GetGUID() << " moved to " << obj->GetRegion()
		  << ": migrated to: " 
		  << sid << endl;
		*/
	    }
	}
    }
}

//////////////////////////////////////////////////////////////////////////////

ostream& operator<<(ostream& out, Component& c)
{
    out << "[";
    for (ComponentIter k=c.begin(); k != c.end(); k++) {
	if (k != c.begin()) out << ",";
	out << (*k)->GetGUID();
    }
    out << "]";
    return out;
}

void LoadBalancedPolicy::Migrate(MigrateMap& toMigrate,
				 GObject *locChange) {
    if (locChange != NULL) return;

    TimeVal now;
    OS::GetCurrentTime(&now);

    CostMetric load      = m_Manager->GetLoad(now);

#ifndef NO_CLUSTERING
    cluster(toMigrate, load);
#endif

    /*
      debug("after clustering heuristic:");
      DEBUG_RUN(cerr << toMigrate << endl);
    */

#ifndef NO_LOADSHEDDING
    offload(toMigrate, load);
#endif
}

struct link_weight_less {

    /**
     * Returns true a > b. (sort in descending order -- want guys that have
     * more interests in this guy)
     */
    bool operator()(const pair<SID, uint32>& a, const pair<SID, uint32>& b) const {
	return a.second > b.second ||
	    // break ties by choosing with a total order on the SID
	    // space. This will help nudge guys all on different machines
	    // but in the same cluster to the same machine
	    a.second == b.second && less_SID()(a.first, b.first);
    }
};

void LoadBalancedPolicy::cluster(MigrateMap& toMigrate,
				 CostMetric  load)
{
    TimeVal now;
    OS::GetCurrentTime(&now);

    START(LoadMigrate::Cluster);

    ObjectStore *store   = m_Manager->GetObjectStore();
    GObjectInfoMap *imap = m_Manager->GetObjectInfo();
    NodeLoadMap *lmap    = m_Manager->GetNodeLoadMap();

    /*
      debug("predicted load in=%g out=%g : store_size=%d", load_in, load_out,
      store.size());
      debug("params: low_in=%g low_out=%g high_in=%g high_out=%g",
      server->params->lowwater_in, 
      server->params->lowwater_out, 
      server->params->highwater_in, 
      server->params->highwater_out);
    */

    // For each primary object, find out how much CostMetric is consumed
    // by it by inbound (delta_rate of interests) and 
    // our outbound (our delta_rate to node (if interested))

    store->Begin();
    GObject *prim_obj;
    while ( (prim_obj = store->Next()) != NULL) {
	if ( prim_obj->IsReplica() ) continue;

	if ( !prim_obj->IsMigratable() ) continue;
	if ( prim_obj->ExpectedLifetime() != 0 &&
	     prim_obj->ExpectedLifetime() < MIGRATION_CONSIDER ) {
	    // hystericy -- this guy isn't expected to live much longer
	    // so there is no point to migrate him really.
	    continue;
	}

	DB(1) << "considering object " << prim_obj->GetGUID() << endl;

	GObjectInfo *obj = (*imap)[prim_obj->GetGUID()];
	ASSERT(obj != NULL);

	// map from server => num_links_to_me (sym. links_from_me_to_server)
	map<SID, uint32, less_SID> links;

	InterestLinkMap *interests = obj->GetInterests();
	for (InterestLinkMapIter i = interests->begin(); 
	     i != interests->end(); i++) {
	    InterestLink *link = i->second;

	    // here we assume that links are symmetric, so that if I have
	    // a link to someone on that server, they have a link to me
	    if ( link->IsStable() ) {
		GObjectInfo *tgt = link->GetTarget();
		GObject     *tgt_obj = store->Find(tgt->GetGUID());

		DBG << "stable link: " << link << endl;

		if (tgt_obj == NULL) {
		    DB(1) << "How do we have a stable link to an obj that "
			  << "we do not have in our object store?" << endl;
		    continue;
		}

		SID target_server = tgt->GetSID();
		if ( toMigrate.find(tgt->GetGUID()) != toMigrate.end() ) {
		    // we decided to migrate this last guy who we are
		    // interested in! So for now consider that it is on
		    // the other server already
		    target_server = toMigrate[tgt->GetGUID()];
		}

		map<SID, uint32, less_SID>::iterator p = 
		    links.find( target_server );
		if (p == links.end()) {
		    links.insert(pair<SID,uint32>(target_server, 1));
		} else {
		    p->second++;
		}
	    }
	}

	// sort the link map to find out who has the most links to me
	vector< pair<SID, uint32> > links_vec(links.size());
	for (map<SID, uint32, less_SID>::iterator it = links.begin();
	     it != links.end(); it++) {
	    links_vec.push_back( *it );
	}

	stable_sort(links_vec.begin(), links_vec.end(), link_weight_less());

	// now consider each guy in order until we get to me
	for (uint32 i=0; i<links_vec.size(); i++) {
	    SID target = links_vec[i].first;

	    if (target == m_Manager->GetSID())
		break;

	    CostMetric predict = calcCost(obj, target, store, toMigrate);

	    // first check the load on the target server
	    NodeLoadMapIter p = lmap->find(target);
	    // be conservative and don't move if we have no info
	    if (p == lmap->end()) {
		continue;
	    }

	    if ( p->second->toHighWaterMark - predict >= 0 ) {

		ASSERT( ! prim_obj->IsReplica() );
		toMigrate[ obj->GetGUID() ] = target;
		// conservatively update the target's stats so we don't
		// accidently migrate more stuff to this server if we predict
		// it can not handle it
		p->second->toHighWaterMark -= predict;
		p->second->toLowWaterMark  -= predict;
		p->second->Refresh(p->second->GetTTL(), 
				   now); // XXX should we be touching this?

		// recompute our expected load after moving this object
		CostMetric savings = calcSavings(obj, store, toMigrate);
		load -= savings;

		break;
	    } else {
		DB(1) << "unable to migrate to argmax predict=" << predict
		      << " toHW=" << p->second->toHighWaterMark << endl;
	    }
	}

	/*
	  hash_map<SID, CostMetric, hash_SID, equal_SID> links;
	  CostMetric max   = -1;
	  CostMetric total = 0;
	  SID argmax = SID_NONE;

	  InterestLinkMap *interests = obj->GetInterests();

	  for (InterestLinkMapIter i = interests->begin(); 
	  i != interests->end(); i++) {
	  InterestLink *link = i->second;

	  // examine only "stable" links
	  if ( link->IsStable() ) {
	  GObjectInfo *tgt = link->GetTarget();
	  GObject     *tgt_obj = store->Find(tgt->GetGUID());

	  DBG << "stable link: " << link << endl;

	  if (tgt_obj == NULL) {
	  DB(1) << "How do we have a stable link to an obj that "
	  << "we do not have in our object store?" << endl;
	  continue;
	  }

	  hash_map<SID, CostMetric, hash_SID, equal_SID>::iterator p = 
	  links.find( tgt->GetSID() );
	  if (p == links.end()) {
	  links[tgt->GetSID()] = tgt_obj->GetDeltaCost();
	  p = links.find( tgt->GetSID() );
	  } else {
	  p->second += tgt_obj->GetDeltaCost();
	  }
	  if ( (p->second > max  ||
	  // break ties using total order on SID -- this will
	  // "nudge" cliques that have nodes on all different
	  // servers onto the same one server.
	  && less_SID()(p->first, argmax) )
	  &&
	  // can't use this guy if we don't know its load info
	  (p->first == m_Manager->GetSID() ||
	  lmap->find(p->first) != lmap->end()) ) {
	  max = p->second;
	  argmax = p->first;
	  }
	  total += tgt_obj->GetDeltaCost();
	  }
	  }

	  SIDSet seenBackPtrs;
	  InterestLinkSet *backptrs  = obj->GetBackPtrs();

	  for (InterestLinkSetIter i = backptrs->begin();
	  i != backptrs->end(); i++) {
	  InterestLink *link = *i;

	  // examine only "stable" links
	  if ( link->IsStable() ) {
	  GObjectInfo *src = link->GetSource();

	  // already recorded this dude
	  if (seenBackPtrs.find(src->GetSID()) != seenBackPtrs.end()) {
	  continue;
	  }
	  seenBackPtrs.insert(src->GetSID());

	  hash_map<SID, CostMetric, hash_SID, equal_SID>::iterator p = 
	  links.find( src->GetSID() );
	  if (p == links.end()) {
	  links[src->GetSID()] = prim_obj->GetDeltaCost();
	  p = links.find( src->GetSID() );
	  } else {
	  p->second += prim_obj->GetDeltaCost();
	  }
	  if ( (p->second > max || p->second == max 
	  // break ties using total order on SID -- this will
	  // "nudge" cliques that have nodes on all different
	  // servers onto the same one server.
	  && less_SID()(p->first, argmax) )
	  &&
	  // can't use this guy if we don't know its load info
	  (p->first == m_Manager->GetSID() ||
	  lmap->find(p->first) != lmap->end()) ) {
	  max = p->second;
	  argmax = p->first;
	  }
	  total += prim_obj->GetDeltaCost();
	  }
	  }

	  // this object has no stable interests
	  if (argmax == SID_NONE) {
	  DB(1) << "object has no stable interests" << endl;
	  continue;
	  }

	  DB(1) << "argmax was " << argmax << " max=" << max << " total="
	  << total << endl;

	  // more interests of this guy are on argmax -- migrate him!
	  if ( argmax != m_Manager->GetSID() ) {
	  // first check the load on the target server
	  NodeLoadMapIter p = lmap->find(argmax);
	  // we ensured this didn't occur above...
	  ASSERT(p != lmap->end());

	  // TODO: this is a conservative estimate (i.e., the target
	  // is probably already receiving pubs for these replicas)
	  // client_in + cost[all replicas not on target]
	  CostMetric predict  = prim_obj->GetFixedCost() + total - max;

	  // TODO: fudge_factor?
	  if (p->second->toHighWaterMark - predict >= 0 ||
	  // Aggressively force a migration if we are above capacity
	  // ignoring what the other guy is at (this is to try to
	  // get out of "really bad" situations. Not like we can make it
	  // much worse (this way we at least reduce global load)
	  load > m_Manager->GetCapacity()) {

	  ASSERT( ! prim_obj->IsReplica() );
	  toMigrate[ obj->GetGUID() ] = argmax;
	  // conservatively update the target's stats so we don't
	  // accidently migrate more stuff to this server if we predict
	  // it can not handle it
	  p->second->toHighWaterMark -= predict;
	  p->second->toLowWaterMark  -= predict;
	  p->second->Refresh(p->second->GetTTL(), 
	  now); // XXX should we be touching this?

	  // recompute our expected load after moving this object
	  CostMetric savings = total;
	  hash_map<SID, CostMetric, hash_SID, equal_SID>::iterator p = 
	  links.find(m_Manager->GetSID());
	  if (p != links.end()) 
	  // the amount saved is less when we it the migration will
	  // create replica links that weren't there before
	  savings -= p->second;
	  load -= savings;
	  } else {
	  DB(1) << "unable to migrate to argmax predict=" << predict
	  << " toHW=" << p->second->toHighWaterMark << endl;
	  }
	  }
	*/
    }

    STOP(LoadMigrate::Cluster);
}

void LoadBalancedPolicy::offload(MigrateMap& toMigrate, 
				 CostMetric& load) {

    TimeVal now;
    OS::GetCurrentTime(&now);

    START(LoadMigrate::Offload);

    ObjectStore *store   = m_Manager->GetObjectStore();
    GObjectInfoMap *imap = m_Manager->GetObjectInfo();
    NodeLoadMap *lmap    = m_Manager->GetNodeLoadMap();

    bool over  = load > m_Manager->GetHighWaterMark();


    DB(1) << "highwater=" << m_Manager->GetHighWaterMark()
	  << " lowwater=" << m_Manager->GetLowWaterMark() << endl;
    if (!over) {
	DB(-5) << "not over high water: load=" << load 
	       << " target=" << m_Manager->GetLoadTarget() << endl;
	STOP(LoadMigrate::Offload);
	return;
    } else {
	DB(-5) << "over high water! load=" << load 
	       << " highwater=" << m_Manager->GetHighWaterMark()
	       << " load-target=" << (load-m_Manager->GetLoadTarget()) << endl;
    }

    // find candidates we may offload to...
    SIDVec topick;
    for (NodeLoadMapIter it = lmap->begin(); it != lmap->end(); it++) {
	if (it->second->toLowWaterMark > 0 && it->first != m_Manager->GetSID())
	    topick.push_back(it->first);
    }

    // can't do anything if there is no one to offload to
    if (topick.size() == 0) {
	DB(-5) << "no one to migrate to! loadmap.size="
	       << lmap->size() << endl;
	STOP(LoadMigrate::Offload);
	return;
    }

    SIDSet candidates;
    // permute total set
    for (int i=0; i<(int)topick.size(); i++) {
	int index = (int)(drand48()*topick.size());
	SID temp  = topick[i];
	topick[i] = topick[index];
	topick[index] = temp;
    }
    //DBG << "total=" << topick.size() << " randfrac=" << topick.size()*LOAD_SHEDDING_CANDIDATE_FRAC << endl;
    // pick a random FRAC of them to consider
    for (uint32 i=0; i<MIN(LOAD_SHEDDING_CANDIDATES_MAX,topick.size()); i++) {
	candidates.insert(topick[i]);
    }

    // Find connected components
    Component visited;
    Components cc;

    START(LoadMigrate::Offload::DFS);

    // DFS on the primaries to find the connected components
    store->Begin();
    GObject *prim_obj;
    while ( (prim_obj = store->Next()) != NULL) {
	if ( prim_obj->IsReplica() ) continue;
	if ( prim_obj->ExpectedLifetime() != 0 &&
	     prim_obj->ExpectedLifetime() < MIGRATION_CONSIDER )
	    continue;

	GObjectInfo *obj = (*imap)[prim_obj->GetGUID()];

	ASSERT(obj != NULL);

	if (visited.find(obj) != visited.end())
	    continue;
	Component c;
	cc.push_back(c);
	markLocal(visited, cc[cc.size()-1], obj, store);
    }

    STOP(LoadMigrate::Offload::DFS);

    DBG_DO {
	DBG << "Components:" << endl;
	for (ComponentsIter i = cc.begin(); i != cc.end() ; i++) {
	    Component c = *i;
	    DBG << " " << c << endl;
	}
	DBG << endl;
    }

    Savings save;
    save.reserve(cc.size());
    LoadQueue loadvec;
    calcSavings(cc, save, store);
    calcLoadVec(candidates, lmap, loadvec);

    DBG_DO {
	DBG << "Savings:" << endl;
	for (SavingsIter i = save.begin() ; i != save.end(); i++) {
	    int index = i->orig_index;
	    Component c = cc[index];
	    DBG << " [" << i->orig_index
		<< "," << i->savings << ","
		<< c << "]" << endl;
	}
	DBG << endl;
    }

    START(LoadMigrate::Offload::Savings);

    int iter = 0;
    for (SavingsIter i = save.begin() ; i != save.end(); i++) {
	iter++;

	// Considering smallest components first...
	int index = i->orig_index;
	Component c = cc[index];

	bool isMigratable = true;
	for (ComponentIter it = c.begin(); it != c.end(); it++) {
	    GObject *obj = store->Find((*it)->GetGUID());
	    ASSERT(obj != NULL);
	    if (! obj->IsMigratable() ) {
		isMigratable = false;
		break;
	    }
	}
	if (!isMigratable) {
	    DBG << "Cluster is not migratable, skipping: " << c << endl;
	    continue;
	}

	DB(-5) << "Trying to offload: " << c << endl;

	while (loadvec.size() > 0) {
	    NodeLoadInfo *consider_next = loadvec.top();
	    loadvec.pop();

	    ASSERT ( consider_next->sid != m_Manager->GetSID() );
	    CCPair cost; // -savings = costs
	    calcCost(&cost, consider_next->sid, c, store);

	    // multiply by fudge factor to overestimate (account for
	    // variance) so to avoid oscillation.
	    cost.savings *= LOAD_OFFLOAD_FACTOR;

	    if (-cost.savings <= consider_next->toHighWaterMark) {
		// found a server that can take these objects!
		for (ComponentIter k=c.begin(); 
		     k != c.end(); k++) {
		    toMigrate[(*k)->GetGUID()] = consider_next->sid;
		}
		// multiply by fudge factor to overestimate (account for
		// variance) so to avoid oscillation.
		load -= LOAD_OFFLOAD_FACTOR * i->savings;
		consider_next->toHighWaterMark += cost.savings; // negative!
		consider_next->toLowWaterMark  += cost.savings; // negative!

		DB(-5) << "Offloading " << c << " to " << consider_next->sid << endl;
		DB(-5) << "Cost: " << consider_next->sid << " " << -cost.savings << endl;
		DB(-5) << "Save: " << LOAD_OFFLOAD_FACTOR * i->savings << endl;

		// if this guy still has space, keep putting stuff on it
		// until it reaches about the load target (just passes it)
		//CostMetric toTarget = (consider_next->toHighWaterMark - consider_next->toLowWaterMark)/2;
		if (consider_next->toLowWaterMark > 0)
		    loadvec.push(consider_next);
		break;
	    } else {
		DBG << "Cost " << -cost.savings << " > ToHighWater " << consider_next->toHighWaterMark << endl;
	    }
	}

	// keep offloading until we are below the high water mark
	// (to avoid oscillation)
	if (load < m_Manager->GetHighWaterMark() || loadvec.size() == 0) {
	    DB(-5) << "Now below high water mark! Yay!" << endl;
	    break;
	}
    }

    STOP(LoadMigrate::Offload::Savings);

    // Set the load in the load manager so that we converge to the new load
    // immediately. Otherwise the next time we come around to consider
    // migration, we may still believe we are over the high water mark
    m_Manager->SetLoad( MAX(0, load) );

    STOP(LoadMigrate::Offload);
}

bool LoadBalancedPolicy::localInterest(GObjectInfo *obj,
				       MigrateMap& toMigrate)
{
    bool yes = true;
    InterestLinkSet *backptrs  = obj->GetBackPtrs();

    for (InterestLinkSetIter i = backptrs->begin();
	 i != backptrs->end(); i++) {
	InterestLink *link = *i;

	// examine only "stable" links
	if ( link->IsStable() ) {
	    GObjectInfo *src = link->GetSource();

	    if ( src->GetSID() == m_Manager->GetSID() &&
		 // if we are migrating the target, we will get savings
		 // if we move both this guy and the target
		 toMigrate.find(src->GetGUID()) == toMigrate.end() ) {
		yes = false;
		break;
	    }
	}
    }
    return yes;
}

CostMetric LoadBalancedPolicy::calcSavings(GObjectInfo *obj,
					   ObjectStore *store,
					   MigrateMap& toMigrate)
{
    // we get savings from moving this guy if we have no one else
    // interested in it on this server (this is an underestimate
    // because we perform migration iteratively, so guys we moved
    // earlier don't have savings computed if guys we moved later
    // are still on it; that is ok)

    GObject *gobj = store->Find(obj->GetGUID());
    ASSERT(gobj);

    if (localInterest(obj, toMigrate))
	// if no one else is interested in me on this node
	// we save the delta cost as well as the client cost
	return gobj->GetDeltaCost() + gobj->GetFixedCost();
    else
	// we always save atleat the client cost
	return gobj->GetFixedCost();
}

CostMetric LoadBalancedPolicy::calcCost(GObjectInfo *obj,
					SID& target,
					ObjectStore *store,
					MigrateMap& toMigrate)
{
    // the cost of moving obj to server target is:
    //
    // |{s|s interested in o} - {target}| * delta_cost + fixed_cost

    uint32 nservers = 0;

    if (localInterest(obj, toMigrate))
	nservers++;

    for (ReplicaPtrMapIter rit = obj->GetRegisteredReplicas()->begin();
	 rit != obj->GetRegisteredReplicas()->end(); rit++) {
	if ( rit->first != target && 
	     rit->second.GetNumInterests() > 0 ) {
	    nservers++;
	}
    }

    GObject *gobj = store->Find(obj->GetGUID());
    ASSERT(gobj);

    return nservers * gobj->GetDeltaCost() + gobj->GetFixedCost();
}

void LoadBalancedPolicy::markLocal(Component& visited,
				   Component& component,
				   GObjectInfo *obj,
				   ObjectStore *store) {
    if (visited.find(obj) != visited.end())
	return;
    visited.insert(obj);

    GObject *real_obj = store->Find(obj->GetGUID());

    // hystericy -- don't include objects that are about to
    // expire "die" anyway since moving them won't save us anything
    if ( real_obj && real_obj->ExpectedLifetime() != 0 &&
	 real_obj->ExpectedLifetime() < MIGRATION_CONSIDER )
	return;

    if ( real_obj != NULL && !real_obj->IsReplica() ) {
	component.insert(obj);

	// crawl foward on links
	for (InterestLinkMapIter i=obj->GetInterests()->begin(); 
	     i != obj->GetInterests()->end(); i++) {
	    if ( i->second->IsStable() ) {
		GObjectInfo *next = i->second->GetTarget();
		markLocal(visited, component, next, store);
	    }
	}

	// crawl backward on links
	for (InterestLinkSetIter i=obj->GetBackPtrs()->begin(); 
	     i != obj->GetBackPtrs()->end(); i++) {
	    if ( (*i)->IsStable() ) {
		GObjectInfo *next = (*i)->GetSource();
		markLocal(visited, component, next, store);
	    }
	}
    } else {
	// don't add replicas to the component, but crawl them

	// crawl backward on links
	for (InterestLinkSetIter i=obj->GetBackPtrs()->begin();
	     i != obj->GetBackPtrs()->end(); i++) {
	    if ( (*i)->IsStable() ) {
		GObjectInfo *next = (*i)->GetSource();
		markLocal(visited, component, next, store);
	    }
	}
    }
}

void LoadBalancedPolicy::calcSavings(Components& cc, 
				     Savings& save,
				     ObjectStore *store) {

    int i = 0;
    for (ComponentsIter iter=cc.begin(); 
	 iter != cc.end(); iter++, i++) {
	CCPair p;
	p.orig_index  = i;
	p.savings  = 0;
	calcSavings(&p, *iter, store);
	ASSERT(p.orig_index < (uint32)cc.size());
	save.push_back(p);
    }
    /*
      if (DEBUG) {
      cerr << "InSavings:";
      for (SavingsIter i = save.begin() ; i != save.end(); i++) {
      int index = i->orig_index;
      Component c = cc[index];
      cerr << " [" << i->orig_index <<
      "," << i->in_savings << "," << i->out_savings << "," <<
      "size:" << c.size() << "]";
      }
      cerr << endl;
      }
    */

    // TODO: figure out why "sort" is broken with this code...
    // stable_sort seems to work fine
    stable_sort(save.begin(), save.end(), ascending_All());
    // sort with smallest components first; this is because smaller
    // components will be more likely to "fit" on remote nodes and
    // less likely to cause oscillations than big components.
}

void LoadBalancedPolicy::calcSavings(CCPair *p, 
				     Component& c,
				     ObjectStore *store) {
    /*Component seen(c.begin(), c.end());*/

    for (ComponentIter i = c.begin(); i != c.end(); i++) {
	GObjectInfo *obj = *i;
	GObject *real_obj = store->Find(obj->GetGUID());

	ASSERT( real_obj && !real_obj->IsReplica() );
	// each object we offload saves us:
	// the client connection costs
	p->savings  += real_obj->GetFixedCost();

	/* XXX -- ignore these, we're only optimizing for outbound
	   cost, so we don't care about inbound. The intuition is that
	   since interests are likely mutual, savings at one end will
	   be savings at the other

	   // for each replica that guy is attached to, we save
	   // its outgoing deltas (NOTE: this is only true
	   // if *all* the objects interested in the replica are
	   // in our component c. We assume this by the way the
	   // component was constructed.)
	   for (InterestLinkMapIter j = obj->GetInterests()->begin();
	   j != obj->GetInterests()->end(); j++) {
	   InterestLink *link = j->second;
	   GObjectInfo *next = link->GetTarget();
	   if ( !link->IsStable() ) 
	   continue;

	   if (seen.find(next) != seen.end())
	   continue;
	   seen.insert(next);

	   GObject *next_obj = store->Find(next->GetGUID());
	   if (next_obj == NULL)
	   continue;
	   ASSERT(next_obj->IsReplica());

	   p->savings += next_obj->GetDeltaCost();
	   }
	*/

	// for each node that has a stable interested in us, we save
	// the outgoing deltas for this object
	int num_stable_reps = 0;
	for (ReplicaPtrMapIter rit = obj->GetRegisteredReplicas()->begin();
	     rit != obj->GetRegisteredReplicas()->end(); rit++) {
	    // only save the costs if the remote replica is stable
	    // TODO: should we weight them more if they have more stable
	    // interests?
	    if ( rit->second.GetNumInterests() > 0 ) {
		num_stable_reps++;
	    }
	}
	p->savings += num_stable_reps * real_obj->GetDeltaCost();
    }
}

void LoadBalancedPolicy::calcCost(CCPair *p, SID& target, 
				  Component& c,
				  ObjectStore *store) {
    /*Component seen(c.begin(), c.end());*/

    for (ComponentIter i = c.begin(); i != c.end(); i++) {
	GObjectInfo *obj = *i;
	GObject *real_obj = store->Find(obj->GetGUID());

	ASSERT( real_obj && !real_obj->IsReplica() );

	// each object we take on costs us:
	// the client connection costs
	p->savings  -= real_obj->GetFixedCost();

	/* XXX -- ignore these, we're only optimizing for outbound
	   cost, so we don't care about inbound. The intuition is that
	   since interests are likely mutual, savings at one end will
	   be savings at the other

	   // for each replica that guy is attached to, we incur
	   // the costs of its incoming publications from other
	   // servers (but we also *save* the incoming costs of
	   // the primary's pubs)
	   for (InterestLinkMapIter j = obj->GetInterests()->begin();
	   j != obj->GetInterests()->end(); j++) {
	   InterestLink *link = j->second;
	   GObjectInfo *next = link->GetTarget();
	   if ( !link->IsStable() ) 
	   continue;
	   if (seen.find(next) != seen.end())
	   continue;
	   seen.insert(next);

	   GObject *next_obj = store->Find(next->GetGUID());
	   if (next_obj == NULL)
	   continue;
	   ASSERT(next_obj->IsReplica());

	   if (next->GetSID() != target) {
	   p->savings -= next_obj->GetDeltaCost();
	   } else {
	   p->savings += next_obj->GetDeltaCost();
	   }
	   }
	*/

	// for each other node that is interested in it, we incur
	// costs of the outgoing deltas for this object
	int num_stable_reps = 0;
	for (ReplicaPtrMapIter rit = obj->GetRegisteredReplicas()->begin();
	     rit != obj->GetRegisteredReplicas()->end(); rit++) {
	    // only save the costs if the remote replica is stable
	    // TODO: should we weight them more if they have more stable
	    // interests?
	    if ( rit->first != target && 
		 rit->second.GetNumInterests() > 0 ) {
		num_stable_reps++;
	    }
	}
	p->savings -= num_stable_reps * real_obj->GetDeltaCost();
	// never have to add in the current node because after
	// offloading the connected component, it should definitely
	// not be interested in the object
    }
}

void LoadBalancedPolicy::calcLoadVec(SIDSet& candidates,
				     NodeLoadMap *lmap,
				     LoadQueue& load) {
    for (SIDSetIter i = candidates.begin(); 
	 i != candidates.end(); i++) {
	NodeLoadMapIter p = lmap->find(*i);
	ASSERT( p != lmap->end() );
	load.push( p->second );
    }
}

// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
