////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************
  ManagerParams.cpp

begin           : Nov 6, 2002
copyright       : (C) 2002-2003 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2002-2003 Justin Weisz       ( jweisz@cs.cmu.edu )

***************************************************************************/

#include <mercury/NetworkLayer.h>
#include <om/ManagerParams.h>

    namespace ManagerParams {

	///////////////////////////////////////////////////////////////////////////
	///// SOFT-STATE TTLS

	uint32 REPLICA_MAINTAIN_TTL = 2000;
	uint32 INTEREST_LINK_TTL    = 500;
	uint32 REPLICA_INTEREST_TTL = 500;

	uint32 DELETED_GUID_TTL     = 60000;
	uint32 FORWARDING_PTR_TTL   = 60000;
	uint32 NODE_LOAD_INFO_TTL   = 5000;

	uint32 PUBSUB_TTL_FUDGE     = 500;
	uint32 PUBSUB_TTL_EXTRA     = 150;

	///////////////////////////////////////////////////////////////////////////
	///// TIMERS

	// faster than frame rate => always
	uint32 PUBLICATION_INTERVAL          = 50;
	uint32 REPLICA_MAINTAIN_HB           = 500; // =< REPLICA_MAINTAIN_TTL/4

	uint32 MIGRATION_RESP_RETRANSMIT     = 500;   
	uint32 BROADCAST_LOAD_INFO_TIMER     = 2000;
	uint32 PUBLISH_LOAD_INFO_TIMER       = 5000;
	uint32 SEARCH_LOAD_INFO_TIMER        = 10000;
	uint32 LOAD_ESTIMATION_TIMER         = 250;
	uint32 INTEREST_STABILIZATION_TIME   = 50;
	uint32 MIGRATION_CONSIDER            = 5000;
	uint32 SOFTSTATE_EXPIRE_GRAINULARITY = 2;

	///////////////////////////////////////////////////////////////////////////
	///// LOAD PREDICTION INFO

	CostMetric LOW_WATER_MARK             = 180*1024;
	CostMetric HIGH_WATER_MARK            = 280*1024;
	CostMetric CAPACITY                   = 500*1024;
	float LOAD_WINDOW_ALPHA               = 0.75;
	CostMetric LOAD_MIN_WINDOW            = 1024; // ~8Kbps XXX FIXME?
	//CostMetric LOAD_MAX_HIGH_WATER_MARK = 1000*1024;
	float LOAD_EMA_WEIGHT                 = 0.8;
	float LOAD_OFFLOAD_FACTOR             = 2.0;

	uint32 LOAD_MIN_AGGREGATION_SAMPLES   = 5; // XXX TODO?

	char LOAD_INFO_HUB[255];

	uint32 BROADCAST_LOAD_INFO_PORT = 80000;
	char   BROADCAST_LOAD_INFO_OTHERS[4096];

	///////////////////////////////////////////////////////////////////////////
	///// FLAGS

	bool PUBLISH_NEW_DELTAS    = false;
	bool PROACTIVE_REPLICATION = false;
	bool WAIT_ALL_JOIN         = false;
	bool ENABLE_LOAD_INFO      = false;
	bool BROADCAST_LOAD_INFO   = false;
	bool FREQUENT_MAINTENANCE  = false;
	bool CHECK_INVARIANTS      = false;

	///////////////////////////////////////////////////////////////////////////
	///// OTHER

	char MERC_ATTR_BOUNDS_STR[255];
	char PLACEMENT_POLICY[255];
	char REGION_MAP_STR[4096];
	char TERMINAL_PASSWORD[255];
	char CONFIG_OPTS[2048];
	char CONFIG_FILE[255];

	hash_map<string, Constraint, hash_string> MERC_ATTR_BOUNDS;
	hash_map<uint32, SID> REGION_MAP;
    };

using namespace ManagerParams;

// This method is called from om/Manager.cpp 
// Check if it is commented there or not.
void ScaleManagerParameters (float factor)
{
    ASSERT (factor > 1.0);

    ///////////////////////////////////////////////////////////////////////////
    ///// SOFT-STATE TTLS
#undef scale_by_factor
#define scale_by_factor(var)   var = (uint32) (var * factor)
#define scale_down(var) var = (float)(var / factor)

    scale_by_factor (REPLICA_MAINTAIN_TTL);  
    scale_by_factor (INTEREST_LINK_TTL);     
    scale_by_factor (REPLICA_INTEREST_TTL);  

    scale_by_factor (DELETED_GUID_TTL);      
    scale_by_factor (FORWARDING_PTR_TTL);    
    scale_by_factor (NODE_LOAD_INFO_TTL);    

    scale_by_factor (PUBSUB_TTL_FUDGE);      
    scale_by_factor (PUBSUB_TTL_EXTRA);      

    ///////////////////////////////////////////////////////////////////////////
    ///// TIMERS

    // faster than frame rate => always
    scale_by_factor (PUBLICATION_INTERVAL);           
    scale_by_factor (REPLICA_MAINTAIN_HB);

    scale_by_factor (MIGRATION_RESP_RETRANSMIT);      
    scale_by_factor (BROADCAST_LOAD_INFO_TIMER);      
    scale_by_factor (PUBLISH_LOAD_INFO_TIMER);        
    scale_by_factor (SEARCH_LOAD_INFO_TIMER);         
    scale_by_factor (LOAD_ESTIMATION_TIMER);          
    scale_by_factor (INTEREST_STABILIZATION_TIME);    
    scale_by_factor (MIGRATION_CONSIDER);             

    // bandwidth constants have to be scaled down by the time slowdown
    scale_down( LOW_WATER_MARK );
    scale_down( HIGH_WATER_MARK );
    scale_down( CAPACITY );
    scale_down( LOAD_MIN_WINDOW );

#undef scale_by_factor
}

///////////////////////////////////////////////////////////////////////////

#include <util/Options.h>

typedef struct {
    int flags;
    void *varptr;
    char *name;
} Param;

#define P(flags, var) { flags, &(var), # var }

static Param params[] = {
    P(OPT_INT, REPLICA_MAINTAIN_TTL),
    P(OPT_INT, INTEREST_LINK_TTL),
    P(OPT_INT, REPLICA_INTEREST_TTL),
    P(OPT_INT, DELETED_GUID_TTL),
    P(OPT_INT, FORWARDING_PTR_TTL),
    P(OPT_INT, NODE_LOAD_INFO_TTL),
    P(OPT_INT, PUBSUB_TTL_FUDGE),
    P(OPT_INT, PUBSUB_TTL_EXTRA),

    P(OPT_INT, PUBLICATION_INTERVAL),
    P(OPT_INT, REPLICA_MAINTAIN_HB),
    P(OPT_INT, MIGRATION_RESP_RETRANSMIT),
    P(OPT_INT, BROADCAST_LOAD_INFO_TIMER),
    P(OPT_INT, LOAD_ESTIMATION_TIMER),
    P(OPT_INT, PUBLISH_LOAD_INFO_TIMER),
    P(OPT_INT, SEARCH_LOAD_INFO_TIMER),
    P(OPT_INT, INTEREST_STABILIZATION_TIME),
    P(OPT_INT, MIGRATION_CONSIDER),
    P(OPT_INT, SOFTSTATE_EXPIRE_GRAINULARITY),

    P(OPT_FLT, LOW_WATER_MARK),
    P(OPT_FLT, HIGH_WATER_MARK),
    P(OPT_FLT, CAPACITY),
    P(OPT_FLT, LOAD_WINDOW_ALPHA),
    P(OPT_FLT, LOAD_MIN_WINDOW),
    P(OPT_FLT, LOAD_EMA_WEIGHT),

    P(OPT_STR, LOAD_INFO_HUB),
    P(OPT_INT, BROADCAST_LOAD_INFO_PORT),
    P(OPT_STR, BROADCAST_LOAD_INFO_OTHERS),

    P(OPT_BOOL, WAIT_ALL_JOIN),
    P(OPT_BOOL, PUBLISH_NEW_DELTAS),
    P(OPT_BOOL, PROACTIVE_REPLICATION),
    P(OPT_BOOL, ENABLE_LOAD_INFO),
    P(OPT_BOOL, BROADCAST_LOAD_INFO),
    P(OPT_BOOL, FREQUENT_MAINTENANCE),
    P(OPT_BOOL, CHECK_INVARIANTS),

    P(OPT_STR, MERC_ATTR_BOUNDS_STR),
    P(OPT_STR, PLACEMENT_POLICY),
    P(OPT_STR, REGION_MAP),
    P(OPT_STR, TERMINAL_PASSWORD),

    {0, 0, 0}
};

void _AssignValue(Param * ptr, char *value)
{
    if (ptr->flags & OPT_CHR) {
	*(char *) ptr->varptr = *value;
    } else if (ptr->flags & OPT_INT) {
	*(int *) ptr->varptr = atoi(value);
    } else if (ptr->flags & OPT_FLT) {
	*(float *) ptr->varptr = atof(value);
    } else if (ptr->flags & OPT_STR) {
	strcpy((char *) ptr->varptr, value);
    } else if (ptr->flags & OPT_BOOL) {
	bool toSet = (strcmp(value, "1") == 0 ? true : false);
	*((bool *) ptr->varptr) = toSet;
    }
}

void _ParseLine(char *line, int curr_line)
{
    char name[LINEBUF_SIZE], val[LINEBUF_SIZE];

    if (sscanf(line, "%s %s", name, val) != 2) {
	if (line >= 0)
	    Debug::die("#fields on line %d != 2 (%s)\n", curr_line, line);
	else
	    Debug::die("#fields in --omopts != 2 (%s)\n", line);
    }

    bool found = false;
    Param *ptr = params;
    while (ptr->name) {
	if (!strcmp(name, ptr->name)) {
	    //
	    // Process this param now.
	    //
	    _AssignValue(ptr, val);
	    found = true;
	    break;
	} else {
	    //DBG << "'" << name << "' DOESN'T MATCH '" << ptr->name << "'" << endl;
	}
	ptr++;
    }

    if (!found) {
	Debug::die("unknown parameter name %s on line %d", name, curr_line);
    }
}

static void _ParseBounds(char *line, int attr)
{
    char names[LINEBUF_SIZE], mins[LINEBUF_SIZE], maxs[LINEBUF_SIZE];

    if (sscanf(line, "%s %s %s", names, mins, maxs) != 3) {
	Debug::die("bad bounds given to --mercbounds: %s\n", line);
    }

    string name(names);
    Value min(mins);
    Value max(maxs);
    // XXX Check for error?
    /*
      uint64 min   = strtoll(mins, &end, 0);
      bad ||= end != mins;
      uint64 max   = strtoll(maxs, &end, 0);
      bad ||= end != maxs;

      if (bad) {
      Debug::die("bad bounds given to --mercbounds: %s\n", line);
      }
    */

    MERC_ATTR_BOUNDS.insert(pair<string,Constraint>(name,Constraint(attr, min, max)));
}

static void _ParseRegion(char *line)
{
    char id[LINEBUF_SIZE], addr[LINEBUF_SIZE];

    if (sscanf(line, "%s %s", id, addr) != 2) {
	Debug::die("bad entry given to --region-map: %s\n", line);
    }

    uint32 region = atoi(id);
    SID sid = SID(addr);
    if (sid == SID_NONE) {
	Debug::die("bad sid: %s\n", addr);
    }

    REGION_MAP.insert(pair<uint32,SID>(region, sid));
}

void ConfigManagerParams()
{
    if ( strcmp(CONFIG_FILE, "") ) {

	FILE *fp = fopen(CONFIG_FILE, "r");
	if (! fp) {
	    Debug::die("Couldn't open Manager config file: %s", CONFIG_FILE);
	}

	char line[LINEBUF_SIZE];
	int curr_line = 1;

	while (fgets(line, LINEBUF_SIZE, fp)) {
	    if (line[strlen(line) - 1] != '\n') {
		Debug::die("internal error: line %d longer than 255 characters, bailing",
			   curr_line);
	    }
	    if (line[0] == '#' || line[0] == '\n') {
		curr_line++;
		continue;
	    }
	    line[strlen(line) - 1] = '\0';

	    _ParseLine(line, curr_line);
	    curr_line++;
	}
    }

    // options override config file
    char *str = strtok(CONFIG_OPTS, ",;");
    while (str != NULL) {
	_ParseLine(str, -1);
	str = strtok(NULL, ",;");
    }

    // get mercury bounds
    int index = 0;
    str = strtok(MERC_ATTR_BOUNDS_STR, ",;");
    while (str != NULL) {
	_ParseBounds(str, index++);
	str = strtok(NULL, ",;");
    }

    // load region map
    str = strtok(REGION_MAP_STR, ",;");
    while (str != NULL) {
	_ParseRegion(str);
	str = strtok(NULL, ",;");
    }
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
