////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************
  Manager.cpp

  Object Management Interface.

begin           : Nov 6, 2002
copyright       : (C) 2002-2003 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2002-2003 Justin Weisz       ( jweisz@cs.cmu.edu )

***************************************************************************/

#include <sstream>
#include <stdarg.h>
#include <om/Manager.h>
#include <om/GObject.h>
#include <om/OMMessage.h>
#include <om/OMEvent.h>
#include <om/MigrationPolicy.h>
#include <om/ReplicaConnection.h>
#include <om/EventDemux.h>
#include <om/TestLogs.h>
#include <om/OMLogs.h>
#include <util/Benchmark.h>
#include <util/Utils.h>
#include <wan-env/RealNet.h>
#include <wan-env/WANMercuryNode.h>
#include <mercury/RoutingLogs.h>
#include <mercury/ObjectLogs.h>

#define streq(a,b) (!strcmp(a,b))

#define CHECK_ALWAYS_INVARIANTS() \
if (ManagerParams::CHECK_INVARIANTS) { CheckAlwaysInvariants(); }
#define CHECK_ENTRY_INVARIANTS() \
if (ManagerParams::CHECK_INVARIANTS) { CheckEntryInvariants(); }
#define CHECK_EXIT_INVARIANTS() \
if (ManagerParams::CHECK_INVARIANTS) { CheckExitInvariants(); }

///////////////////////////////////////////////////////////////////////////////
///// GLOBALS

sid_t g_LocalSID;

Manager *Manager::m_Instance = NULL;

Manager *Manager::GetInstance(GameAdaptor *game)
{
    ASSERT(m_Instance == NULL);
    m_Instance = new Manager(game);
    return m_Instance;
}

Manager *Manager::GetInstance()
{
    ASSERT(m_Instance != NULL);
    return m_Instance;
}

///////////////////////////////////////////////////////////////////////////////
///// CONSTRUCTOR + DESTRUCTOR

Manager::Manager(GameAdaptor *game, bool startNetwork) : m_Game(game) 
{
    srand48(getpid()^time(NULL));

    ///// Configure Parameters
    ConfigManagerParams();
#if 0
    if (g_Preferences.slowdown_factor > 1.0) 
	ScaleManagerParameters (g_Preferences.slowdown_factor);
#endif

    ///// Init Object Allocation Pools
    m_GObjectInfoPool = new ClassChunker<GObjectInfo>();
    m_InterestLinkPool = new ClassChunker<InterestLink>();
    GObjectInfo::SetLinkAllocator(m_InterestLinkPool);

    m_TimedGUIDPool = new ClassChunker<TimedGUID>();
    m_TimedSIDPool = new ClassChunker<TimedSID>();

    ///// Migration Policy
    if (streq(PLACEMENT_POLICY, "load")) {
	ASSERT(ENABLE_LOAD_INFO);
	m_MigrationPolicy = new LoadBalancedPolicy();
    } else if (streq(PLACEMENT_POLICY, "region")) {
	ASSERT(REGION_MAP.size() > 0);
	m_MigrationPolicy = new RegionPlacementPolicy();
    } else if (streq(PLACEMENT_POLICY, "rand")) {
	m_MigrationPolicy = new RandomPlacementPolicy();
    } else if (streq(PLACEMENT_POLICY, "static")) {
	m_MigrationPolicy = new StaticPlacementPolicy();
    } else {
	WARN << "unknown migration placement policy: " 
	     << PLACEMENT_POLICY << endl;
	ASSERT(0);
    }
    m_MigrationPolicy->SetManager(this);

    ///// Init Local SID
    struct hostent *entry = gethostbyname(g_Preferences.hostname);
    if (entry)
	g_LocalSID.m_IP = ((struct in_addr *) entry->h_addr)->s_addr;
    else
	g_LocalSID.m_IP = inet_addr(g_Preferences.hostname);
    g_LocalSID.m_Port = g_Preferences.port + 5000; 
    // as long as we don't have 5000 different nodes, 
    // we should be alright (even if we run on one machine)!

    m_UniqueGUID.m_IP       = g_LocalSID.GetIP();
    m_UniqueGUID.m_Port     = g_LocalSID.GetPort();
    m_UniqueGUID.m_LocalOID = 1; //rand(); // start at random value

    ///// Init Debugging
    DBG_INIT(&g_LocalSID);

    ///// Init Logging
    if (g_MeasurementParams.enabled) {
	InitRoutingLogs(g_LocalSID);
	InitObjectLogs(g_LocalSID);
	InitOMLogs(g_LocalSID);
	InitOMTestLogs(g_LocalSID);
    }

    ///// Init Network
    if (startNetwork) {

	// this can be disabled for debugging

	DBG << "starting up PubSubRouter..." << endl;

	m_PubSubRouter = WANMercuryNode::GetInstance(g_Preferences.port);
	m_PubSubRouter->FireUp();

	m_PubSubEvents = new EventDemux(m_PubSubRouter);

	// wait for self to join
	while (!m_PubSubRouter->IsJoined()) {
#ifndef ENABLE_REALNET_THREAD
	    // Do periodic Network processing
	    RealNet::DoWork(100);
#endif
	    /// Any timeout value is okay here... - Ashwin [03/20/2005]
	    m_PubSubRouter->DoWork (200);
	}

	// wait for everyone else to join?
	if (WAIT_ALL_JOIN) {
	    INFO << "waiting for everyone to get joined..." << endl;
	    while (!m_PubSubRouter->AllJoined()) {
#ifndef ENABLE_REALNET_THREAD
		// Do periodic Network processing
		RealNet::DoWork(50);
#endif
		m_PubSubRouter->DoWork (200);
	    }
	}
	TINFO << "JOINED!" << endl;

	// obtain cached copy of hub constraints
	// right now we assume these never change
	InitializeHubConstraints();

	TDB (-1) << "PubSubRouter started." << endl;

	TDB (-1) << "starting up DirectNet..." << endl;

	m_DirectRouter = DirectRouter::GetInstance(g_LocalSID, this);

	TDB (-1) << "DirectNet started." << endl;

	// remote terminal
#ifdef ENABLE_TERMINAL
	m_Terminal = new Terminal(g_Preferences.port+10000, this);
	m_Terminal->Start();
#endif

	///// Init the load manager
	if (ENABLE_LOAD_INFO) {
	    m_LoadManager = new OMLoadManager(string(LOAD_INFO_HUB),
					      this, 
					      m_DirectRouter->GetNetwork(),
					      BROADCAST_LOAD_INFO ?
					      OMLOADAGG_BROADCAST :
					      OMLOADAGG_MERCURY);
	} else {
	    m_LoadManager = NULL;
	}
    }

    ///// Init the pubsub manager
    m_PubSubManager = new PubSubManager(this, m_PubSubRouter);

    TDB (-1) << "finished initialization of Manager" << endl;
}

// Dummy Constructor
Manager::Manager()
{
    ///// Configure Parameters
    ConfigManagerParams();

    ///// Init Object Allocation Pools
    m_GObjectInfoPool = new ClassChunker<GObjectInfo>();
    m_InterestLinkPool = new ClassChunker<InterestLink>();
    GObjectInfo::SetLinkAllocator(m_InterestLinkPool);

    m_TimedGUIDPool = new ClassChunker<TimedGUID>();
    m_TimedSIDPool = new ClassChunker<TimedSID>();

    m_LoadManager = NULL;
    m_PubSubManager = NULL;
    m_PubSubEvents = NULL;
#ifdef ENABLE_TERMINAL
    m_Terminal = NULL;
#endif

    ///// Init Debugging
    DBG_INIT(&SID_NONE);
}

Manager::~Manager() {
    m_PubSubRouter->Shutdown();
    delete m_PubSubRouter;

    DirectRouter::DestroyInstance();

    if (ENABLE_LOAD_INFO) {
	delete m_LoadManager;
    }
    delete m_PubSubManager;
    delete m_PubSubEvents;

#ifdef ENABLE_TERMINAL
    delete m_Terminal;
#endif

    delete m_GObjectInfoPool;
    delete m_InterestLinkPool;
    delete m_TimedGUIDPool;
    delete m_TimedSIDPool;
}

void Manager::InitializeHubConstraint(OMAttrInfo& info)
{
    const Value& min = info.cons.GetMin();
    const Value& max = info.cons.GetMax();

    ASSERT(min < max);

    mpf_class range;
    Value range_z(max);
    range_z -= min;
    mpf_set_z(range.get_mpf_t(), &range_z);
    info.range = range;

    ASSERT(mpf_get_d(range.get_mpf_t()) > 0);

    pair<real64, real64> rCons;
    rCons.first = mpz_get_d(&(min));
    rCons.second = mpz_get_d(&(max));
    info.asreals = rCons;

    ASSERT(rCons.second > rCons.first);
}

void Manager::InitializeHubConstraints()
{
    m_AttrNameMap.clear();
    m_AttrInfo.clear();

    if (MERC_ATTR_BOUNDS.size() > 0) {
	// get bounds from commandline
	for (hash_map<string, Constraint, hash_string>::iterator it = MERC_ATTR_BOUNDS.begin(); it != MERC_ATTR_BOUNDS.end(); it++) {
	    int attr = it->second.GetAttrIndex();
	    m_AttrNameMap.insert(pair<string,int>(it->first, attr));
	    OMAttrInfo info;
	    info.cons = it->second;

	    m_AttrInfo.insert(pair<int,OMAttrInfo>(attr, info));
	}
	ASSERT(m_AttrInfo.size() > 0);

	INFO << "Manager read merc attr ranges (command-line) as: " << endl;
    } else {
	// get bounds from merc
	vector< pair<int,string> > names = m_PubSubRouter->GetHubNames();
	vector<Constraint> cons = m_PubSubRouter->GetHubConstraints();
	ASSERT(names.size() == cons.size());
	set<int> seen;
	for (uint32 i=0; i<names.size(); i++) {
	    m_AttrNameMap.insert(pair<string,int>(names[i].second,
						  names[i].first));
	    seen.insert(names[i].first);
	}
	for (uint32 i=0; i<cons.size(); i++) {
	    int attr = cons[i].GetAttrIndex();
	    OMAttrInfo info;
	    info.cons = cons[i];

	    m_AttrInfo.insert(pair<int,OMAttrInfo>(attr, info));
	}
	ASSERT(m_AttrInfo.size() > 0);

	INFO << "Manager read merc attr ranges (mercury-node) as: " << endl;
    }

    for (hash_map<string, int, hash_string>::iterator it = m_AttrNameMap.begin(); it != m_AttrNameMap.end(); it++) {
	const string& name = it->first;
	int attr = it->second;
	const OMAttrInfo& info = m_AttrInfo[it->second];

	INFO << name << " (" << attr << "): " 
	     << info.cons.GetMin() << ","
	     << info.cons.GetMax() << endl;
    }

    for (hash_map<string, int, hash_string>::iterator it = m_AttrNameMap.begin(); it != m_AttrNameMap.end(); it++) {
	hash_map<int, OMAttrInfo>::iterator infop = 
	    m_AttrInfo.find(it->second);
	ASSERT(infop != m_AttrInfo.end());
	InitializeHubConstraint(infop->second);
    }
}

///////////////////////////////////////////////////////////////////////////////
///// OBJECT ALLOCATION

int Manager::GetAttrIndex(const string& attr)
{
    hash_map<string, int, hash_string>::iterator attrp =
	m_AttrNameMap.find(attr);
    ASSERT(attrp != m_AttrNameMap.end());
    return attrp->second;
}

Value Manager::ConvertToValue(int attr, real64 from, real64 min, real64 max)
{
    hash_map<int, OMAttrInfo>::iterator info = m_AttrInfo.find(attr);
    ASSERTDO(info != m_AttrInfo.end(), WARN << "attr=" << attr << endl);
    const mpf_class& range = info->second.range;
    const Constraint& cons = info->second.cons;

    ASSERTDO(from >= min && from <= max, WARN << "attr=" << attr << " from=" << from << " min=" << min << " max=" << max << endl);

    mpf_t from_m, range_m, tmp1, tmp2;

    // res = (from-min)*(merc_max - merc_min)/(max - min) + merc_min

    mpf_init_set_d(from_m, from-min);
    mpf_init_set_d(range_m, max-min);

    mpf_init(tmp1);
    mpf_init(tmp2);

    mpf_mul(tmp1, from_m, range.get_mpf_t());
    mpf_div(tmp2, tmp1, range_m);

    Value res;
    mpz_set_f(&res, tmp2);
    res += cons.GetMin();

    mpf_clear(from_m);
    mpf_clear(range_m);
    mpf_clear(tmp1);
    mpf_clear(tmp2);

    //DB(-5) << "attr=" << attr << " in=" << from << " out=" << res << endl;
    //real64 back = ConvertFromValue(attr, res, min, max);

    ASSERT(res >= cons.GetMin());
    ASSERT(res <= cons.GetMax());

    return res;
}

Value Manager::ConvertToValue(const string& name, real64 from, 
			      real64 min, real64 max)
{
    return ConvertToValue(GetAttrIndex(name), from, min, max);
}

Value Manager::ConvertToValue(const string& name, real64 from)
{
    return ConvertToValue(GetAttrIndex(name), from);
}

Value Manager::ConvertToValue(int attr, real64 from)
{
    hash_map<int, OMAttrInfo>::iterator info = m_AttrInfo.find(attr);
    ASSERT(info != m_AttrInfo.end());
    const pair<real64,real64>& rCons = info->second.asreals;

    return ConvertToValue(attr, from, rCons.first, rCons.second);
}

real64 Manager::ConvertFromValue(int attr, const Value& from, real64 min, real64 max)
{
    hash_map<int, OMAttrInfo>::iterator info = m_AttrInfo.find(attr);
    ASSERTDO(info != m_AttrInfo.end(), WARN << "attr=" << attr);
    const mpf_class& range = info->second.range;
    const Constraint& cons = info->second.cons;

    mpf_t from_r, range_m, tmp1, tmp2;

    // res = (from-merc_min)*(max - min)/(merc_max - merc_min) + min
    Value fromt(from);
    fromt -= cons.GetMin();

    mpf_init(from_r);
    mpf_set_z(from_r, &fromt);

    mpf_init_set_d(range_m, max-min);

    mpf_init(tmp1);
    mpf_init(tmp2);

    mpf_mul(tmp1, from_r, range_m);
    mpf_div(tmp2, tmp1, range.get_mpf_t());

    real64 res = mpf_get_d(tmp2);

    mpf_clear(from_r);
    mpf_clear(range_m);
    mpf_clear(tmp1);
    mpf_clear(tmp2);

    //DB(-5) << "attr=" << attr << " in=" << from << " out=" << res << endl;
    res += min;

    ASSERTDO(res >= min, WARN << "res=" << res << " min=" << min << endl);
    ASSERTDO(res <= max, WARN << "res=" << res << " max=" << max << endl);

    return res;
}

real64 Manager::ConvertFromValue(const string& name, const Value& from,
				 real64 min, real64 max)
{
    return ConvertFromValue(GetAttrIndex(name), from, min, max);
}

real64 Manager::ConvertFromValue(const string& name, const Value& from)
{
    return ConvertFromValue(GetAttrIndex(name), from);
}

real64 Manager::ConvertFromValue(int attr, const Value& from)
{
    hash_map<int, OMAttrInfo>::iterator info = m_AttrInfo.find(attr);
    ASSERT(info != m_AttrInfo.end());
    const pair<real64,real64>& rCons = info->second.asreals;

    return ConvertFromValue(attr, from, rCons.first, rCons.second);
}

OMEvent *Manager::CreateEvent(const GUID& guid) {
    // TODO: more intelligent allocation
    return new OMEvent(guid, GetSID());
}

void Manager::FreeEvent(OMEvent *ev) {
    // TODO: more intelligent deallocation
    delete ev;
}

OMInterest *Manager::CreateInterest() {
    // TODO: more intelligent allocation
    OMInterest *in = new OMInterest(GUID::CreateRandom());
    return in;
}

void Manager::FreeInterest(OMInterest *in) {
    // TODO: more intelligent deallocation
    delete in;
}

GObjectInfo *Manager::AllocGObjectInfo(GUID guid, SID sid, bool isReplica)
{
    // XXX FIXME: right now we assume we are single threaded or that
    // whoever calls this holds the manager lock

    GObjectInfo *info;
    GObjectInfoMapIter p = m_ObjectInfo.find(guid);
    if (p != m_ObjectInfo.end()) {
	info = p->second;

	// we must maintain the invariant that only a single object points
	// to this at a given time, because otherwise we will delete it
	// twice!
	ASSERT(FindObject(guid) == NULL);
	ASSERT(info->GetGUID() == guid);
	// This might have changed because of staleness
	ASSERT(info->GetSID() == sid || info->GetSID() != GetSID());
	// Must be a replica, can't create standalone info for primaries
	ASSERT(info->IsReplica());
	info->SetSID(sid);
	// This can not possibly have changed (it must be explicitly set)
	ASSERT(info->IsReplica() == isReplica);
	// Treat as if new TTL
	info->Refresh();
    } else {
	info = m_GObjectInfoPool->alloc();

	info->SetManager( this );
	info->SetGUID( guid );
	info->SetSID( sid );
	info->SetIsReplica( isReplica );
	m_ObjectInfo[guid] = info;

	if (g_MeasurementParams.enabled && isReplica) {
	    ReplicaLifetimeEntry e;
	    e.guid = guid;
	    e.type = (char)ReplicaLifetimeEntry::OBJINFO;
	    LOG(ReplicaLifetimeLog, e);
	}
    }

    return info;
}

void Manager::FreeGObjectInfo(GObjectInfo *info)
{
    // XXX FIXME: right now we assume we are single threaded or that
    // whoever calls this holds the manager lock

    GUID guid = info->GetGUID();
    GObjectInfoMapIter p = m_ObjectInfo.find(guid);
    ASSERT(p != m_ObjectInfo.end());

    m_ObjectInfo.erase(p);

    // Interest Links are cleaned up by ~GObjectInfo
    m_GObjectInfoPool->free(info);
}

TimedGUID *Manager::AllocTimedGUID(GUID guid, uint32 ttl)
{
    TimedGUID *tguid = m_TimedGUIDPool->alloc();

    tguid->Copy(guid);
    tguid->Refresh(ttl);
    return tguid;
}

void Manager::FreeTimedGUID(TimedGUID *tguid)
{
    m_TimedGUIDPool->free(tguid);
}

TimedSID *Manager::AllocTimedSID(SID sid, uint32 ttl)
{
    TimedSID *tsid = m_TimedSIDPool->alloc();
    tsid->Copy(sid);
    tsid->Refresh(ttl);
    return tsid;
}

void Manager::FreeTimedSID(TimedSID *tsid)
{
    m_TimedSIDPool->free(tsid);
}

MsgAppRegister *Manager::GetCurrRegisterMsg(SID sid)
{
    // aggregate registrations
    RegisterMapIter reg = m_ToRegister.find(sid);
    MsgAppRegister *msg;
    if (reg == m_ToRegister.end()) {
	msg = new MsgAppRegister();
	m_ToRegister[sid] = msg;
    } else {
	msg = reg->second;
    }

    return msg;
}

///////////////////////////////////////////////////////////////////////////////
///// PUBLIC UTILITY FUNCTIONS

guid_t Manager::CreateGUID() {
    m_UniqueGUID.m_LocalOID++;

    return m_UniqueGUID;
}

sid_t Manager::GetSID() {
    return g_LocalSID;
}

//
// call-back when app adds a new object
//
void Manager::AddObject(GObject *obj)
{
    //Lock();

    ASSERT(obj->GetGUID() != GUID_NONE);
    ASSERT(!obj->IsReplica());
    ASSERT(obj->GetSID() == GetSID());

    // Hmm.. guess no need for this :)

    //ASSERT(m_ObjectInfo.find(guid) == m_ObjectInfo.end());

    //GObject *obj = m_Game->GetObjectStore()->Find(guid);
    //ASSERT(obj != NULL);

    // Jeff: auto-called on info-allocation
    // add to interest graph
    //m_ObjectInfo[guid] = obj->GetInfo();

    // inform the manager that we will now be tracking
    // info for this object.

    //Unlock();
}

//
// call-back when app removes an object
//
void Manager::DeleteObject(GObject *obj)
{
    Lock();

    ASSERT(obj != NULL);
    GUID guid = obj->GetGUID();

    DB(5) << " deleting guid " << guid << endl;

    // Jeff: no longer necessary, the callback to free the
    // object info will do the removal for us.
    // find node in interest graph -- delete it
    //GObjectInfoMapIter p = m_ObjectInfo.find(guid);
    //ASSERT(p != m_ObjectInfo.end());
    //m_ObjectInfo.erase(p);

    // Jeff: This is no longer required as the info is deleted in
    // GObject's destructor.
    //
    // Should we inform the replicas here of the deletion?
    // Not really necessary, since the next time the app sends
    // a register req, it will trigger an ack telling it the replica
    // is deleted.

    // This will:
    // (1) delete the node in the interest graph
    // (2) remove all links attached to it
    // (3) clean up the registered replicas data structure
    //FreeGObjectInfo(p->second);

    // Only remember objects we own(ed)
    if (!obj->IsReplica()) {
	// remember that we deleted this object
	ASSERT(m_DeletedGUIDs.find(guid) == m_DeletedGUIDs.end());
	TimedGUID *tguid = AllocTimedGUID(guid, DELETED_GUID_TTL);
	m_DeletedGUIDs[guid] = tguid;
    }

    Unlock();
}

//
// Look for obj in object store, then in pending store
GObject *Manager::FindObject(GUID& guid)
{
    GObject *obj = m_Game->GetObjectStore()->Find(guid);
    if (!obj)
	obj = m_Game->GetPendingStore()->Find(guid);
    return obj;
}

///////////////////////////////////////////////////////////////////////////////
///// PRIMARY HANDLER CALLBACKS

void Manager::Merc(uint32 timeout)
{
    //DB(5) << "calling pubsubrouter ... " << endl;

    float rnetfrac = 0.35;
    m_PubSubRouter->DoWork ((int) ((1 - rnetfrac) * timeout));
#ifndef ENABLE_REALNET_THREAD
    RealNet::DoWork((int) (rnetfrac * timeout));
#endif	
}

typedef multimap<GUID, GUID, less_GUID> GUIDMMap;
// some statistics for detecting hystericies
static uint32 pubCount   = 0;
static uint32 subCount   = 0;
static uint32 matchCount = 0;

void Manager::Send(uint32 timeout)
{
    NOTE(Manager::SEND_TIMEOUT, timeout);
    START(SEND);

    uint64 stop = TIME_STOP_USEC(timeout);

    Lock();

    CHECK_ENTRY_INVARIANTS();

    GObject *obj;

    TimeVal now;
    OS::GetCurrentTime(&now);

    START(SEND::InitLoadInfo);

    // determine if we send out load info this round
    bool transmitLoadInfo = false;
    CostMetric toLowWaterMark = 0, toHighWaterMark = 0, load = 0;
    SIDSet sentDelta;
    if (ENABLE_LOAD_INFO) {
	PERIODIC( BROADCAST_LOAD_INFO_TIMER, now, 
	{ 
	    transmitLoadInfo = true;
	    load = GetLoad(now);
	    DBG << "current load: " << load << endl;
	    toLowWaterMark = 
		MAX(0, GetLowWaterMark() - load);
	    toHighWaterMark =
		MAX(0, GetHighWaterMark() - load);
	} );
    }

    STOP(SEND::InitLoadInfo);

    //
    // (0) See if any of our primary objects want to be proactively replicated
    //

    ObjectStore *store = m_Game->GetObjectStore();

    if (ManagerParams::PROACTIVE_REPLICATION) {

	START(SEND::ProactiveReplicationCalc);

	GUIDMMap proactiveMap;
	GUIDList guid_list;

	store->Begin();
	while ( (obj = store->Next()) != NULL ) {
	    if ( ! obj->IsReplica() ) {
		obj->AttachedTo(&guid_list);
		for (GUIDListIter it = guid_list.begin();
		     it != guid_list.end(); it++) {
		    proactiveMap.insert(pair<GUID,GUID>(*it,obj->GetGUID()));
		}
		guid_list.clear();

		obj->AttachTo(&guid_list);
		for (GUIDListIter it = guid_list.begin();
		     it != guid_list.end(); it++) {
		    proactiveMap.insert(pair<GUID,GUID>(obj->GetGUID(), *it));
		}
		guid_list.clear();
	    }
	}

	STOP(SEND::ProactiveReplicationCalc);

	START(SEND::ProactiveReplicationFill);

	for (OutgoingReplicaConnectionMapIter it = 
		 m_ReplicaConnSenderMap.begin();
	     it != m_ReplicaConnSenderMap.end(); it++) {
	    DeltaMap *deltas = it->second->GetDeltaMap();

	    for (DeltaMapIter dit = deltas->begin(); 
		 dit != deltas->end(); dit++) {
		GUIDMMap::iterator start = 
		    proactiveMap.lower_bound(dit->first);
		GUIDMMap::iterator end =
		    proactiveMap.upper_bound(dit->first);
		for ( ; start != end; start++) {
		    //INFO << it->first << " attaching: " << start->first
		    //	 << " <- " << start->second << endl;
		    it->second->RegisterGUID( start->second, 
					      false, false, true );
		}
	    }
	}

	STOP(SEND::ProactiveReplicationFill);
    }

    //
    // (1) Send deltas to all interested nodes
    //

    START(SEND::SendDeltas);

    for (OutgoingReplicaConnectionMapIter it = m_ReplicaConnSenderMap.begin();
	 it != m_ReplicaConnSenderMap.end(); it++) {
	SID target = it->first;
	START(SEND::CreateUpdate);
	MsgAppUpdate *msg = it->second->CreateUpdate();
	STOP(SEND::CreateUpdate);
	if (msg != NULL) {
	    if ( transmitLoadInfo ) {
		msg->SetLoadInfo(toLowWaterMark, toHighWaterMark, load);
	    } else {
		msg->ClearLoadInfo();
	    }

	    ///// MEASUREMENT
	    if (g_MeasurementParams.enabled) {
		for (GUpdateMapIter it2 = msg->updates.begin();
		     it2 != msg->updates.end(); it2++) {
		    if (it2->second->nonce != 0) {
			ASSERT(it2->second->isNew || it2->second->isDelete);
			DiscoveryLatEntry ent(DiscoveryLatEntry::UPDATE_SEND, 
					      0, it2->second->nonce);
			LOG(DiscoveryLatLog, ent);
		    }
		}
	    }
	    ///// MEASUREMENT

	    DBG << "sending(" << target << "): " << msg << endl;

	    START(SEND::SendUpdate);
	    m_DirectRouter->SendMsg(&target, msg);
	    STOP(SEND::SendUpdate);
	}
    }

    STOP(SEND::SendDeltas);

    START(SEND::PubSubManager::DoWork);
    // clean up stable entries in PubSub data
    m_PubSubManager->Maintenance( TIME_LEFT_MSEC(stop) );
    STOP(SEND::PubSubManager::DoWork);

    START(SEND::FillIntererests);
    // ask appication to register/unregister subscriptions
    InterestList reg, unreg;
    GUIDMap assoc;
    m_Game->FillInterests(&reg, &unreg, &assoc,
			  m_PubSubManager->GetCurrInterests());
    STOP(SEND::FillIntererests);

    START(SEND::RegisterInterests);
    for (InterestListIter it = reg.begin(); it != reg.end(); it++) {
	//NOTE(SUB_COUNT, 1);
	subCount++;
	m_PubSubManager->RegisterInterest(*it);
    }
    for (InterestListIter it = unreg.begin(); it != unreg.end(); it++) {
	//NOTE(UNSUB_COUNT, 1);
	m_PubSubManager->UnregisterInterest(*it);
    }

    m_PubSubManager->ChangeInterestAssoc(&assoc);
    STOP(SEND::RegisterInterests);

    //
    // (2) send pubs (mercury layer should aggregate)
    //

    START(SEND::PubsAndRUpdates);
    // aggregate remote updates
    RemoteUpdateMap rupdates;

    store->Begin();
    while ( (obj = store->Next()) != NULL ) {
	if ( ! obj->IsReplica() ) {

	    //START(SEND::PUB::SendPub);

	    EventList reg;
	    EventList unreg;
	    //START(SEND::PUB::GObject::GetCurrEvents);
	    EventSet *old = m_PubSubManager->GetCurrEvents(obj->GetGUID());
	    //STOP(SEND::PUB::GObject::GetCurrEvents);
	    //START(SEND::PUB::GObject::FillEvents);
	    obj->FillEvents(&reg, &unreg, old);
	    //STOP(SEND::PUB::GObject::FillEvents);

	    //START(SEND::PUB::RegisterEvents);
	    for (EventListIter it = reg.begin(); 
		 it != reg.end(); it++) {
		//NOTE(PUB_COUNT, 1);
		pubCount++;

		OMEvent *ev = *it;

		// If publications aren't stored in the system, we
		// have to republish as often as the game freq
		if (!g_Preferences.enable_pubtriggers) {
		    ev->SetLifeTime( (uint32)(1.5*m_Game->
					      GetSendInterval()) );
		}

		if (ManagerParams::PUBLISH_NEW_DELTAS && obj->IsNew()) {
		    //DBG << "obj " << obj->GetGUID() 
		    //	<< " was new, adding newobj tuple in pub" << endl;

		    // XXX: too many copies here
		    Packet *buf = new Packet(GUpdate::MAX_UPDATE_SIZE);
		    obj->PackUpdate(buf, obj->GetInitDeltaMask());

		    GUpdate *update = new GUpdate(false, obj->GetGUID());
		    update->isNew = true;
		    update->mask  = obj->GetInitDeltaMask();
		    update->delta = buf;
		    ASSERT(update->delta);

		    ev->SetUpdate(update);
		    /*					
						Packet pkt(update.GetLength());
						update.Serialize(&pkt);

						ev->AddTuple(ATTR_NEWOBJ, Value(pkt.m_BufPosition,
						pkt.m_Buffer));
		    */
		}

		//DB(5) << "sending pub for obj=" << obj << ": " 
		//	  << ev << endl;

		ev->SetSeqno(((GObjectInfo *)obj->GetInfo())
			     ->NextEventSeqno());

		m_PubSubManager->RegisterEvent(ev);
	    }
	    //STOP(SEND::PUB::RegisterEvents);

	    //START(SEND::PUB::UnRegisterEvents);
	    for (EventListIter it = unreg.begin(); 
		 it != unreg.end(); it++) {
		//NOTE(UNPUB_COUNT, 1);
		m_PubSubManager->UnregisterEvent(*it);
	    }
	    //STOP(SEND::PUB::UnRegisterEvents);

	    //STOP(SEND::PUB::SendPub);
	}

	//
	// (3) aggregate remote updates
	//

	else {

	    if ( obj->IsDirty() ) {
		START(SEND::SendRUpdate);

		IPEndPoint target = obj->GetSID();
		ASSERT(target != GetSID());

		if (rupdates.find(target) == rupdates.end()) {
		    rupdates[target] = new MsgAppRemoteUpdate();
		}

		Packet *delta = new Packet(GUpdate::MAX_UPDATE_SIZE);
		obj->PackUpdate(delta, obj->GetDeltaMask()); // pack update

		// XXX TODO: remote deletions?

		GUpdate *update = new GUpdate(true, obj->GetGUID());
		update->mask    = obj->GetDeltaMask();
		update->delta   = delta;

		obj->ClearDeltaMask();

		rupdates[target]->updates[obj->GetGUID()] = update;

		STOP(SEND::SendRUpdate);
	    }

	}
    }

    STOP(SEND::PubsAndRUpdates);

    START(SEND::SendRemoteUpdates);

    //
    // (4) Send all aggregated remote updates
    //

    for (RemoteUpdateMapIter it = rupdates.begin(); 
	 it != rupdates.end(); it++) {
	// remote updates should be reliable
	// XXX TODO: however, they do not necessarily need to be ordered,
	// need some way to push this back to the application...
	// XXX TODO: shouldn't use TCP...
	SID target = it->first;
	m_DirectRouter->SendMsg(&target, it->second, 
				// eschew 100% correctness for now to
				// get better performance :)
				PROTO_UDP
				/* PROTO_TCP */);
	delete it->second;
    }
    STOP(SEND::SendRemoteUpdates);

    START(SEND::ClearDeltaMasks);
    //
    // (5) Clear delta masks
    //
    store->Begin();
    while ( (obj = store->Next()) != NULL ) {
	obj->ClearDeltaMask();
    }
    STOP(SEND::ClearDeltaMasks);

    CHECK_ALWAYS_INVARIANTS();

    //
    // Consider frequent migrations (for some policies)
    //
    START(SEND::MigrateConsiderFreq);
    // OPTIMIZE OBJECT PLACEMENT WITH MIGRATION
    HandleMigrateConsiderFreq(now);
    STOP(SEND::MigrateConsiderFreq);

    //
    // (6) Perform object migrations that were requested
    //
    START(SEND::PerformMigrations);
    PerformMigrations();
    STOP(SEND::PerformMigrations);

    CHECK_ALWAYS_INVARIANTS();

    //
    // (7) Look over the cached pubs we have again and see if they give
    //     us any new matches...

    OMEvent *ev;
    m_PubSubManager->BeginCachedEventIter();
    while ( (ev = dynamic_cast<OMEvent *>(m_PubSubManager->GetNextCachedEvent())) != NULL ) {
	DB(5) << "processing old: " << ev << endl;
	START(SEND::HandleEvent);
	HandleEvent(ev, true);
	STOP(SEND::HandleEvent);
    }

    //
    // (8) Let Mercury do its thing
    //

    START(SEND::PubSubManager::DoWork2);
    m_PubSubManager->Send( TIME_LEFT_MSEC(stop) );
    STOP(SEND::PubSubManager::DoWork2);

    // (8.5) DirectRouter may need todo work?
    m_DirectRouter->DoWork (0);

#ifndef ENABLE_REALNET_THREAD
    //
    // (9) Do periodic Network processing
    //
    // START(SEND::RealNet::DoWork);     // XXX ASHWIN: We do this in the beginning of Recv()
    // anyway; we can just give it a larger timeout there, if needed.
    // RealNet::DoWork();
    // STOP(SEND::RealNet::DoWork);
#endif

    //
    // (10) Check for pathological cases and warn if needed!
    // 
    DBG << "Sending: pubCount=" << pubCount
	<< " subCount=" << subCount 
	<< " matchCount=" << matchCount << endl;

    // warn on pathological cases!
    if (pubCount > MAX_PUBCOUNT_WARN) {
	WARN << "Application sent A LOT of pubs in one frame: " << pubCount
	     << endl;
    }
    if (subCount > MAX_SUBCOUNT_WARN) {
	WARN << "Application send A LOT of subs in one frame: " << subCount
	     << endl;
    }
    if (matchCount > MAX_MATCHCOUNT_WARN) {
	WARN << "Application got A LOT of matches in one frame: " << matchCount
	     << endl;
    }
    pubCount = 0;
    subCount = 0;
    matchCount = 0;

    //
    // Done!
    //

    CHECK_EXIT_INVARIANTS();

    Unlock();

    STOP(SEND);

    return;
}


void Manager::Recv(uint32 timeout, bool doMaintenance)
{
    NOTE(Manager::RECV_TIMEOUT, timeout);
    uint64 stop = TIME_STOP_USEC(timeout);

    Lock();

    CHECK_ENTRY_INVARIANTS();

    GObject *obj;
    MsgApp *appmsg;
    OMEvent *ev;

    DBG << "Dumping ObjectStore:\n" << DumpObjectStore();
    DBG << "Dumping ObjectInfo:\n"  << DumpObjectInfo();
    DBG << "Dumping NodeLoadMap:\n" << DumpNodeLoadMap();
    DBG << "Dumping ReplicaConnections:\n" << DumpReplicaConnections();

    //// XXX ASHWIN: this is the only place RealNet is given
    //// a timeslice now. how much time is appropriate? 
#ifndef ENABLE_REALNET_THREAD
    // (7) Do periodic Network processing
    RealNet::DoWork( MIN(timeout / 3, 1) );       // CHANGED FROM 10 -> 1
#endif

    //// XXX: ASHWIN: this Recv() does not respect timeouts now
    //// worst case, we fall behind badly in frame rates. 

    // give some time to ensure forward progress
    m_PubSubRouter->Recv ( MAX(TIME_LEFT_MSEC(stop), 1) );

    START(RECV);

    if (ENABLE_LOAD_INFO)
	m_LoadManager->DoWork();

    // Messages from PubSubRouter
    while ( (ev = m_PubSubEvents->ReadEvent<OMEvent>(PUB_OM)) != NULL ) {
	// XXX HACK: use pubs as a mechanism for discovering other
	// nodes for the random placement policy
	if (streq(PLACEMENT_POLICY, "rand") && ev->GetSID() != GetSID()) {
	    ((RandomPlacementPolicy *)m_MigrationPolicy)->
		AddServer(ev->GetSID());
	}

	DB(5) << "processing match: " << ev << endl;
	START(RECV::HandleEvent);
	HandleEvent(ev, false);
	matchCount++;
	STOP(RECV::HandleEvent);

	if ( TIME_LEFT_MSEC(stop) <= 3 )
	    break; // leave at least 3 msec to to the rest

	//delete ev; -- given to pubsubmanager
    }

    // warn on some pathological cases
    if (matchCount > MAX_MATCHCOUNT_WARN) {
	WARN << "Received too many matches in one frame: " << matchCount << endl;
    }

    CHECK_ALWAYS_INVARIANTS();

    // Handle local matches (pubs we sent that matched local Interests)
    while ( (ev = dynamic_cast<OMEvent *>(m_PubSubManager->GetNextLocalMatch())) != NULL ) {
	DB(5) << "processing local: " << ev << endl;
	START(RECV::HandleEvent);
	HandleEvent(ev, true);
	STOP(RECV::HandleEvent);
	//delete ev; -- given to pubsubmanager

	if ( TIME_LEFT_MSEC(stop) <= 2 )
	    break; // leave at least 2 msec to to the rest
    }

    CHECK_ALWAYS_INVARIANTS();

    /*
    // First process the "new" events that we received
    PubSubProcessLatestInterests();

    // some events we receive have "data" in them (like full objects)
    // that we should process
    while ( (ev = m_PubSubManager->GetNextDataEvent()) != NULL ) {
    DB(5) << "processing: " << ev << endl;
    HandleDataEvent(ev);
    }
    */

    ///////////////////////////////////////////////////////////////////////////

    // Note: we initially add all new objects to the PendingStore.
    // Once all new objects have been processed, we resolve the references
    // to the new objects for everything in the PendingStore.
    // Finally, we check to see if any of the objects in the PendingStore
    // now have all their references resolved and add those to the
    // ObjectStore.

    START(RECV::ProcessMessages);

    // Messages from DirectRouter
    while ( (appmsg = m_DirectRouter->ReadMsg()) != NULL ) {
	DB(5) << "processing: " << appmsg << endl;

	if (ENABLE_LOAD_INFO)
	    RecordLoadInfo(appmsg);

	MsgType type = appmsg->GetType();
	if (type == MSG_APP_UPDATE) {
	    START(RECV::HandleUpdate);
	    HandleUpdate((MsgAppUpdate *)appmsg);
	    STOP(RECV::HandleUpdate);

	    CHECK_ALWAYS_INVARIANTS();
	} else if (type == MSG_APP_UPDATE_ACK) {
	    START(RECV::HandleUpdateAck);
	    HandleUpdateAck((MsgAppUpdateAck *)appmsg);
	    STOP(RECV::HandleUpdateAck);

	    CHECK_ALWAYS_INVARIANTS();

	} else if (type == MSG_APP_REMOTE_UPDATE) {
	    START(RECV::HandleRemoteUpdate);
	    HandleRemoteUpdate((MsgAppRemoteUpdate *)appmsg);
	    STOP(RECV::HandleRemoteUpdate);

	    CHECK_ALWAYS_INVARIANTS();
	} else if (type == MSG_APP_REGISTER) {
	    START(RECV::HandleReplicaRegister);
	    HandleReplicaRegister((MsgAppRegister *)appmsg);
	    STOP(RECV::HandleReplicaRegister);

	    CHECK_ALWAYS_INVARIANTS();
	} else if (type == MSG_APP_FETCHREQ) {
	    START(RECV::HandleFetchReq);
	    HandleFetchReq((MsgAppFetchReq *)appmsg);
	    STOP(RECV::HandleFetchReq);

	    CHECK_ALWAYS_INVARIANTS();
	} else if (type == MSG_APP_FETCHRESP) {
	    START(RECV::HandleFetchResp);
	    HandleFetchResp((MsgAppFetchResp *)appmsg);
	    STOP(RECV::HandleFetchResp);

	    CHECK_ALWAYS_INVARIANTS();
	} else if (type == MSG_APP_MIGRATEREQ) {
	    HandleMigrateReq((MsgMigrateReq *)appmsg);

	    CHECK_ALWAYS_INVARIANTS();
	} else {
	    WARN << "Manager got unknown event type " << type << endl;
	}

	m_DirectRouter->FreeMsg(appmsg);

	if ( TIME_LEFT_MSEC(stop) <= 1 )
	    break; // leave at least 1 msec to to the rest
    }

    STOP(RECV::ProcessMessages);

    START(RECV::RegisterObjects);
    // PERFORM OBJECT POINTER RESOLUTION
    // Now see if any pending objects had all their references resolved
    CHECK_ALWAYS_INVARIANTS();
    RegisterObjects(&m_ObjectArrivals);
    m_ObjectArrivals.clear();
    STOP(RECV::RegisterObjects);


    ///// NOW DO PERIODIC MANAGEMENT AND TRANSMISSIONS /////

    TimeVal now;
    OS::GetCurrentTime(&now);

    CHECK_ALWAYS_INVARIANTS();

    START(RECV::RegistrationReqs);

    // determine if we send out load info this round
    bool transmitLoadInfo = false;
    CostMetric toLowWaterMark = 0, toHighWaterMark = 0, load = 0;
    SIDSet sent;
    if (ENABLE_LOAD_INFO) {
	PERIODIC( BROADCAST_LOAD_INFO_TIMER, now, 
	{
	    transmitLoadInfo = true;
	    load = GetLoad(now);
	    DBG << "current load: " << load << endl;
	    toLowWaterMark = 
		MAX(0, GetLowWaterMark() - load);
	    toHighWaterMark =
		MAX(0, GetHighWaterMark() - load);
	} );
    }

    // SEND AGGREGATED REGISTRATION REQs
    for (RegisterMapIter it = m_ToRegister.begin(); 
	 it != m_ToRegister.end(); it++) {

	if ( transmitLoadInfo ) {
	    it->second->SetLoadInfo(toLowWaterMark, toHighWaterMark, load);

	    DBG << "sending repreg to " << it->first
		<< " with load info: toLowWaterMark=" 
		<< toLowWaterMark << " toHighWaterMark="
		<< toHighWaterMark << endl;
	}

	///// MEASUREMENT
	if (g_MeasurementParams.enabled) {
	    for (vector<GInterest>::iterator it2=it->second->interests.begin();
		 it2 != it->second->interests.end(); it2++) {
		if (it2->nonce != 0) {
		    DiscoveryLatEntry ent(DiscoveryLatEntry::REG_SEND, 
					  0, it2->nonce);
		    LOG(DiscoveryLatLog, ent);
		}
	    }
	}
	///// MEASUREMENT

	SID target = it->first;
	m_DirectRouter->SendMsg(&target, it->second,
				// no reason these need to be reliable
				// this is softstate
				PROTO_UDP
				/* PROTO_TCP */);
	//DB(0) << "Sending Registration(" << it->first << ": " 
	//	  << it->second << endl;
	delete it->second;
    }
    m_ToRegister.clear();

    STOP(RECV::RegistrationReqs);

    CHECK_ALWAYS_INVARIANTS();

    START(RECV::NewObjects);

    // SEND NEW-OBJECTS TO NODES THAT JUST REGISTERED THEM
    for (OutgoingReplicaConnectionMapIter it = m_ReplicaConnSenderMap.begin();
	 it != m_ReplicaConnSenderMap.end(); it++) {
	SID target = it->first;
	MsgAppUpdate *msg = it->second->CreateUpdate(true);
	if (msg != NULL) {
	    DBG << "sending onlyNew(" << target << "): " << msg << endl;

	    ///// MEASUREMENT
	    if (g_MeasurementParams.enabled) {
		for (GUpdateMapIter it2 = msg->updates.begin();
		     it2 != msg->updates.end(); it2++) {
		    if (it2->second->nonce != 0) {
			ASSERT(it2->second->isNew || it2->second->isDelete);
			DiscoveryLatEntry ent(DiscoveryLatEntry::UPDATE_SEND, 
					      0, it2->second->nonce);
			LOG(DiscoveryLatLog, ent);
		    }
		}
	    }
	    ///// MEASUREMENT

	    m_DirectRouter->SendMsg(&target, msg);
	}
    }

    STOP(RECV::NewObjects);

    CHECK_ALWAYS_INVARIANTS();

    // for legacy code that doesn't know about Manager::Maintenance()
    if (doMaintenance) {
	Maintenance( TIME_LEFT_MSEC(stop) );
    }

    STOP(RECV);

    CHECK_EXIT_INVARIANTS();

    Unlock();

    return;
}

void Manager::Maintenance(uint32 timeout)
{
    START(MAINTENANCE);
    uint64 stop = TIME_STOP_USEC(timeout);

    // DirectRouter may need todo work

    START(MAINTENANCE::DR::DoWork);
    m_DirectRouter->DoWork ( TIME_LEFT_MSEC(stop) );
    STOP(MAINTENANCE::DR::DoWork);

    /* 
     * XXX ASHWIN: Boy, this is unnecessary. WANMercuryNode::Recv() is called at 
     * least once anyways. 
     */
    /*
      START(MAINTENANCE::PubSubMaintenance);
      m_PubSubRouter->Maintenance ( TIME_LEFT_MSEC(stop) );
      STOP(MAINTENANCE::PubSubMaintenance);
    */

    START(RECV::Timers);

    TimeVal now;
    OS::GetCurrentTime(&now);

    // Force ourselves to do this even if we don't have time left
    // -- otherwise we will lose replicas, etc.
    //
    // DO TRANSMISSIONS OF REPLICA REGISTER HEART BEATS
    PERIODIC( REPLICA_MAINTAIN_HB, now, HandleReplicaPeriodic(now) );

    CHECK_ALWAYS_INVARIANTS();

    // GARBAGE COLLECT EXPIRED SOFT STATE
    if (TIME_LEFT_MSEC(stop) > 0) {
	PERIODIC( MIN(MIN(REPLICA_MAINTAIN_TTL, INTEREST_LINK_TTL), 
		      REPLICA_INTEREST_TTL)/SOFTSTATE_EXPIRE_GRAINULARITY,
		  now, HandleObjectInfoPeriodic(now) );
    }

    CHECK_ALWAYS_INVARIANTS();

    if (TIME_LEFT_MSEC(stop) > 0) {
	PERIODIC( REPLICA_MAINTAIN_TTL/SOFTSTATE_EXPIRE_GRAINULARITY, 
		  now, HandleReplicaConnPeriodic(now) );
    }

    CHECK_ALWAYS_INVARIANTS();

    if (TIME_LEFT_MSEC(stop) > 0) {
	PERIODIC( DELETED_GUID_TTL/SOFTSTATE_EXPIRE_GRAINULARITY, 
		  now, HandleDeletedGUIDsPeriodic(now) );

	CHECK_ALWAYS_INVARIANTS();

	PERIODIC( FORWARDING_PTR_TTL/SOFTSTATE_EXPIRE_GRAINULARITY, 
		  now, HandleForwardingPtrsPeriodic(now) );
    }

    STOP(RECV::Timers);

    CHECK_ALWAYS_INVARIANTS();

    if (TIME_LEFT_MSEC(stop) > 0) {
	START(RECV::MigrateConsider);
	// OPTIMIZE OBJECT PLACEMENT WITH MIGRATION
	PERIODIC( MIGRATION_CONSIDER, now, HandleMigrateConsider(now) );
	STOP(RECV::MigrateConsider);
    }

    STOP(MAINTENANCE);
}

void Manager::Exec(uint32 timeout)
{
    uint64 stop = TIME_STOP_USEC(timeout);
    Send( timeout );
    Recv( TIME_LEFT_MSEC(stop) );
}

///////////////////////////////////////////////////////////////////////////////
///// DELTA, UPDATE, PUBLICATION HANDLING

void Manager::HandleEvent(OMEvent *ev, bool cached)
{
    if (! HandlePub(ev, cached) ) {
	delete ev;
    }
}

bool Manager::HandlePub(OMEvent *ev, bool cached)
{
    guid_t guid = ev->GetGUID();
    sid_t  sid  = ev->GetSID();

    DB(5) << "processing pub " << ev << endl;

    //
    // Quickly Filter out objects we deleted ourselves
    // 

    if (sid == GetSID() && 
	m_DeletedGUIDs.find(guid) != m_DeletedGUIDs.end()) {
	DB(1) << "Got pub for object we deleted; ignoring" << endl;
	return false;
    }

    //
    // Is our state sane?
    //

    CHECK_ALWAYS_INVARIANTS();

    //
    // This function is called a lot, so optimize these data structures a bit
    //

    static bool inited = false;
    // List of objects that are potentially interested in this event
    static vector<GObjectInfo *> hasInterest;
    // How long each object requested the interest to last
    static vector<uint32>        interestTimes;
    if (!inited) {
	inited = true;
	hasInterest.reserve(32);
	interestTimes.reserve(32);
    }
    hasInterest.clear();
    interestTimes.clear();

    //
    // Filter out pubs that we are not really interested in
    //

    Tags tags = ev->GetTags();
    if (tags.size() == 0) {
	// This is a bug I believe
	WARN << "Got event with no tags: " << ev << endl;
	return false;
    }

    // See if we already have a copy
    GObject *other_obj = FindObject(guid);
    // The maximum time we should keep this interest
    uint32 maxInterestTime = 0;
    // Map from InterestID to object GUIDs
    InterestIDMap *assoc = m_PubSubManager->GetInterestIDMap();
    // Current interests
    InterestMap *imap    = m_PubSubManager->GetCurrInterests();

    // DEBUG: Some Debugging Stats
    uint32 numValidTags = 0;
    uint32 numMatchObjs = 0;
    uint32 numValidObjs = 0;

    for (Tags::iterator it = tags.begin(); it != tags.end(); it++) {
	GUID inid = *it;

	// the guid is the interest's guid -- look up each associated obj
	InterestIDMapIter start = assoc->lower_bound(inid);
	InterestIDMapIter end   = assoc->upper_bound(inid);
	if (start == end) {
	    DB(1) << "got a tag for Interest that we don't have: "
		  << inid << endl;
	    continue;
	}

	// DEBUG:
	numValidTags++;

	for ( ; start != end; start++) {
	    GUID my_guid = start->second;
	    if (my_guid == guid)
		// objects are obviously interested in themselves; don't care
		continue;

	    // DEBUG:
	    numMatchObjs++;

	    GObjectInfoMapIter p = m_ObjectInfo.find(my_guid);
	    if (p == m_ObjectInfo.end()) {
		DB(1) << "got a tag in pub for object we don't have " 
		      << "an info entry for! guid=" << my_guid << " pub=" 
		      << ev << endl;
		// This is actually OK, since the object could have been 
		// deleted or migrated in the meantime. But this should be 
		// rare.
		continue;
	    }
	    GObject *obj = m_Game->GetObjectStore()->Find(p->first);
	    if (obj == NULL) {
		// this is OK, since the object could have been deleted or 
		// migrated but it means we are not the owner and don't care 
		// about making interest links from it
		continue;
	    }
	    if (obj->IsReplica()) {
		DB(1) << "got a tag in a pub for replica! "
		      << "guid=" << my_guid << " pub=" << ev << endl;
		// this is OK, since the object could have migrated
		// but it means we are not the owner and don't care about 
		// making interest links from it
		continue;
	    }

	    // DEBUG:
	    numValidObjs++;

	    // Check to see if our primary is really interested in this
	    // event...
	    uint32 interestTime;
	    if (other_obj != NULL)
		// If we already have a copy of the other object, it contains
		// more up-to-date info than the event, so use it to determine
		// interest rather than the event.
		//
		// Jeff: Note, we can NOT just ignore this pub even if we
		// already have a copy of the object because now we are
		// reliant on pubs to discover *new* interests, even if we
		// already have those objects in our object store. I.E.,
		// if A->B, and we have A and B, but don't get a pub from
		// B matching one of A's interests, we will never create
		// a the link A->B to begin with. ... We may want to change
		// this... to avoid having to use tags to begin with...
		interestTime = obj->InterestTime(other_obj, imap);
	    else
		// Otherwise we don't have a copy yet, so we have to ask the
		// object if it is "maybe" interested in the event.
		interestTime = obj->InterestTime(ev, imap);

	    if (interestTime > 0) {
		GObjectInfo *prim = p->second;
		hasInterest.push_back(prim);
		interestTimes.push_back(interestTime);
		maxInterestTime = MAX(maxInterestTime, interestTime);
	    }
	}
    }

    if (hasInterest.size() == 0) {
	DB(3) << "Discarding pub that no one is interested in: "
	      << guid << "(numValidTags=" << numValidTags
	      << " numMatchObjs=" << numMatchObjs
	      << " numValidObjs=" << numValidObjs << ")" << endl;

	// looks like no one is actually interested... can discard event
	return false;
    }

    //
    // If we get here, then at least one of our objects is interested
    //

    bool isNew = false;

    // We are (still) interested in this object, remember that fact
    GObjectInfoMapIter p = m_ObjectInfo.find(guid);
    GObjectInfo *other;
    if (p == m_ObjectInfo.end()) {
	if (sid == GetSID()) {
	    // This means (I hope) that there is a pub still in the system
	    // for an object we deleted recently... it should hopefully
	    // be deleted by itself soon (because it is soft state)
	    DB(1) << "Got pub from self but we don't have the objinfo: "
		  << ev << endl;
	    return false;
	}
	other = AllocGObjectInfo(guid, sid, true);
	isNew = true;

	// debug
	if (cached)
	    NOTE(NEW_PUB_MATCH_CACHE, 1);
	else
	    NOTE(NEW_PUB_MATCH_NOCACHE, 1);
    } else {
	other = p->second;
    }

    //
    // Events may tell us that the location of an object changed
    //

    // if the sid is from us, then don't pay any attention to it, because
    // it must be more stale than the information we have about the object
    // if it is accurate; it may actually be out-of-date
    if (sid != GetSID()) {
	if (other->IsReplica() && 
	    // it could be an object that is currently migrating to us...
	    // in which case we don't want to set the sid because it will
	    // be set to us... XXX actually it shouldn't matter...
	    m_MigrateIncoming.find(other->GetGUID()) == 
	    m_MigrateIncoming.end() &&
	    // it could also be the case that we deleted the obj recently
	    // and some other node still has a stale pub in the system
	    m_DeletedGUIDs.find(other->GetGUID()) ==
	    m_DeletedGUIDs.end() ) {

	    other->SetSID( sid );

	} else {
	    // If we are here, that means we got a pub to an object we own
	    // but the pub says we don't. That means it is stale, so ignore it
	    return false;
	}

	// we are newly interested in this object, send a register request
	// immediately (otherwise we will have to wait for the HB timer)
	if (!cached && isNew) {
	    DB(5) << "sending new register: " << guid << endl;
	    MsgAppRegister *msg = GetCurrRegisterMsg(sid);
	    GInterest in(guid, 0);
	    in.newRegister = true;

	    if (!cached) {
		// only measure the latency of non-cached events, since if
		// it is new and was in our cache, then counting it would
		// count the lantecy of the time during which it was in our
		// cache also, which isn't what we want to measure

		///// MEASUREMENT
		OBJECT_LOG_DO( in.nonce = ev->GetNonce() );
		///// MEASUREMENT
	    }

	    msg->interests.push_back(in);
	}

    } else {
	// If the SID says it was from us, we better have a primary copy,
	// otherwise the event is stale...
	if (!other_obj || other_obj->IsReplica()) {
	    return false;
	} 
    }

    ASSERT(other->GetGUID() == guid);
    ASSERT(other->GetSID()  == sid);

    //
    // Register or refresh our interest in this particular object.
    //

    other->Refresh( maxInterestTime + ManagerParams::REPLICA_INTEREST_TTL );

    //
    // For each of our own objects that said it was interested in this event
    // refresh the interest-link between it and the remote object
    //

    for (int i=0; i<(int)hasInterest.size(); i++) {
	GObjectInfo *prim = hasInterest[i];
	uint32 ttl = interestTimes[i];

	//DB(0) << "Refreshing interest from: " << prim->GetGUID()
	//	  << " -> " << other->GetGUID() << endl;

	DB(5) << "refreshing interest: " << prim->GetGUID()
	      << " -> " << other->GetGUID() << endl;
	prim->RefreshInterest(other, ttl + ManagerParams::INTEREST_LINK_TTL);
    }

    //
    // Maybe check debug invariants now...
    //

    CHECK_ALWAYS_INVARIANTS();

    //
    // If the event contained a full object, then construct it!
    //

    if (ev->GetSID() != GetSID() && // no need to construct if ours!
	ev->GetUpdate() != NULL &&
	// was ours but we deleted it!
	m_DeletedGUIDs.find(ev->GetGUID()) == m_DeletedGUIDs.end() && 
	// is about to be ours!
	m_MigrateIncoming.find(ev->GetGUID()) == m_MigrateIncoming.end() &&
	other_obj == NULL // don't add if we already have a copy!
	) {
	// This pub contained a new object, construct it right away
	DB(1) << "Event contained new object; constructing: " << ev << endl;

	//START(HANDLE_PUB::NewObj);

	if (g_MeasurementParams.enabled) {
	    ReplicaLifetimeEntry e;
	    e.guid = guid;
	    e.type = (char)ReplicaLifetimeEntry::PCREATE;
	    LOG(ReplicaLifetimeLog, e);
	}

	GUpdate *update = ev->GetUpdate();
	ASSERT(update->delta);
	SIDMap unresolved;
	GObject *obj = m_Game->Construct(other,
					 update->delta, 
					 update->mask, 
					 &unresolved);
	ASSERT(obj != NULL);
	ASSERT(obj->GetSID() != GetSID());

	m_Game->GetPendingStore()->_ManagerAdd(obj);
	m_ObjectArrivals[obj->GetGUID()] = obj;

	Fetch(&unresolved, obj->GetGUID());

	//STOP(HANDLE_PUB::NewObj);
    }

    //
    // Save the pub in the PubSubManager's cache so we can look at it again!
    //

    m_PubSubManager->AddCachedEvent(ev);

    //
    // Done!
    //

    return true;
}


void Manager::HandleUpdate(MsgAppUpdate *msg)
{
    sid_t sid = msg->GetSender();

    DBG << "called" << endl;

    IncomingReplicaConnectionMapIter p = m_ReplicaConnReceiverMap.find(sid);
    ReplicaConnectionReceiver *conn;
    if (p == m_ReplicaConnReceiverMap.end()) {
	conn = new ReplicaConnectionReceiver(sid, msg->epoch,
					     msg->updateSeqno-1, this);
	m_ReplicaConnReceiverMap[sid] = conn;
    } else {
	conn = p->second;
    }

    ///// MEASUREMENT
    if (g_MeasurementParams.enabled) {
	for (GUpdateMapIter it = msg->updates.begin();
	     it != msg->updates.end(); it++) {
	    if (it->second->nonce != 0) {
		ASSERT(it->second->isNew || it->second->isDelete);
		DiscoveryLatEntry ent(DiscoveryLatEntry::UPDATE_RECV, 1, 
				      it->second->nonce);
		LOG(DiscoveryLatLog, ent, msg->recvTime); // use recv time
	    }
	}
    }
    ///// MEASUREMENT

    conn->HandleUpdate(msg);

    MsgAppUpdateAck ack = MsgAppUpdateAck();
    ack.epoch    = conn->GetEpoch();
    ack.ackSeqno = conn->GetLastSeqNoAck();

    // xxx debug
    OS::GetCurrentTime(&ack.timestamp);

    START(RECV::HandleUpdate::SendMsg);
    m_DirectRouter->SendMsg(&sid, &ack);
    STOP(RECV::HandleUpdate::SendMsg);
}

void Manager::HandleUpdateAck(MsgAppUpdateAck *msg)
{
    sid_t sid = msg->GetSender();

    DBG << "called" << endl;

    OutgoingReplicaConnectionMapIter p = m_ReplicaConnSenderMap.find(sid);
    ReplicaConnectionSender *conn;
    if (p == m_ReplicaConnSenderMap.end()) {
	WARN << "Got ack from " << sid << " but we don't have a connection "
	     << "allocated" << endl;
	return;
    }

    conn = p->second;
    conn->HandleUpdateAck(msg);
}

void Manager::HandleRemoteUpdate(MsgAppRemoteUpdate *msg)
{
    GObject *obj;
    SIDMap unresolved;

    DBG << "called" << endl;

    sid_t sid = msg->GetSender();

    for (GUpdateMapIter it = msg->updates.begin();
	 it != msg->updates.end(); it++) {

	GUpdate *update = it->second;
	guid_t guid = update->guid;

	ASSERT(update->isUpdate);
	ASSERT(!update->isNew);
	ASSERT(!update->isDelete); // XXX eventually allow this?

	DBG << "received an update for GUID->" << guid << endl;

	obj = m_Game->GetObjectStore()->Find(guid);

	if (obj == NULL || obj->IsReplica()) {
	    // we don't own this object, see if we have a forwarding pointer
	    FwdPtrMapIter p = m_ForwardingPtrs.find(guid);

	    if (p != m_ForwardingPtrs.end()) {
		DB(1) << "had forwarding pointer for " << guid 
		      << " which we believe now resides on " 
		      << *p->second << endl;

		// if so, forward the update to the new dude
		MsgAppRemoteUpdate msg = MsgAppRemoteUpdate(sid);
		msg.updates[update->guid] = update;
		m_DirectRouter->SendMsg(p->second, &msg,
					PROTO_UDP
					/* PROTO_TCP */);

		// HACK: avoid duplicate delete of update
		msg.updates.clear();
		continue;
	    } else if (m_DeletedGUIDs.find(guid) !=
		       m_DeletedGUIDs.end()) {
		// just deleted it, ignore the update
		DBG << "remote update for deleted object " << guid << endl;
		continue;
	    } else if (m_MigrateIncoming.find(guid) !=
		       m_MigrateIncoming.end()) {
		// got a (probably forwarded) update for object that migrated
		// to us but hasn't made it into our object store yet
		// apply it to the temp replica...
		obj = m_Game->GetPendingStore()->Find(guid);
		ASSERT(obj != NULL);

	    } else {
		WARN << "not primary object for update for " << guid 
		     << " and no forwarding pointer (dropping)" << endl;
		continue;
	    }

	} else {
	    ASSERT(obj->GetSID() == GetSID());
	}

	ASSERT(update->delta);

	obj->UnpackUpdate(update->delta, update->mask, &unresolved);
	// this caused a change in object state, so we need to broadcast
	// this change to all subscribers the next Send() round...
	obj->GetDeltaMask().Merge( update->mask );

	// XXX What to do with the update if it results in
	// unresolved pointers? ignore the update? that isn't too
	// good... Let the obj deal with that for now

	Fetch(&unresolved, guid);
    }
}

///////////////////////////////////////////////////////////////////////////////
///// INTEREST INFERENCE

/*
  void Manager::PubSubProcessLatestInterests()
  {
  PotentialInterests *pot = m_PubSubManager->GetLatestInterests();

  for (PotentialInterestsIter it = pot->begin();
  it != pot->end(); it++) {
  GUID target = it->second->target;
  SID  sid    = it->second->sid;

  GObjectInfoMapIter p = m_ObjectInfo.find(target);
  GObjectInfo *other;
  if (p == m_ObjectInfo.end()) {
  if (sid == GetSID()) {
  // This means (I hope) that there is a pub still in the 
  // system for an object we deleted recently... it should 
  // hopefully be deleted by itself soon (because it is soft 
  // state)
  DB(1) << "interest in obj from  self but we don't "
  << " have the objinfo: " << target << endl;
  continue;
  }
  other = AllocGObjectInfo(target, sid, true);

  // we are newly interested in this object, send a register 
  // request immediately (otherwise we will have to wait for the 
  // HB timer)
  if (sid != GetSID()) {
  //DB(0) << "sending new register: " << guid << endl;
  MsgAppRegister *msg = GetCurrRegisterMsg(sid);
  GInterest in(target, 0);
  in.newRegister = true;

  ///// MEASUREMENT
  OBJECT_LOG_DO( in.nonce = it->second->nonce );
  ///// MEASUREMENT

  msg->interests.push_back(in);
  }
  } else {
  other = p->second;
  }
  other->Refresh();
  }
  }

  void Manager::PubSubInterestsPeriodic(TimeVal& now)
  {
  // XXX: Jeff: Is this even necessary? We should be able to just
  // use the "graph" built up from the PubSubManager... oh well
  // need it for compat for now.
  //
  // Moreover, is this an N^2 computation (N = |objects|)?

  GObject *obj;

  ObjectStore *store = GetObjectStore();
  store->Begin();
  while ((obj = store->Next()) != NULL) {
  if (obj->IsReplica()) {
  continue;
  }
  GUID guid = obj->GetGUID();

  PotentialInterests *pot = 
  m_PubSubManager->GetCurrInterests(guid);

  GObjectInfoMapIter infop = m_ObjectInfo.find(guid);
  ASSERT(infop != m_ObjectInfo.end());
  GObjectInfo *info = infop->second;

  for (PotentialInterestsIter it = pot->begin();
  it != pot->end(); it++) {
  GUID target = it->second->target;

  GObjectInfoMapIter p = m_ObjectInfo.find(guid);

  // assume we just called PubSubProcessLatestInterests(), so
  // all objects seen recently should be registered with some
  // objinfo instance.
  if (p == m_ObjectInfo.end()) {
  WARN << "PubSubManager says " << guid << " is interested in "
  << target << " but we have no obj info for it!" << endl;
  continue;
  }
  GObjectInfo *other = p->second;

  // mark that this particular object is interestd in other
  info->RefreshInterest(other);
  }
  }

  return;
  }
*/

///////////////////////////////////////////////////////////////////////////////
///// OBJECT POINTER RESOLUTION

enum { RESOLVED,    // guaranteed to be resolved
       UNRESOLVED,  // guaranteed to be unresolved
       DEPENDENT }; // resolution dependent on current recursive call

int Manager::MarkResolvable(GObjectMap *new_map,
			    GUIDSet *resolved, 
			    GUIDSet *unresolved,
			    GUIDSet *temp_resolved,
			    GObject *obj) {
    int status = RESOLVED;
    guid_t guid = obj->GetGUID();
    GObjectRefList refs;
    obj->GetRefs(&refs);

    DBG << " Trying to resolve: " << guid << endl;

    // already marked by previous recursive call
    if ( resolved->find(guid) != resolved->end() ) {
	DBG << " ** was resolved before" << endl;
	return RESOLVED;
    }
    if ( temp_resolved->find(guid) != temp_resolved->end() ) {
	DBG << " ** temp resolved during this call" << endl;
	return DEPENDENT;
    }
    if ( unresolved->find(guid) != unresolved->end() ) {
	DBG << " ** was found to be unresolved before" << endl;
	return UNRESOLVED;
    }


    // temporarily assume that we are resolvable. This is because a recursive
    // call might loop back to us, in which case we say "yes" unless we turn
    // out to be unresolvable.
    temp_resolved->insert(guid);

    for (GObjectRefListIter it = refs.begin(); it != refs.end(); it++) {

	GUID target = (*it)->GetTarget();
	GObjectMapIter p = new_map->find(target);
	if ( p != new_map->end() && p->second == NULL ||
	     m_DeletedGUIDs.find(target) != m_DeletedGUIDs.end() ) {
	    // was marked as deleted, resolve the ref immediately
	    obj->ResolveRef(*it, NULL);
	    // resolution status remains whatever it was before
	    // since this object is now resolved as deleted
	    continue;
	}

	GObject *done    = m_Game->GetObjectStore()->Find(target);
	GObject *pending = m_Game->GetPendingStore()->Find(target);

	if (done != NULL) {
	    continue;
	} else if (pending == NULL) {
	    status = UNRESOLVED;

	    DBG << " ** pointer to " << target << " is missing" << endl;
	} else {
	    int ret = MarkResolvable(new_map, resolved, unresolved, 
				     temp_resolved, pending);

	    switch(ret) {
	    case RESOLVED:
		continue;
	    case DEPENDENT:
		status = DEPENDENT;
		continue;
	    case UNRESOLVED:
		status = UNRESOLVED;
		break;
	    default:
		Debug::die("unknown resolution status: %d", ret);
	    }

	    DBG << " ** pointer to " << target << " was " <<
		(done != NULL ? 
		 "in the object store" :
		 status == RESOLVED ? 
		 "resolved recursively" :
		 status == DEPENDENT ?
		 "temp resolved recursively" :
		 "not resolved (see prev msgs)") << endl;
	}

	if (status == UNRESOLVED) break;
    }

    switch(status) {
    case RESOLVED:
	// cool! we know we are resolved because all our children and are not
	// dependent on any objests undergoing resolution in this call
	temp_resolved->erase(guid);
	resolved->insert(guid);
	DBG << " ** resolved!" << endl;
	break;
    case DEPENDENT:
	// can't say anything about this object until the entire call stack
	// returns because it is dependent on an object undergoing resolution
	DBG << " ** dependent!" << endl;
	break;
    case UNRESOLVED:
	// turned out we weren't resolvable
	temp_resolved->erase(guid);
	unresolved->insert(guid);
	DBG << " ** unresolved!" << endl;
	break;
    default:
	Debug::die("unknown resolution status: %d", status);
    }

    return status;
}


void Manager::Resolve(GObject *obj) {
    ASSERT(obj);
    GObjectRefList refs;
    obj->GetRefs(&refs);
    CHECK_ALWAYS_INVARIANTS();

    for (GObjectRefListIter it = refs.begin(); it != refs.end(); it++) {
	//if ( it->second->IsResolved() ) continue;

	GObject *ptr;
	GUID guid = (*it)->GetTarget();
	ptr = FindObject(guid);
	/*m_Game->GetObjectStore()->Find(it->first);
	  if (ptr == NULL) {
	  ptr = m_Game->GetPendingStore()->Find(it->first);
	  }
	*/
	ASSERT(ptr != NULL);
	//it->second->ResolveRef(ptr);
	obj->ResolveRef(*it, ptr);
    }
    CHECK_ALWAYS_INVARIANTS();

    m_Game->GetPendingStore()->_ManagerRemove(obj->GetGUID());
    AddReplica(obj);
}

void Manager::RegisterObjects(GObjectMap *todo) {
    GUIDSet unresolved, resolved, temp_resolved;
    GObject *obj;

    DBG << "registering new objects..." << endl;
    CHECK_ALWAYS_INVARIANTS();

    // Now we need to do DFS to determine which objects are 
    // 100% recursively resolvable.
    m_Game->GetPendingStore()->Begin();
    while ( (obj = m_Game->GetPendingStore()->Next()) != NULL) {

	temp_resolved.clear();
	if ( MarkResolvable(todo, &resolved, &unresolved, 
			    &temp_resolved, obj) == DEPENDENT) {
	    ASSERT(obj->GetGUID() != GUID_NONE);
	    resolved.insert(obj->GetGUID());

	    DBG << " ** resolved at end of callstack!" << endl;
	}

    }
    CHECK_ALWAYS_INVARIANTS();

    // now add the resolved objects to the object store
    for (GUIDSetIter it = resolved.begin(); it != resolved.end(); it++) {
	guid_t guid = *it;
	DBG << "resolving: " << guid << endl;

	GObject *obj = m_Game->GetPendingStore()->Find(guid);
	ASSERT(obj);
	Resolve(obj);

	// see if this object we resolved was one that was migrating to
	// us. if so, then we can now send back a response saying we
	// are a-ok to proceed.
	//
	// XXX: Right now we are adding the newly migrated object to our
	// store as a normal replica... we need some flag on it or something
	// to ensure that it is not accidently garbage collected before
	// the migration process finishes! For now just ensure that the
	// garbage collection timer is longer than the migration timeout
	if (m_MigrateIncoming.find(guid) != m_MigrateIncoming.end()) {
	    HandleMigrateOk(guid);
	}
    }

    DBG_DO {
	for (GUIDSetIter it = unresolved.begin(); 
	     it != unresolved.end(); it++) {

	    DBG << "unresolved: " << *it << endl;
	    obj = m_Game->GetPendingStore()->Find(*it);
	    if (obj != NULL) {
		DBG << " ** missing:";
		GObjectRefList refs;
		obj->GetRefs(&refs);
		for (GObjectRefListIter j = refs.begin(); j != refs.end(); j++) {
		    DBG << " " << (*j)->GetTarget();
		}
		DBG << endl;
	    } else {
		DBG << "** NULL!" << endl;
	    }
	}
    }
}

///////////////////////////////////////////////////////////////////////////////
///// REPLICA MANAGEMENT

void Manager::AddReplica(GObject *obj)
{
    GObjectInfoMapIter p = m_ObjectInfo.find(obj->GetGUID());

    // This is because the info is always allocated before or during
    // object construction...
    ASSERT(p != m_ObjectInfo.end());
    // Every object must have a unique info
    ASSERT(p->second == obj->GetInfo());

    {
	// Sanity check
	GObjectInfo *info = p->second;
	ASSERT(info != NULL);
	if (info->GetSID() == GetSID()) {
	    WARN << "Trying to add replica " << info->GetGUID() 
		 << " that has us marked as the owner!" << endl;
	    GObject *ours = 
		m_Game->GetObjectStore()->Find(info->GetGUID());
	    if (ours != NULL && !ours->IsReplica()) {
		WARN << "It is fucking ours! Ignoring add." << endl;
	    } else if (m_DeletedGUIDs.find(info->GetGUID()) != 
		       m_DeletedGUIDs.end()) {
		WARN << "We fucking deleted it. Ignoring add." << endl;
		return;
	    } else {
		WARN << "What the fuck? It isn't ours? Changing info to rep" 
		     << endl;
		//info->SetSID( obj->GetSID() );
		// um.. this is tautological since they are the same ref
		ASSERT(0);
	    }
	}

	list<GObjectRef *> unresolved;
	obj->GetRefs(&unresolved);
	ASSERT( unresolved.size() == 0 );
    }

    if (g_MeasurementParams.enabled) {
	ReplicaLifetimeEntry e;
	e.guid = obj->GetGUID();
	e.type = (char)ReplicaLifetimeEntry::STORE;
	LOG(ReplicaLifetimeLog, e);
    }

    ASSERT(obj->IsReplica());
    m_Game->GetObjectStore()->_ManagerAdd(obj);
}

void Manager::HandleReplicaPeriodic(TimeVal& now)
{
    DBG << "called" << endl;

    ObjectStore *store = m_Game->GetObjectStore();

    // Send hbs to primaries we are keeping replicas for
    for (GObjectInfoMapIter it = m_ObjectInfo.begin(); 
	 it != m_ObjectInfo.end(); it++) {
	GObject *obj = store->Find(it->first);
	if (obj != NULL && !obj->IsReplica()) continue;

	GObjectInfo *info = it->second;
	SID target = info->GetSID();
	ASSERT(target != SID_NONE);

	ASSERT(target != GetSID());

	MsgAppRegister *msg = GetCurrRegisterMsg(target);

	uint16 numInterests = 0;
	InterestLinkSet *backPtrs = info->GetBackPtrs();
	for (InterestLinkSetIter it2 = backPtrs->begin();
	     it2 != backPtrs->end(); it2++) {
	    if ((*it2)->IsStable()) ++numInterests;
	}

	GInterest in = GInterest(info->GetGUID(), numInterests);			
	msg->interests.push_back(in);
    }
}

void Manager::HandleReplicaConnPeriodic(TimeVal& now)
{
    DBG << "called" << endl;

    for (OutgoingReplicaConnectionMapIter it = m_ReplicaConnSenderMap.begin();
	 it != m_ReplicaConnSenderMap.end(); it++) {
	it->second->GarbageCollectRegistrations();
    }

    // XXX TODO: someway to garbage collect connections that are not being
    // used any longer -- this requires some way for tear down...
}

void Manager::HandleReplicaRegister(MsgAppRegister *msg)
{

    DBG << "called" << endl;

    for (uint32 i=0; i<msg->interests.size(); i++) {
	GInterest in = msg->interests[i];
	in.recvTime = msg->recvTime;
	GUID guid = in.guid;

	DBG << "called for guid=" << guid << endl;

	OutgoingReplicaConnectionMapIter p = 
	    m_ReplicaConnSenderMap.find(msg->GetSender());
	ReplicaConnectionSender *conn;
	if (p == m_ReplicaConnSenderMap.end()) {
	    conn = new ReplicaConnectionSender(msg->GetSender(), this);
	    m_ReplicaConnSenderMap[msg->GetSender()] = conn;
	} else {
	    conn = p->second;
	}

	GObject *obj = m_Game->GetObjectStore()->Find(guid);
	if ( obj == NULL || obj->IsReplica() ) {

	    // we don't own this object, see if we have a forwarding pointer
	    FwdPtrMapIter p = m_ForwardingPtrs.find(guid);

	    if (p != m_ForwardingPtrs.end()) {
		DB(1) << "had forwarding pointer for " << guid 
		      << " which we believe now resides on " 
		      << *p->second << endl;

		if (*p->second == msg->GetSender()) {
		    DB(1) << "forwarding pointer points back to sender;"
			  << " ignoring message" << endl;
		} else {
		    // if so, forward the register the new dude
		    MsgAppRegister rmsg = MsgAppRegister(msg->GetSender());
		    rmsg.interests.push_back(in);
		    m_DirectRouter->SendMsg(p->second, &rmsg,
					    // eschew 100% correctness for now 
					    // to get better performance :)
					    PROTO_UDP
					    /* PROTO_TCP */);
		}
	    } else if (m_DeletedGUIDs.find(guid) != m_DeletedGUIDs.end()) {
		// send back a response saying it was deleted

		DB(1) << "object " << guid << " was deleted" << endl;

		// send a delete signal next delta time
		if (!in.unregister)
		    conn->RegisterGUID(guid, true, in.newRegister, false, &in);
		else
		    conn->UnRegisterGUID(guid);

	    } else if (m_MigrateIncoming.find(guid) != 
		       m_MigrateIncoming.end()) {
		// object migrated to us but we haven't added it to our
		// object store just yet; remember the registration...

		SIDDeltaMap *regs = m_MigrateIncoming[guid].registrations;

		DBG << "saving register to migrating obj: " << msg << endl;

		if (in.unregister) {
		    regs->erase(msg->GetSender());
		} else {
		    if (regs->find(msg->GetSender()) == regs->end()) {
			(*regs)[msg->GetSender()] = DeltaInfo();
		    }
		    (*regs)[msg->GetSender()].Refresh();
		    if (in.newRegister)
			(*regs)[msg->GetSender()].isNew = true;
		}

		DBG << "done saving register to migrating obj" << endl;

	    } else {
		// WARN
		DB(0) << "got RegisterReplica req for guid we are not primary "
		      << "for: " << guid << "; does not have a delete "
		      << "record and we don't have a fwd ptr! dropping" << endl;

		// WTF? This means we got a very stale reg message
	    }
	    continue;
	}

	GObjectInfoMapIter p2 = m_ObjectInfo.find(guid);
	ASSERT(p2 != m_ObjectInfo.end());
	ReplicaPtrMapIter mp = 
	    p2->second->GetRegisteredReplicas()->find(msg->GetSender());

	// mark (or keep) the remote node as registered
	if (!in.unregister) {
	    /*
	      DB(0) << "got register for: " << guid 
	      << " newRegister=" << in.newRegister << endl;
	    */
	    conn->RegisterGUID(guid, false, in.newRegister, false, &in);

	    // For interest management only
	    if (mp == p2->second->GetRegisteredReplicas()->end()) {
		(*p2->second->GetRegisteredReplicas())[msg->GetSender()] = 
		    ReplicaPtr();
	    } else {
		mp->second.Refresh(in.numInterests);
	    }
	}
	// unregister the remote node
	else {
	    conn->UnRegisterGUID(guid);

	    // For interest management only
	    if (mp != p2->second->GetRegisteredReplicas()->end()) {
		p2->second->GetRegisteredReplicas()->erase(mp);
	    }
	}
    }

}

///////////////////////////////////////////////////////////////////////////////
///// SOFTSTATE GARBAGE COLLECTION

void Manager::HandleObjectInfoPeriodic(TimeVal & now)
{
    ObjectStore *store = m_Game->GetObjectStore();

    DBG << "garbage collecting stale replica info" << endl;

    // ##### (1) Garbage collect interest links that are no longer valid
    for (GObjectInfoMapIter it = m_ObjectInfo.begin(); 
	 it != m_ObjectInfo.end(); it++) {
	//
	// First remove all stale interest links. This function will also
	// refresh interests that are about to timeout if we are still
	// interested.
	//
	it->second->GarbageCollectInterests(false, now);
    }

    // build a list of objects we should not delete because we have pointers
    // to them... (XXX: this isn't very efficient to build this list every
    // time... should do this as a callback from the app maybe? but that
    // would entail a lot of work for the app if pointers change often.
    list<GObjectRef *> tmp;
    GUIDSet remoteRefs;
    GObject *obj;
    store->Begin();
    while( (obj = store->Next()) != NULL ) {
	if ( obj->IsReplica() ) continue;
	obj->GetAllRefs(&tmp);
    }
    for (list<GObjectRef *>::iterator it = tmp.begin(); 
	 it != tmp.end(); it++) {
	remoteRefs.insert((*it)->GetTarget());
    }
    tmp.clear();

    // ##### (2) Remove info for replicas that we are no longer interested in
    for (GObjectInfoMapIter it = m_ObjectInfo.begin(); 
	 it != m_ObjectInfo.end(); ) {
	GUID guid = it->first;
	GObject *obj = FindObject(guid);
	GObjectInfo *info = it->second;

	// don't garbage collect primary objects!
	if (obj != NULL && !obj->IsReplica()) {

	    // Remove registration for replicas no longer interested in us
	    // (This shouldn't be necessary because they should send explicit
	    // unregister requests to us. but save this to handle failures...)
	    ReplicaPtrMap *pmap = info->GetRegisteredReplicas();
	    for (ReplicaPtrMapIter sit = pmap->begin(); sit != pmap->end(); ) {
		if (sit->second.TimedOut(now)) {
		    DBG << sit->first << " no longer interested in " 
			<< guid << endl;

		    ReplicaPtrMapIter osit = sit;
		    osit++;
		    pmap->erase(sit);
		    sit = osit;
		} else {
		    sit++;
		}
	    }

	    it++;
	    continue;
	}

	// Don't destroy the replica if it is migrating to us!
	// Don't destroy the replica if we still have a pointer to it!
	if ( info->TimedOut(now) &&
	     m_MigrateIncoming.find(guid) ==
	     m_MigrateIncoming.end() &&
	     remoteRefs.find(guid) ==
	     remoteRefs.end() ) {
	    GObjectInfoMapIter oit = it;
	    oit++;

	    // if we are no longer interested, then no one better
	    // be pointing to this dude!
	    if (info->GetBackPtrs()->size() != 0) {
		WARN << "WTF do we still have back ptrs?" << endl;
		WARN << "info=" << info << endl;
		for (InterestLinkSetIter sit=info->GetBackPtrs()->begin();
		     sit != info->GetBackPtrs()->end(); sit++) {
		    WARN << "other=" << (*sit)->GetSource() << endl;
		}
		ASSERT(0);
	    }

	    // no longer interested!
	    DBG << "no longer interested in " << guid << endl;

	    // aggregate un-registrations
	    MsgAppRegister *msg = GetCurrRegisterMsg(info->GetSID());

	    // unregister interest in the object
	    GInterest in(guid, 0);
	    in.unregister = true;
	    msg->interests.push_back(in);

	    // inform app it can remove the rep
	    if (obj) {
		DB(1) << "destroying uninteresting object: " << guid << endl;
		m_Game->Destroy(guid);
		CHECK_ALWAYS_INVARIANTS();
	    } else {
		// Only necessary if there was no object. because if there
		// was, then the destructor of the object would call this.
		FreeGObjectInfo(info);
	    }

	    if (g_MeasurementParams.enabled) {
		ReplicaLifetimeEntry e;
		e.guid = guid;
		e.type = (char)ReplicaLifetimeEntry::TDESTROY;
		LOG(ReplicaLifetimeLog, e);
	    }

	    ASSERT(m_ObjectInfo.find(guid) == m_ObjectInfo.end());
	    ASSERT(FindObject(guid) == NULL);

	    // have to be careful with this iterator because the object
	    // destruction will remove an element from the set...
	    it = oit;
	} else {
	    it++;
	}
    }

}

void Manager::HandleDeletedGUIDsPeriodic(TimeVal & now)
{
    DBG << "garbage collecting expired deleted GUIDs" << endl;

    START(Manager::HandleDeletedGUIDsPeriodic);
    for (DeletedSetIter it = m_DeletedGUIDs.begin(); 
	 it != m_DeletedGUIDs.end(); ) {
	if (it->second->TimedOut(now)) {
	    DeletedSetIter oit = it;
	    oit++;
	    FreeTimedGUID(it->second);
	    m_DeletedGUIDs.erase(it);
	    it = oit;
	} else {
	    it++;
	}
    }
    STOP(Manager::HandleDeletedGUIDsPeriodic);
}

void Manager::HandleForwardingPtrsPeriodic(TimeVal & now)
{
    DBG << "garbage collecting expired forwarding pointers" << endl;

    START(Manager::HandleDeletedGUIDsPeriodic);
    for (FwdPtrMapIter it = m_ForwardingPtrs.begin();
	 it != m_ForwardingPtrs.end(); ) {
	if (it->second->TimedOut(now)) {
	    FwdPtrMapIter oit = it;
	    oit++;
	    FreeTimedSID(it->second);
	    m_ForwardingPtrs.erase(it);
	    it = oit;
	} else {
	    it++;
	}
    }
    STOP(Manager::HandleDeletedGUIDsPeriodic);
}

///////////////////////////////////////////////////////////////////////////////
///// REPLICA FETCHING

void Manager::Fetch(SIDMap *tofetch, GUID ignore)
{
    for (SIDMapIter it = tofetch->begin();
	 it != tofetch->end(); it++) {

	if (it->second == GetSID()) {
	    DB(1) << "fetching object " << it->first << " from self?? "
		  << "ignoring" << endl;
	    continue;
	}

	if (it->first == ignore) {
	    DBG << "unresolved refs contained ref to self, ignoring " 
		<< ignore << endl;
	    continue;
	}

	if (m_PendingFetchReqs.find(it->first) != m_PendingFetchReqs.end()) {
	    // already an outstanding req, skip (XXX retransmit when?)
	    continue;
	}
	m_PendingFetchReqs.insert(it->first);

	DB(5) << "fetching GUID=" << it->first << " from " << it->second << endl;

	// Send these async
	MsgAppFetchReq req = MsgAppFetchReq(it->first);
	m_DirectRouter->SendMsg(&(it->second), &req, 
				// XXX -- retransmits!
				PROTO_UDP);
    }
}

void Manager::HandleFetchReq(MsgAppFetchReq *req) {

    DBG << "handling FetchReq for: " << req->guid << endl;

    GObject *obj = m_Game->GetObjectStore()->Find(req->guid);

    if (obj == NULL || obj->IsReplica()) {
	// we don't own this object, see if we have a forwarding pointer
	FwdPtrMapIter p = m_ForwardingPtrs.find(req->guid);

	if (p != m_ForwardingPtrs.end()) {
	    DB(1) << "had forwarding pointer for " << req->guid 
		  << " which we believe now resides on " 
		  << *p->second << endl;

	    if (*p->second == req->GetSender()) {
		DB(1) << "forwarding pointer points back to sender;"
		      << " ignoring" << endl;
	    } else {
		// if so, forward the update to the new dude
		req->SetIsForwarded(true);
		m_DirectRouter->SendMsg(p->second, req,
					PROTO_TCP);
	    }
	} else if (m_DeletedGUIDs.find(req->guid) != m_DeletedGUIDs.end()) {
	    // send back a response saying it was deleted

	    DB(1) << "object " << req->guid 
		  << " was deleted -- sending fetch resp" << endl;

	    SID target = req->GetSender();
	    MsgAppFetchResp resp(req->guid, DELTA_MASK_NONE, NULL);
	    m_DirectRouter->SendMsg(&target, &resp);
	} else {

	    WARN << "no object for update for " << req->guid 
		 << ", no forwarding pointer, and no record that "
		 << "it was ever deleted... (dropping)" << endl;
	}
	return;
    }

    Packet *delta = new Packet(GUpdate::MAX_UPDATE_SIZE);
    obj->PackUpdate(delta, obj->GetInitDeltaMask());
    MsgAppFetchResp resp(req->guid, obj->GetInitDeltaMask(), delta);

    DBG << &resp << endl;

    // handle these asynchronously to speed up object resolution
    SID target = req->GetSender();
    m_DirectRouter->SendMsg(&target, &resp, PROTO_TCP);

}

void Manager::HandleFetchResp(MsgAppFetchResp *resp) {

    DBG << "handling FetchResp" << endl;

    SIDMap unresolved;
    GUID guid = resp->update->guid;

    if ( m_PendingFetchReqs.find(guid) ==
	 m_PendingFetchReqs.end() ) {
	// Wasn't looking for this... ignore it
	DB(1) << "Got FetchResponse for unknown GUID: "
	      << guid << endl;
	return;
    }

    // See if we already have a copy
    GObject *obj = FindObject(guid);

    if ( obj != NULL && !obj->IsReplica() ||
	 m_DeletedGUIDs.find(guid) != m_DeletedGUIDs.end() ) {
	// Got a fetch response to object we are primary for? Must be
	// race condition, ignore
	DB(1) << "Got a FetchResponse for GUID we are primary for: " 
	      << guid << endl;
	return;
    }

    // Jeff: avoid adding objects if we already have a copy...
    if (obj != NULL) {
	DB(1) << "Got a FetchResponse for GUID already have a replica for: " 
	      << guid << endl;
	return;
    }

    //ASSERT(obj->GetSID() != GetSID());

    // I dunno how this is causes bad things, but somehow it does, so ignore it
    // This should almost  never happen anyway
    if (m_MigrateIncoming.find(guid)   != m_MigrateIncoming.end() ) {
	DB(1) << "Got a FetchResponse for a GUID currently migrating to us:"
	      << guid << "; ignoring" << endl;
	return;
    }

    if (resp->update->isDelete) {
	//INFO << "got deletion fetch resp for: " << guid << " from "
	//	 << resp->sender << endl;

	// object was deleted
	obj = NULL;
    } else {
	if (g_MeasurementParams.enabled) {
	    ReplicaLifetimeEntry e;
	    e.guid = guid;
	    e.type = (char)ReplicaLifetimeEntry::FCREATE;
	    LOG(ReplicaLifetimeLog, e);
	}

	GObjectInfo *info = AllocGObjectInfo(guid, resp->GetSender(), true );

	ASSERT(resp->update->delta);
	obj = m_Game->Construct(info, resp->update->delta, 
				resp->update->mask, &unresolved);
	ASSERT(obj != NULL);
	m_Game->GetPendingStore()->_ManagerAdd(obj); // We'll process this later
    }

    m_ObjectArrivals[guid] = obj;

    if (obj != NULL) {
	Fetch(&unresolved, obj->GetGUID());
    }
}

///////////////////////////////////////////////////////////////////////////////
///// PRIMARY MIGRATION



void Manager::Migrate(GObject *obj, sid_t target)
{
    // Queue it, to be done on next call to Send()
    m_MigrateQueue.push_back(MigrationPending(obj->GetGUID(), target));
}

void Manager::PerformMigrations()
{
    for (MigrateQueueIter it = m_MigrateQueue.begin(); 
	 it != m_MigrateQueue.end(); it++) {
	GUID guid  = (*it).guid;
	SID target = (*it).target;

	GObject *obj = GetObjectStore()->Find(guid);
	// I suppose it is possible that the object was deleted or something
	if (!obj || obj->IsReplica())
	    continue;

	_PerformMigration(obj, target);
    }
    m_MigrateQueue.clear();
}

void Manager::_PerformMigration(GObject *obj, sid_t target)
{
    ASSERT(obj);
    ASSERT(!obj->IsReplica());
    ASSERT(obj->IsMigratable());

    DB(5) << "called: " << obj->GetGUID() << " to " << target << endl;
    DB(5) << "obj: " << obj << endl;

    GUID guid = obj->GetGUID();

    MigrationInfo info( obj->BeginMigration(target) );

    MsgMigrateReq req = MsgMigrateReq();
    req.update->guid  = obj->GetGUID();
    req.update->isNew = true;

    Packet *delta = new Packet(GUpdate::MAX_UPDATE_SIZE);
    obj->PackUpdate(delta, obj->GetInitDeltaMask());
    req.update->delta = delta;
    req.update->mask  = obj->GetInitDeltaMask();

    // the rest of this is system information

    req.eventSeqno = ((GObjectInfo *)obj->GetInfo())->LastEventSeqno();
    req.costEstimate = *((GObjectInfo *)obj->GetInfo())->GetCostEstimate();

    for (OutgoingReplicaConnectionMapIter it = m_ReplicaConnSenderMap.begin();
	 it != m_ReplicaConnSenderMap.end(); it++) {
	DeltaMap *deltas = it->second->GetDeltaMap();
	DeltaMapIter p = deltas->find(guid);

	// target doesn't need to be in reg. map anymore since it will
	// now be the primary owner	
	if ( p != deltas->end() ) {
	    if ( it->first != target )
		req.registrations[it->first] = p->second;
	    deltas->erase(p); // we can do this since we know we are committing
	}
    }

    info.from = GetSID();
    info.to   = target;
    info.guid = guid;

    if (g_MeasurementParams.enabled) {
	MigrationLogEntry e(guid, target);
	LOG(MigrationLog, e);
    }

    //m_MigrateOutgoing[obj->GetGUID()] = info;

    // XXX Eventually we need to fix this to use our own migration protocol
    // but for now it is too painful :P
    m_DirectRouter->SendMsg(&target, &req, PROTO_TCP);

    //m_MigrateOutgoing.erase(p);

    // ... since we made the migration reliable, we can just commit it now

    // find the entry in the object graph
    //GObjectInfoMapIter p3 = m_ObjectInfo.find(guid);
    //ASSERT(p3 != m_ObjectInfo.end());
    //ASSERT(p3->second == obj->GetInfo());
    GObjectInfo *oinfo = (GObjectInfo *)obj->GetInfo();
    // clear the registered replicas since we are no longer owner
    oinfo->GetRegisteredReplicas()->clear();
    // remove all the interest links since we are no longer owner
    oinfo->GarbageCollectInterests(true);
    // reset the event seqno (not really necessary)
    oinfo->SetEventSeqno( 0 );
    // clear the cost estimate (not really necessary)
    oinfo->GetCostEstimate()->Set(0, 0);
    // refresh the timer, 
    // so the replica isn't garbage collected immediately
    oinfo->Refresh();

    obj->CommitMigration(target);
    obj->SetIsReplica(true);
    obj->SetSID(target);

    // save a forwarding pointer to the new guy
    FwdPtrMapIter fpp = m_ForwardingPtrs.find(guid);
    if (fpp == m_ForwardingPtrs.end()) {
	m_ForwardingPtrs[guid] = AllocTimedSID(target,
					       FORWARDING_PTR_TTL);
    } else {
	m_ForwardingPtrs[guid]->Copy(target);
	m_ForwardingPtrs[guid]->Refresh();
    }

    if (g_MeasurementParams.enabled) {
	ReplicaLifetimeEntry e;
	e.guid = guid;
	e.type = (char)ReplicaLifetimeEntry::OBJINFO;
	LOG(ReplicaLifetimeLog, e);
	e.type = (char)ReplicaLifetimeEntry::MCREATE;
	LOG(ReplicaLifetimeLog, e);
    }
}

void Manager::HandleMigrateReq(MsgMigrateReq *req) 
{
    if (m_MigrateIncoming.find(req->update->guid) !=
	m_MigrateIncoming.end()) {
	WARN << "got a second migration for guid " << req->update->guid
	     << "; ignoring... ???" << endl;
	return;
    }

    GUID guid = req->update->guid;

    /*
      if (m_MigrateProcessing.find(req->ev.GetGUID()) !=
      m_MigrateProcessing.end()) {
      DB(1) << "got a migration req for guid " << req->ev.GetGUID()
      << "that we are already in the process if migrating; "
      << "ignoring... ???" << endl;
      return;
      }
    */

    DB(5) << "called" << endl;

    if (g_MeasurementParams.enabled) {
	ReplicaLifetimeEntry e;
	e.guid = guid;
	e.type = (char)ReplicaLifetimeEntry::MDESTROY;
	LOG(ReplicaLifetimeLog, e);
    }

    // XXX: FIXME: I don't think this is safe! We should do a "replace"
    // not an delete/add! I think this breaks a lot of stuff FIXME!!

    // if we have an old copy of this object in the store, force the app
    // to remove it. That way we ensure that we will use the new copy
    GObject *old = FindObject(guid);
    if (old != NULL) {
	ASSERT(old->IsReplica());
	m_Game->Destroy(guid);

	// Jeff: unneeded, will be deleted automatically by the destructor
	//m_ObjectInfo.erase(guid);
    }
    ASSERT(FindObject(guid) == NULL);
    // This is ok, we may have gotten a pub for it, in that case, just let
    // it be since we'll assign it later anyway.
    //ASSERT(m_ObjectInfo.find(guid) == m_ObjectInfo.end());

    SIDMap unresolved;

    GObjectInfo *info = AllocGObjectInfo(guid, req->GetSender(), true );
    info->SetEventSeqno( req->eventSeqno );
    info->GetCostEstimate()->Set( req->costEstimate );

    // construct the primary copy
    GObject *obj = m_Game->Construct(info,
				     req->update->delta, 
				     req->update->mask, 
				     &unresolved);

    ASSERT(obj != NULL);
    ASSERT(obj->IsReplica());
    ASSERT(obj->GetSID() != GetSID());

    /*
      GObject *old = m_Game->GetObjectStore()->Find(obj->GetGUID());
      if (old && !old->IsReplica()) {
      WARN << "Got a migration req for object we own! ignoring" << endl;
      return;
      }
    */
    GObjectInfoMapIter p = m_ObjectInfo.find(obj->GetGUID());
    ASSERT(p != m_ObjectInfo.end());
    ASSERT(p->second == obj->GetInfo());

    // We'll process this later
    m_Game->GetPendingStore()->_ManagerAdd(obj);

    m_ObjectArrivals[guid]  = obj;

    MigrationInfo minfo(0);
    minfo.guid           = obj->GetGUID();
    minfo.from           = req->GetSender();
    minfo.to             = GetSID();
    m_MigrateIncoming[guid] = minfo;
    m_MigrateIncoming[guid].registrations = 
	new SIDDeltaMap(req->registrations);

    DB(5) << "got: " << obj->GetGUID() << " from " << req->GetSender() << endl;
    DB(5) << "recv: " << obj << endl;

    Fetch(&unresolved, obj->GetGUID());
}

void Manager::HandleMigrateOk(guid_t guid) 
{
    MigrationInfoMapIter p = m_MigrateIncoming.find(guid);
    ASSERT(p != m_MigrateIncoming.end());

    DB(5) << "called: " << guid << endl;

    GObject *obj = m_Game->GetObjectStore()->Find(guid);
    ASSERT(obj != NULL);

    obj->CommitMigration(GetSID());
    obj->SetIsReplica(false);
    obj->SetSID(GetSID());

    // add to interest graph, if necessary -- should NEVER happen
    GObjectInfoMapIter infop = m_ObjectInfo.find(guid);
    ASSERT(infop != m_ObjectInfo.end());
    ASSERT(infop->second = (GObjectInfo *)obj->GetInfo());
    /*
      if (m_ObjectInfo.find(guid) == m_ObjectInfo.end()) {
      m_ObjectInfo[guid] = AllocGObjectInfo(guid);
      }
      m_ObjectInfo[guid]->SetSID( GetSID() );
    */

    // add registrations for those interested in this object
    for (SIDDeltaMapIter it = p->second.registrations->begin();
	 it != p->second.registrations->end(); it++) {
	if (it->first == GetSID()) {
	    WARN << "Migrated object had self-node registered; ignoring"
		 << endl;
	    continue;
	}
	// find the connection for it
	OutgoingReplicaConnectionMapIter cp =
	    m_ReplicaConnSenderMap.find(it->first);
	ReplicaConnectionSender *conn;
	if (cp == m_ReplicaConnSenderMap.end()) {
	    conn = new ReplicaConnectionSender(it->first, this);
	    m_ReplicaConnSenderMap[it->first] = conn;
	} else {
	    conn = cp->second;
	}
	// add the delta info
	(*conn->GetDeltaMap())[guid] = it->second;
    }

    m_MigrateIncoming.erase(p);
}

///////////////////////////////////////////////////////////////////////////////
///// MIGRATION HEURISTICS

void Manager::ProcessMigrateMap(MigrateMap& todo)
{
    //INFO << "map: " << todo << endl;

    for (MigrateMapIter i = todo.begin(); i != todo.end(); i++) {
	GUID guid = (*i).first;
	SID to = (*i).second;

	GObject *obj = m_Game->GetObjectStore()->Find(guid);
	ASSERT(obj != NULL);
	ASSERT(!obj->IsReplica());

	/*
	  if (m_MigrateOutgoing.find(guid) != m_MigrateOutgoing.end()) {
	  // OOPS! Already migrating, can't migrate again!

	  DB(1) << "Trying to migrate an object that is already "
	  << "being migrated! guid=" << guid << endl;

	  continue;
	  }
	*/

	Migrate(obj, to);
    }
}

void Manager::HandleMigrateConsider(TimeVal& now)
{
    DB(5) << "called" << endl;

    MigrateMap todo;

    m_MigrationPolicy->Migrate(todo, NULL);
    ProcessMigrateMap(todo);
}

void Manager::HandleMigrateConsiderFreq(TimeVal& now)
{
    DB(5) << "called" << endl;

    MigrateMap todo;

    m_MigrationPolicy->MigrateFreq(todo);
    ProcessMigrateMap(todo);
}

///////////////////////////////////////////////////////////////////////////////
///// LOAD PREDICTION

/*
  void Manager::HandleLoadPeriodic(TimeVal& now)
  {
  DBG << "called" << endl;

  m_LoadManager->DoWork();

  ////// Recalculating load value

  CostMetric objectCost = 0;
  ObjectStore *store = m_Game->GetObjectStore();
  GObject *obj;

  store->Begin();
  while ( (obj = store->Next()) != NULL) {
  if (!obj->IsReplica())
  objectCost += obj->GetFixedCost();
  objectCost += obj->GetDeltaCost();
  }

  // XXX TODO: Add in Cost from Mercury?

  CostMetric  inbound = m_DirectRouter->GetInboundUsage(now);
  CostMetric outbound = m_DirectRouter->GetOutboundUsage(now);

  DB(5) << " inbound=" << inbound
  << " outbound=" << outbound
  << " objcosts=" << objectCost << endl;

  CostMetric new_load_val = inbound + outbound + objectCost;

  m_Load = 
  LOAD_EMA_WEIGHT*new_load_val + (1-LOAD_EMA_WEIGHT)*new_load_val;

  //////////

  CostMetric load = m_Load;

  DB(5) << "=========" << endl;
  DB(5) << "curr_load=" << load << " target=" << m_LoadTarget << endl;

  DB(5) << "curr_hw=" << GetHighWaterMark() << endl;
  DB(5) << "curr_lw=" << GetLowWaterMark() << endl;

  // dynamically recalculate water marks
  if (m_NodeLoadMap.size() > 0) {
  CostMetric sysAvgLoad = 0;
  CostMetric sysStdLoad = 0;

  for (NodeLoadMapIter it = m_NodeLoadMap.begin(); 
  it != m_NodeLoadMap.end(); it++) {

  DB(5) << "load: " << it->first << "=" 
  << it->second->load  << endl;
  sysAvgLoad += it->second->load;
  }
  sysAvgLoad += load;
  // XXX don't weight this so much if size() is small!
  sysAvgLoad /= (m_NodeLoadMap.size()+1);

  for (NodeLoadMapIter it = m_NodeLoadMap.begin(); 
  it != m_NodeLoadMap.end(); it++) {
  CostMetric d = 
  (sysAvgLoad - it->second->load);
  sysStdLoad += d*d;
  }
  CostMetric my_d = sysAvgLoad - load;
  sysStdLoad += my_d*my_d;
  sysStdLoad = 
  sqrt(sysStdLoad / (m_NodeLoadMap.size()+1));

  DB(5) << "sys_avg_load="  << sysAvgLoad
  << " sys_std_load=" << sysStdLoad << endl;

  m_WindowTarget = 
  LOAD_EMA_WEIGHT*m_WindowTarget + (1-LOAD_EMA_WEIGHT)*4*sysStdLoad;
  m_WindowTarget = MAX(LOAD_MIN_WINDOW, m_WindowTarget);

  m_LoadTarget = 
  LOAD_EMA_WEIGHT*m_LoadTarget + (1-LOAD_EMA_WEIGHT)*sysAvgLoad;
  m_LoadTarget = MAX(m_LoadTarget, m_WindowTarget/2);
  m_LoadTarget = MIN(m_LoadTarget, 
  LOAD_MAX_HIGH_WATER_MARK-m_WindowTarget/2);

  m_HighWaterMark = m_LoadTarget + m_WindowTarget/2;
  m_LowWaterMark = m_LoadTarget - m_WindowTarget/2;

  DB(5) << "new_target=" << m_LoadTarget << endl;
  DB(5) << "new_window=" << m_WindowTarget << endl;
  DB(5) << "new_hw=" << GetHighWaterMark() << endl;
  DB(5) << "new_lw=" << GetLowWaterMark() << endl;
  }

  #if 0
  Event *evt = new Event(GUID_NONE, GetSID());
  evt->AddTuple(ATTR_TO_LOW_WATER_MARK, 
  Value((float)(GetLowWaterMark()-load)));
  evt->AddTuple(ATTR_TO_HIGH_WATER_MARK, 
  Value((float)(GetHighWaterMark()-load)));
  evt->AddTuple(ATTR_LOAD, 
  Value((float)(load)));

  m_PubSubRouter->SendEvent(evt);

  // subscribe to load info
  //if (load > GetLowWaterMark()) {
  //	DBG << "I am above low water " << load << " > "
  //		<< GetLowWaterMark() << " so I am registering interest to "
  //		<< "everyone with lower load than me" << endl;

  // XXX For now subscribe to load from everyone
  OMInterest *in = new OMInterest();
  in->SetLifeTime( (uint32)(PUBLISH_LOAD_INFO_TIMER * 1.5) );
  in->AddConstraint( ATTR_TO_HIGH_WATER_MARK, OP_ANY,
  Value((float)0) );
  //in->AddConstraint( ATTR_TO_HIGH_WATER_MARK, OP_LESS_THAN,
  //				   Value(GetHighWaterMark()-load) );
  m_PubSubRouter->RegisterInterest(in);
  //}
  #endif
  }
*/

void Manager::RecordLoadInfo(MsgApp *msg)
{
    if ( msg->HasLoadInfo() ) {
	//DB(1) << "called " << msg->GetSender() << endl;

	SID sid = msg->GetSender();

	DB(5) << "recording new load info from msg for node " << sid 
	      << ": toLowWaterMark=" << msg->GetToLowWaterMark()
	      << " toHighWaterMark=" << msg->GetToHighWaterMark() 
	      << " load=" << msg->GetLoad() << endl;

	NodeLoadInfo info( sid,
			   msg->GetToLowWaterMark(),
			   msg->GetToHighWaterMark(),
			   msg->GetLoad() );
	// we know this info is fresh!
	info.Refresh();

	m_LoadManager->Record(info);
    }
}

CostMetric Manager::GetLoad(TimeVal& now)
{
    return m_LoadManager->GetLoad();
}

CostMetric Manager::GetLoadTarget()
{
    return m_LoadManager->GetLoadTarget();
}

CostMetric Manager::GetHighWaterMark()
{
    return m_LoadManager->GetHighWaterMark();
}

CostMetric Manager::GetLowWaterMark()
{
    return m_LoadManager->GetLowWaterMark();
}

CostMetric Manager::GetCapacity()
{
    return m_LoadManager->GetCapacity();
}

///////////////////////////////////////////////////////////////////////////////
///// DEBUGGING

static void FillBuf(char **bufp, char *fmt, ...)
{
    va_list     args;	
    va_start(args, fmt);
    *bufp += vsprintf(*bufp, fmt, args);
    va_end(args);
}

const char *Manager::DumpInterestGraphToGraphViz()
{
    char *bufPos = m_OutputBuffer;
    bufPos[0] = '\0';
    GUIDVec missing, reps, prims;
    map<GUID, float, less_GUID> time_map;

    TimeVal now;
    OS::GetCurrentTime(&now);

    FillBuf(&bufPos, "digraph G {\n{overlap=scale}\n");

    for (GObjectInfoMapIter it = m_ObjectInfo.begin(); 
	 it != m_ObjectInfo.end(); it++) {
	GObject *obj = m_Game->GetObjectStore()->Find(it->first);
	if (obj == NULL) {
	    // rep we don't have
	    missing.push_back(it->first);
	} else if (obj->IsReplica()) {
	    // rep we have
	    reps.push_back(it->first);
	} else {
	    // primary we own
	    prims.push_back(it->first);
	}
	float left = (float)it->second->TimeLeft()/MSEC_IN_SEC;
	time_map[it->first] = left;
    }

    if (missing.size() > 0) {
	FillBuf(&bufPos, "{node [shape=octagon]");
	for (int i=0; i<(int)missing.size(); i++) {
	    FillBuf(&bufPos, " \"%s (%.1f)\"", missing[i].ToString(),
		    time_map[missing[i]]);
	}
	FillBuf(&bufPos, "}\n");
    }
    if (reps.size() > 0) {
	FillBuf(&bufPos, "{node [shape=octagon,style=filled,color=cyan]");
	for (int i=0; i<(int)reps.size(); i++) {
	    FillBuf(&bufPos, " \"%s (%.1f)\"", reps[i].ToString(),
		    time_map[reps[i]]);
	}
	FillBuf(&bufPos, "}\n");
    }
    if (prims.size() > 0) {
	FillBuf(&bufPos, "{node [shape=box,style=filled]");
	for (int i=0; i<(int)prims.size(); i++) {
	    FillBuf(&bufPos, " \"%s (%.1f)\"", prims[i].ToString(),
		    time_map[prims[i]]);
	}
	FillBuf(&bufPos, "}\n");
    }

    FillBuf(&bufPos, "edge [len=3]\n");

    for (GObjectInfoMapIter it = m_ObjectInfo.begin(); 
	 it != m_ObjectInfo.end(); it++) {
	InterestLinkMap *imap = it->second->GetInterests();
	for (InterestLinkMapIter it2 = imap->begin(); 
	     it2 != imap->end(); it2++) {
	    GObjectInfo *s = it2->second->GetSource();
	    GObjectInfo *t = it2->second->GetTarget();
	    char strbuf1[128], strbuf2[128];
	    strncpy(strbuf1, s->GetGUID().ToString(), 128);
	    strncpy(strbuf2, t->GetGUID().ToString(), 128);
	    float weight = (float)it2->second->LifeTime()/MSEC_IN_SEC;
	    FillBuf(&bufPos, 
		    "\"%s (%.1f)\" -> \"%s (%.1f)\" [label=\"%.1f\"];\n",
		    strbuf1, time_map[s->GetGUID()],
		    strbuf2, time_map[t->GetGUID()], weight);
	}
    }
    FillBuf(&bufPos, "}\n");

    return m_OutputBuffer;
}

const char *Manager::DumpInterestGraph()
{
    char *bufPos = m_OutputBuffer;
    bufPos[0] = '\0';

    for (GObjectInfoMapIter it = m_ObjectInfo.begin(); 
	 it != m_ObjectInfo.end(); it++) {
	InterestLinkMap *imap = it->second->GetInterests();
	for (InterestLinkMapIter it2 = imap->begin(); 
	     it2 != imap->end(); it2++) {
	    GObjectInfo *s = it2->second->GetSource();
	    GObjectInfo *t = it2->second->GetTarget();
	    char strbuf1[128], strbuf2[128];
	    strncpy(strbuf1, s->GetGUID().ToString(), 128);
	    strncpy(strbuf2, t->GetGUID().ToString(), 128);
	    uint32 weight = it2->second->LifeTime();
	    FillBuf(&bufPos, "%s,%s,%lu\n",
		    strbuf1, strbuf2, weight);
	}
    }

    return m_OutputBuffer;
}

const char *Manager::DumpObjectStore()
{
    char *bufPos = m_OutputBuffer;
    bufPos[0] = '\0';

    m_Game->GetObjectStore()->Begin();
    GObject *obj;
    while ( (obj = m_Game->GetObjectStore()->Next()) != NULL ) {
	FillBuf( &bufPos, "%s sid=%s replica=%d\n", 
		 obj->GetGUID().ToString(), 
		 obj->GetSID().ToString(),
		 obj->IsReplica() );
    }

    return m_OutputBuffer;
}

const char *Manager::DumpObjectInfo()
{
    char *bufPos = m_OutputBuffer;
    bufPos[0] = '\0';

    for (GObjectInfoMapIter it = m_ObjectInfo.begin(); 
	 it != m_ObjectInfo.end(); it++) {
	stringstream sbuf;
	sbuf << it->second;
	FillBuf(&bufPos, "%s\n", sbuf.str().c_str());
    }

    return m_OutputBuffer;
}

const char *Manager::DumpNodeLoadMap()
{
    if (ENABLE_LOAD_INFO) {
	NodeLoadMap *nmap = m_LoadManager->GetLoadMap();

	char *bufPos = m_OutputBuffer;
	bufPos[0] = '\0';

	for (NodeLoadMapIter it = nmap->begin(); 
	     it != nmap->end(); it++) {
	    FillBuf( &bufPos, "%s,%f,%f,%f,%lu\n",
		     it->first.ToString(),
		     it->second->load,
		     it->second->toLowWaterMark,
		     it->second->toHighWaterMark,
		     it->second->TimeLeft() );
	}

	return m_OutputBuffer;
    } else {
	return "";
    }
}

const char *Manager::DumpForwardingPtrs()
{
    char *bufPos = m_OutputBuffer;
    bufPos[0] = '\0';

    for (FwdPtrMapIter it = m_ForwardingPtrs.begin(); 
	 it != m_ForwardingPtrs.end(); it++) {
	FillBuf( &bufPos, "%s,%s,%lu\n",
		 it->first.ToString(), 
		 it->second->ToString(),
		 it->second->TimeLeft() );
    }

    return m_OutputBuffer;
}

const char *Manager::DumpDeletedGUIDs()
{
    char *bufPos = m_OutputBuffer;
    bufPos[0] = '\0';

    for (DeletedSetIter it = m_DeletedGUIDs.begin(); 
	 it != m_DeletedGUIDs.end(); it++) {
	FillBuf( &bufPos, "%s,%lu\n",
		 it->first.ToString(), 
		 it->second->TimeLeft() );
    }

    return m_OutputBuffer;
}

const char *Manager::DumpReplicaConnections()
{
    char *bufPos = m_OutputBuffer;
    bufPos[0] = '\0';

    FillBuf(&bufPos, "Outgoing Connections:\n");

    for (OutgoingReplicaConnectionMapIter it = m_ReplicaConnSenderMap.begin(); 
	 it != m_ReplicaConnSenderMap.end(); it++) {
	stringstream sbuf;
	sbuf << it->second;
	FillBuf(&bufPos, "%s\n", sbuf.str().c_str());
    }

    FillBuf(&bufPos, "Incoming Connections:\n");

    for (IncomingReplicaConnectionMapIter it = 
	     m_ReplicaConnReceiverMap.begin(); 
	 it != m_ReplicaConnReceiverMap.end(); it++) {
	stringstream sbuf;
	sbuf << it->second;
	FillBuf(&bufPos, "%s\n", sbuf.str().c_str());
    }

    return m_OutputBuffer;
}

///////////////////////////////////////////////////////////////////////////////

void Manager::CheckAlwaysInvariants()
{
    // true before or after entering Manager
    ObjectStore *store = m_Game->GetObjectStore();
    GObject *obj, *obj2;

    store->Begin();
    while( (obj = store->Next()) != NULL ) {
	if ( !obj->IsReplica() ) {
	    ASSERT( obj->GetSID() == GetSID() );
	    ASSERT( m_ObjectInfo.find(obj->GetGUID()) != 
		    m_ObjectInfo.end() );
	} else {
	    ASSERT( obj->GetSID() != GetSID() );
	    GObjectInfoMapIter p = m_ObjectInfo.find(obj->GetGUID());
	    ASSERT(p != m_ObjectInfo.end());
	}
    }

    ObjectStore *pending = m_Game->GetPendingStore();
    pending->Begin();
    while ( (obj = pending->Next()) != NULL ) {
	ASSERT( obj->IsReplica() );
	ASSERT( obj->GetSID() != GetSID() );

	GObjectInfoMapIter p = m_ObjectInfo.find(obj->GetGUID());
	if (p != m_ObjectInfo.end()) {
	    ASSERT(p->second->GetSID() != GetSID());
	}

	ASSERT(m_DeletedGUIDs.find(obj->GetGUID()) ==
	       m_DeletedGUIDs.end());
    }

    pending->Begin();
    while ( (obj = pending->Next()) != NULL ) {
	store->Begin();
	while( (obj2 = store->Next()) != NULL ) {
	    ASSERT(obj != obj2);
	}
    }

    for (GObjectInfoMapIter it = m_ObjectInfo.begin(); 
	 it != m_ObjectInfo.end(); it++) {
	GObjectInfo *info = it->second;
	GObject *obj = store->Find(it->first);

	if (info->GetSID() == GetSID()) {
	    ASSERT(obj);
	    ASSERT(! obj->IsReplica() );
	}

	ASSERT(info->GetSID() != SID_NONE);

	if (obj) {
	    ASSERT(obj->GetInfo() == info);
	}

	if (obj && !obj->IsReplica()) {
	    ASSERT(info->GetSID() == GetSID());

	    ReplicaPtrMap *ptrs = info->GetRegisteredReplicas();
	    for (ReplicaPtrMapIter pit = ptrs->begin();
		 pit != ptrs->end(); pit++) {
		ASSERT(pit->first != GetSID());
	    }

	} else {
	    ASSERT(info->GetRegisteredReplicas()->size() == 0);
	    ASSERT(info->GetInterests()->size() == 0);
	}
    }

    for (OutgoingReplicaConnectionMapIter it = m_ReplicaConnSenderMap.begin();
	 it != m_ReplicaConnSenderMap.end(); it++) {
	SID target = it->first;
	ASSERT(target != GetSID());
	DeltaMap *dmap = it->second->GetDeltaMap();
	for (DeltaMapIter it = dmap->begin(); it != dmap->end(); it++) {
	    GObject *obj = store->Find(it->first);
	    ASSERT(obj || m_DeletedGUIDs.find(it->first) != m_DeletedGUIDs.end());
	    if (obj) {
		ASSERT(!obj->IsReplica());
		ASSERT(obj->GetSID() == GetSID());
	    }
	    obj = pending->Find(it->first);
	    ASSERT(obj == NULL);
	}
    }

    m_Game->CheckInvariants();
}

void Manager::CheckEntryInvariants()
{
    // true before entering manager
    CheckAlwaysInvariants();
}

void Manager::CheckExitInvariants()
{
    // true after entering manager
    CheckAlwaysInvariants();
}

///////////////////////////////////////////////////////////////////////////////

#ifdef ENABLE_LB_TEST

#include <test/om/TestObject.h>
#include <test/om/TestAdaptor.h>

// for LBTest
class _LBTestObject : public GObject {
private:

    GUID m_GUID;
    SID  m_SID;
    bool m_Replica;

public:
    _LBTestObject(GUID id) : Migratable(true) { m_GUID = id; DeltaCost = 1; }
    _LBTestObject(Event * pub, SIDMap *unresolved);
    virtual ~_LBTestObject() {}

    void SetIsReplica(bool val) { m_Replica = val; }
    void SetIsMigratable(bool val) { Migratable = val; }

    ////////// Implements GObject //////////

    const guid_t GetGUID() { return m_GUID; }
    const sid_t GetSID() { return m_SID; }
    void SetSID(sid_t sid) { m_SID = sid; }
    bool IsReplica() { return m_Replica; }
    bool IsDirty() { return true; }
    bool IsNew() { return false; }

    void FillPub(Event *ev) { ASSERT(false); }
    void FillEvent(Event *ev) { ASSERT(false); }
    void FillFetchResp(Event *resp) { ASSERT(false); }
    void FillInterest(Interest *in) { ASSERT(false); }
    void HandleEvent(Event *ev, SIDMap *unresolved) { ASSERT(false); }

    bool Migratable;

    bool IsMigratable() { return Migratable; }
    uint32 BeginMigration(sid_t target, Event *marshall) { 
	ASSERT(false); return 0; 
    }
    void CancelMigration(sid_t target) { ASSERT(false); }
    void CommitMigration(sid_t target) { ASSERT(false); }

    CostMetric DeltaCost;

    CostMetric GetFixedCost()  { return 1; }
    CostMetric GetDeltaCost()  { return DeltaCost; }
    CostMetric GetUpdateCost() { return 0; }
};

void Manager::DoLoadBalancedPolicyTest(int argc, char **argv)
{
    ENABLE_AUTO_MIGRATE = true;
    TestAdaptor *apt = new TestAdaptor();
    Manager *mgr = new Manager(apt, false);

    if ( ! mgr->LoadBalancedPolicyTest(argc, argv) ) {
	WARN << "test failed!" << endl;
    } else {
	INFO << "test passed!" << endl;
    }
}
void Manager::MakeStableLink(GObjectInfo *from, GObjectInfo *to)
{
    from->RefreshInterest(to);
    InterestLink *link = (*from->GetInterests() )[to->GetGUID()];
    ASSERT(link != NULL);

    link->SetLifeTime(INTEREST_STABILIZATION_TIME + 10);
    ASSERT(link->IsStable());
}

#define SET_OBJ(index, sid) { \
    info[index]->SetSID(sid); \
	obj[index]->SetSID(sid); \
	obj[index]->SetIsReplica(sid != GetSID()); \
	m_Game->GetObjectStore()->_ManagerAdd(obj[index]); \
	m_ObjectInfo[guid[index]] = info[index]; \
}

#define SET_LOAD(index, ld) { \
    load[index]->toLowWaterMark = LOW_WATER_MARK - ld; \
	load[index]->toHighWaterMark = HIGH_WATER_MARK - ld; \
	m_NodeLoadMap[sid[index]] = load[index]; \
}

bool Manager::LoadBalancedPolicyTest(int argc, char **argv)
{
    // Dummy variables

    GUID guid[1024];
    for (int i=0; i<1024; i++) { guid[i].m_LocalOID = i+1; }
    SID sid[1024];
    for (int i=0; i<1024; i++) { sid[i].m_Port = i+1; }
    GObjectInfo *info[1024];
    for (int i=0; i<1024; i++) { info[i] = AllocGObjectInfo(guid[i]); }
    _LBTestObject *obj[1024];
    for (int i=0; i<1024; i++) { obj[i] = new _LBTestObject(guid[i]); }
    NodeLoadInfo *load[1024];
    for (int i=0; i<1024; i++) { load[i] = AllocNodeLoadInfo(sid[i]); }

    SID me = GetSID();
    m_UseDummyLoad = true;

    // Set up the test with dummy state in the manager

    SET_OBJ(0, me);

    SET_OBJ(1, sid[1]);
    //SET_OBJ(2, sid[1]);

    SET_OBJ(3, sid[3]);
    SET_OBJ(4, sid[3]);

    SET_OBJ(5, me);
    SET_OBJ(6, me);
    SET_OBJ(7, me);

    MakeStableLink(info[5], info[6]); //obj[5]->DeltaCost = HIGH_WATER_MARK;
    MakeStableLink(info[6], info[5]);
    MakeStableLink(info[0], info[7]);

    MakeStableLink(info[0], info[1]);
    MakeStableLink(info[0], info[2]);
    MakeStableLink(info[0], info[3]);
    MakeStableLink(info[0], info[4]);

    LOW_WATER_MARK  = 5;
    HIGH_WATER_MARK = 10;

    m_DummyLoad = 20;
    SET_LOAD(1, 4);
    SET_LOAD(2, 4);

    // Run the test

    const char *graph = DumpInterestGraph();
    DBG << "Interests:\n" << graph;
    const char *loadmap = DumpNodeLoadMap();
    DBG << "LoadMap:\n" << loadmap;

    MigrateMap todo;

    m_MigrationPolicy->Migrate(todo, NULL);

    for (MigrateMapIter i = todo.begin(); i != todo.end(); i++) {
	GUID guid = (*i).first;
	SID to = (*i).second;
	INFO << guid << " -> " << to << endl;
    }

    return true;
}

#endif
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
