////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************
  GameAdaptor.h

  An Adaptor interface for the Manager to interact with the game state.

begin           : Nov 6, 2002
copyright       : (C) 2002-2003 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2002-2003 Justin Weisz       ( jweisz@cs.cmu.edu )

***************************************************************************/

#ifndef __GAME_ADAPTOR_H__
#define __GAME_ADAPTOR_H__

#include <mercury/ID.h>
#include <om/DeltaMask.h>
#include <om/GObjectInfo.h>
#include <om/Terminal.h>

    class GObject;
class Event;

/**
 * Interface to one of the game's object stores. Used by the Manager for
 * the object store (for all active objects), and the "pending" store,
 * used for storage of replica objects that do not yet have all their
 * references ("pointers") resolved.
 *
 * Jeff New:
 * For our sanity, maintain the invariant that a particular GUID is not
 * in both the pending and object store at the same time. In addition,
 * if we already have a copy of an object GUID, we CAN NOT add it *either*
 * the object store or the pending store without explicitly removing the
 * old copy first.
 *
 * XXX TODO: We should fix this in the future so that the application doesn't
 * implement this, but just uses this as an interface that the Manager
 * provides... There is certainly no need for the application to know about
 * the pending store. XXX: However, we should still leave this skeleton in
 * place, because the application probably has a better idea about optimal
 * memory allocation for objects than us.
 */
class ObjectStore {
 private:

    friend class Manager;

 public:

    ObjectStore() {}
    virtual ~ObjectStore() {}

    /**
     * Set the object store iterator positon to the start
     */
    virtual void Begin() = 0;

    /**
     * Fetch the next object in the object store iterator. NULL if none.
     */
    virtual GObject *Next() = 0;

    /**
     * Find the object in the object store identified by guid. NULL id none
     */
    virtual GObject *Find(guid_t guid) = 0;

    /**
     * Add the object to the object store, possibly replacing an old version
     * with the same guid. The logic in this function should dictate 
     * whether the replacement should take place (e.g., based on version# of
     * the object or something).
     */
    virtual void _ManagerAdd(GObject *obj) = 0;

    /**
     * Remove an object from the object store. This method is expected only
     * to be used by the Manager and not the application. When the manager
     * calls "Remove" it should only be removed from the store, NOT
     * garbage collected, because we may use this method to move an object
     * from the pending to the object store.
     *
     * XXX Calling this method must have strict constraints, because we do
     * not know exactly what other objects may have pointers to this object.
     * For now, I am assuming that this only gets called on the PendingStore.
     */
    virtual void _ManagerRemove(guid_t guid) = 0;

    /**
     * When the application adds an object to the store, it should use
     * this function. It can assume that this is just a wrapper
     * around _ApplicationAdd(GObject *obj).
     *
     * We need a separate method because the Manager must be notified
     * when an addition occurs. DO NOT OVERRIDE THIS METHOD!
     */
    void ApplicationAdd(GObject *obj);

    /**
     * When the application deleted an object from the store, it should
     * use this function. It can assume that this is just a wrapper
     * around _ApplicationDelete(guid_t guid).
     *
     * We need a separate method because the Manager must be notified
     * when a deletion occurs. DO NOT OVERRIDE THIS METHOD!
     *
     * You may NOT remove objects that do not exist.
     * You should NOT use this to remove replicas.
     * You should NOT use this within a callback (such as 
     * GameAdaptor::Destroy()); only use it within the application code
     * proper.
     */
    void ApplicationRemove(guid_t guid);

 protected:

    /**
     * The application should implement this method as to how it adds
     * objects to the object store. However, it should access this
     * method via ApplicationAdd(...), IT SHOULD NOT call this directly.
     */
    virtual void _ApplicationAdd(GObject *obj) = 0;

    /**
     * The application should implement this method as to how it removes
     * objects from the object store. However, it should access this
     * method via ApplicationDelete(...), IT SHOULD NOT call this directly.
     */
    virtual void _ApplicationRemove(guid_t guid) = 0;
};


/**
 * Adaptor class for the Manager to interact with the game. Maintains
 * callbacks that the Manager within Manager::Send() and Manager::Recv().
 */
class GameAdaptor {

 public:

    GameAdaptor() {}
    virtual ~GameAdaptor() {}

    /**
     * This should return how often the game is going to call the
     * Send() functions of the Manager. This allows the
     * Manager to predict how long it has to set the subscription 
     * soft-state timer so that subs do not expire before they are 
     * refreshed by a call to Send(). It is better to overestimate
     * than underestimate. However, overestimating too much will
     * mean we get lots of pubs for stale subs.
     */
    virtual uint32 GetSendInterval() = 0;

    /**
     * Retreive a reference to the game's object store, containing all
     * usable primaries and replicas. The invariant we maintain is that
     * all pointers in objects in the object store must point to other
     * objects in the object store.
     */
    virtual ObjectStore *GetObjectStore() = 0;

    /**
     * Retreive a reference to the pending store, containing all 
     * "half-complete" objects. An object in the pending store may have
     * pointers to objects in both the object and pending stores, or
     * pointers to objects that have not been resolved.
     */
    virtual ObjectStore *GetPendingStore() = 0;

    /**
     * Modify the interests of the application. The application is
     * responsible for setting approprite TTLs in the interests, etc.
     *
     * The GUID in the Interest should be unique among interests
     * currently held.
     *
     * @param reg the new interests to register
     * @param unreg the interests in curr to unregister
     * @param reg_assoc the change in mapping between object GUID -> interest
     *        GUID. If a mapping remains the same since the previous call, then
     *        there is no need to place it in this map.
     * @param curr the current mapping of interest GUID to interests.
     * 
     */
    virtual void FillInterests(InterestList *reg,
			       InterestList *unreg,
			       GUIDMap *reg_assoc,
			       InterestMap *curr) = 0;

    /**
     * Construct a new replica GObject from a "constructable" network
     * event, like a publication or a fetch response. If there are
     * unresolved references, the adaptor should set the appropriate
     * set in the returned GObject and in the unresolved SIDMap. The
     * constructed object is always a replica.
     *
     * @param info the Manager level meta info about the object.
     * @param pkt the serialized object (serilized with PackUpdate()
     *        using a full delta mask)
     * @param mask the "full" delta mask that tells what to construct.
     * @param unresolved a list of unresolved references that need to
     *        be fetched before the object can be completed.
     *
     * An unresolved reference is defined as an pointer to an object that is 
     * neither in our object store, nor in our pending store, and hence
     * needs to be fetched.
     */
    virtual GObject *Construct(GObjectInfoIface *info,
			       Packet *pkt, 
			       const DeltaMask& mask,
			       SIDMap *unresolved) = 0;

    /**
     * Destroy a replica because it was deleted by the primary node and
     * does not exist any longer in this system. The application should
     * remove the object from BOTH the pending and object stores (it will
     * be in exactly one) AND it should DELETE the object in question.
     * Do NOT use the ApplicationRemove() function to do the removal from
     * the object store; that should only be used during the execution of
     * the application on primary objects.
     *
     * @param guid the GUID of the object to destroy.
     */
    virtual void Destroy(GUID guid) = 0;

    ///////////////////////////////////////////////////////////////////////////
    ///// DEBUGING

    /**
     * Called within Manager each time we check invariants. Throw an assertion
     * if some game internal invariants fail. See the 
     * CHECK_INVARIANTS_DBG_LEVEL flag in Manager.h.
     */
    virtual void CheckInvariants() {}

    /**
     * The application may install some handler callbacks that are used
     * by the remote execution terminal to inspect or change in-game
     * state. This is called once when the Manager is first constructed.
     *
     * @param tofill a vector in which the adaptor should place any
     *        handlers it wants to install.
     */
    virtual void InstallTerminalHandlers(InputHandlers *tofill) {}

};

#endif
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
