////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************
  GObjectInfo.h

begin           : Nov 6, 2002
copyright       : (C) 2002-2004 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2003-2004 Jeff Pang      ( jeffpang@cs.cmu.edu   )

***************************************************************************/

#ifndef __GOBJECT_INFO_H__
#define __GOBJECT_INFO_H__

#include <list>
#include <util/TimedStruct.h>
#include <util/DataChunker.h>
#include <om/common.h>
#include <om/OMInterest.h>
#include <om/OMEvent.h>
#include <om/ManagerParams.h>

    class GObject;

using namespace ManagerParams;

///////////////////////////////////////////////////////////////////////////////
///// Cost Estimate (persistent estimate of an object's delta cost)

class CostEstimate : public Serializable {
    // we will get an update about once every .1 or .05 seconds
    // when we are updating a remote host
    // we want an estimate that is relatively long running (on the
    // order of 15-30 seconds, so weight new updates accordingly)
    static const float EMA_ALPHA = 0.99; // how much to weight history
    uint32     m_NumUpdates;
    CostMetric m_Estimate;
 public:
    CostEstimate() : m_NumUpdates(0),
	m_Estimate(0) {}
    CostEstimate(const CostEstimate& o) : 
	m_NumUpdates(o.m_NumUpdates),
	m_Estimate(o.m_Estimate) {}

    CostMetric GetEstimate() const;
    uint32     GetNumUpdates() const;
    void Set(CostMetric init, uint32 num_updates);
    void Set(const CostEstimate& old);
    void Update(CostMetric update);

    CostEstimate(Packet *);
    void Serialize(Packet *);
    uint32 GetLength();
    void Print(FILE *) {}
};

///////////////////////////////////////////////////////////////////////////////
///// Remote Replica Info

/**
 * Soft-state record that records how many objects a remote node is
 * interested in and how long it has been interested. Stored in GObjectInfo.
 */
class ReplicaPtr : public TimedStruct {
    // How many primaries have stable interests in this
    uint32 m_NumInterests;
 public:
    ReplicaPtr(TimeVal now = TIME_NONE) : 
	m_NumInterests(0),
	TimedStruct(REPLICA_MAINTAIN_TTL, now) {}

    inline void Refresh(uint32 numInterests)
	{
	    TimedStruct::Refresh(GetTTL());
	    m_NumInterests = numInterests;
	}

    inline void SetNumInterests(uint32 num) { m_NumInterests = num; }
    inline uint32 GetNumInterests() { return m_NumInterests; }
};

typedef map<SID, ReplicaPtr, less_SID> ReplicaPtrMap;
typedef ReplicaPtrMap::iterator ReplicaPtrMapIter;

ostream& operator<<(ostream& out, ReplicaPtr& ptr);


///////////////////////////////////////////////////////////////////////////////
///// InterestLinks -- directed edges in the interest graph

class GObjectInfo;

/**
 * An InterestLink from (source => target) means that source is interested
 * in target. A link is not stable until its Lifetime() passes some threshold.
 */
class InterestLink : public TimedStruct {
    GObjectInfo *m_Source; // object that maintains this interest
    GObjectInfo *m_Target; // object that we are interested in
 public:

    InterestLink() : TimedStruct(INTEREST_LINK_TTL) { }
    virtual ~InterestLink() { }

    inline GObjectInfo *GetSource() { return m_Source; }
    inline GObjectInfo *GetTarget() { return m_Target; }
    inline void SetSource(GObjectInfo *i) { m_Source = i; }
    inline void SetTarget(GObjectInfo *i) { m_Target = i; }
    inline bool IsStable() { 
	return LifeTime() >= INTEREST_STABILIZATION_TIME; 
    }
};

typedef set<InterestLink *> InterestLinkSet;
typedef InterestLinkSet::iterator InterestLinkSetIter;
typedef map<GUID, InterestLink *, less_GUID> InterestLinkMap;
typedef InterestLinkMap::iterator InterestLinkMapIter;

ostream& operator<<(ostream& out, InterestLink *link);


///////////////////////////////////////////////////////////////////////////////
///// GObjectInfo (Interest Management) Data Structures

/**
 * An interface for the application. This encapsulates all the Manager-level
 * information of the object that needs to be exported to the application.
 */
class GObjectInfoIface {
 protected:
    GUID            m_GUID;       // guid of the object (redundant, but useful)
    SID             m_SID;        // location of primary
    bool            m_IsReplica;  // true if not primary
 public:
    inline GUID GetGUID() { return m_GUID; }
    inline SID  GetSID() { return m_SID; }
    inline void SetGUID(GUID guid) { 
	ASSERT(guid != GUID_NONE);
	m_GUID = guid; 
    }
    inline void SetSID(SID sid) { 
	ASSERT(sid != SID_NONE);

	m_SID = sid; 
    }
    inline bool IsReplica() { return m_IsReplica; }
    inline void SetIsReplica(bool val) { m_IsReplica = val; }
};

/**
 * A data structure used to store extra information about GObject objects
 * that can not be saved in GObjects. We may have GObjectInfo structures for
 * objects that we do not have (yet); this is because the info structure
 * tracks interests in particular objects but treats the contents as
 * opaque. Managed by the Manager.
 */
class GObjectInfo : public GObjectInfoIface, public TimedStruct {

    friend class Manager;
    friend class GObject;
    friend ostream& operator<<(ostream& out, GObjectInfo *info);

    static ClassChunker<InterestLink> *m_LinkAllocator;

    ///////// INTEREST GRAPH MANAGEMENT

    InterestLinkMap m_Interests;  // state of this object's interests
    InterestLinkSet m_BackPtrs;   // local interest links that point to us
    Manager *m_Manager;           // manager that allocated me
    GObject *m_Object;            // maybe NULL if we don't have a copy yet
    uint16   m_EventSeqno;        // last event sequence number sent

    ////////// REMOTE REPLICA MANAGEMENT

    // For primaries, we keep track of the remote replicas that are registered
    // to receive deltas from this dude -- this tells us how many remote dudes
    // have stable interets in us, used for interest link prediction.
    ReplicaPtrMap   m_RegisteredReplicas;

    ////////// COST ESTIMATE MANAGEMENT

    CostEstimate m_CostEstimate;

 protected:

    inline void AddBackPtr(InterestLink *link) {
	ASSERT(link);
	m_BackPtrs.insert(link);
    }
    inline void RemoveBackPtr(InterestLink *link) {
	ASSERT(link);
	m_BackPtrs.erase(link);
    }
    inline void RemoveLink(GUID guid) {
	InterestLinkMapIter linkp = m_Interests.find(guid);
	if (linkp != m_Interests.end()) {
	    m_LinkAllocator->free(linkp->second);
	    m_Interests.erase(linkp);
	}
    }

    inline void SetObject (GObject *obj) {
	m_Object = obj;
    }

    inline void SetManager(Manager *manager) {
	m_Manager = manager;
    }
    inline Manager *GetManager() {
	return m_Manager;
    }

 public:

    static void SetLinkAllocator(ClassChunker<InterestLink> *allocator) {
	m_LinkAllocator = allocator;
    }

    GObjectInfo() : TimedStruct(REPLICA_INTEREST_TTL) { 
	m_Manager    = NULL;
	m_Object     = NULL;
	m_EventSeqno = 0;
	m_GUID       = GUID_NONE;
	m_SID        = SID_NONE;
	m_IsReplica  = false;
    }

    virtual ~GObjectInfo() {
	OnDelete();
    }

    void OnDelete();

    inline void SetEventSeqno(uint16 s) { m_EventSeqno = s; }
    inline uint16 LastEventSeqno() { return m_EventSeqno; }
    inline uint16 NextEventSeqno() { 
	return m_EventSeqno = OMEvent::GetNextSeqno(m_EventSeqno); 
    }

    inline GObject *GetObject() { return m_Object; }
    inline ReplicaPtrMap *GetRegisteredReplicas() 
	{ 
	    return &m_RegisteredReplicas; 
	}
    // THIS SHOULD BE CONST!!
    inline InterestLinkMap  *GetInterests() { return &m_Interests; }
    // THIS SHOULD BE CONST!!
    inline InterestLinkSet *GetBackPtrs() { return &m_BackPtrs; }

    void RefreshInterest(GObjectInfo *other, uint32 ttl);
    void GarbageCollectInterests(bool force_all, TimeVal now = TIME_NONE);

    CostEstimate *GetCostEstimate() {
	return &m_CostEstimate;
    }
};

typedef map<GUID, GObjectInfo *, less_GUID> GObjectInfoMap;
typedef GObjectInfoMap::iterator GObjectInfoMapIter;

ostream& operator<<(ostream& out, GObjectInfo *info);

#endif
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
