////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************
  Manager.cpp

  Object Management Interface.

begin           : Nov 6, 2002
copyright       : (C) 2002-2003 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2002-2003 Justin Weisz       ( jweisz@cs.cmu.edu )

***************************************************************************/

#include <om/GObjectInfo.h>
#include <om/GObject.h>
#include <om/Manager.h>

    ///////////////////////////////////////////////////////////////////////////////
    ///// GOBJECT INFO

    const float CostEstimate::EMA_ALPHA;

ClassChunker<InterestLink> *GObjectInfo::m_LinkAllocator = NULL;

void GObjectInfo::OnDelete() 
{
    // Clean Up Registered Replicas
    m_RegisteredReplicas.clear();

    // Clean Up Interest Links
    for (InterestLinkMapIter it = m_Interests.begin(); 
	 it != m_Interests.end(); it++) {
	it->second->GetTarget()->RemoveBackPtr(it->second);
	m_LinkAllocator->free(it->second);
    }
    m_Interests.clear();

    for (InterestLinkSetIter it = m_BackPtrs.begin();
	 it != m_BackPtrs.end(); it++) {
	(*it)->GetSource()->RemoveLink( GetGUID() );
    }
    m_BackPtrs.clear();
}

void GObjectInfo::RefreshInterest(GObjectInfo *other, uint32 ttl) {
    InterestLink *link;
    ASSERT(other);
    // we are obviously interested in ourselves; don't add self-loops
    if (other->GetGUID() == GetGUID()) {
	ASSERT(this == other);
	DBG << "ignoring refresh interest to self: " << other << endl;
	return;
    }
    InterestLinkMapIter p = m_Interests.find(other->GetGUID());
    if (p == m_Interests.end()) {
	link = m_LinkAllocator->alloc();
	link->SetSource(this);
	link->SetTarget(other);
	m_Interests[other->GetGUID()] = link;
	other->AddBackPtr(link);
	DBG << "added new link: " << link << endl;
    } else {
	link = p->second;
	DBG << "updating existing link: " << link << endl;
    }
    ASSERT(link);
    link->Refresh(ttl);
}

void GObjectInfo::GarbageCollectInterests(bool force_all, TimeVal now)
{
    if (now == TIME_NONE) {
	OS::GetCurrentTime(&now);
    }

    ASSERT(!IsReplica() || m_Interests.size() == 0);

    for (InterestLinkMapIter it = m_Interests.begin(); 
	 it != m_Interests.end(); ) {
	InterestLinkMapIter oit = it;
	oit++;

	InterestLink *link = it->second;
	ASSERT(link);
	// are we about to delete the replica? if so we better check
	// to see if we are still interested int it!
	bool notInterested = 
	    link->TimedOut(now) || link->GetTarget()->TimedOut(now);

	if (!force_all && notInterested) {
	    GObject *obj  = GetObject();
	    ASSERT(obj); // must have it -- we own it!
	    GObject *other = link->GetTarget()->GetObject();

	    //
	    // Before timing out the link, make sure we aren't interested
	    // in it anymore...
	    //

	    if (other) {
		uint32 interestTime = 
		    obj->InterestTime(other,
				      GetManager()->
				      m_PubSubManager->GetCurrInterests());
		if (interestTime > 0) {
		    link->Refresh(interestTime + 
				  ManagerParams::INTEREST_LINK_TTL);
		    link->GetTarget()->
			Refresh(interestTime +
				ManagerParams::REPLICA_INTEREST_TTL);
		    notInterested = false;
		}
	    }
	}

	if (force_all || notInterested) {
	    DBG << "garbage collecting link " << link << endl;
	    link->GetTarget()->RemoveBackPtr(link);
	    m_LinkAllocator->free(link);
	    m_Interests.erase(it);
	}

	it = oit;
    }
}

///////////////////////////////////////////////////////////////////////////////

CostMetric CostEstimate::GetEstimate() const
{
    return m_Estimate;
}

uint32 CostEstimate::GetNumUpdates() const
{
    return m_NumUpdates;
}

void CostEstimate::Set(CostMetric init, uint32 num_updates)
{
    m_Estimate = init;
    m_NumUpdates = num_updates;
}

void CostEstimate::Set(const CostEstimate& old)
{
    m_Estimate = old.m_Estimate;
    m_NumUpdates = old.m_NumUpdates;
}

void CostEstimate::Update(CostMetric update)
{
    if (m_NumUpdates == 0)
	// first update, do a set (overest better than underest)
	m_Estimate = update;
    else
	// otherwise, exponentially weighted moving average
	m_Estimate = EMA_ALPHA*m_Estimate + (1-EMA_ALPHA)*update;
    m_NumUpdates++;
}

CostEstimate::CostEstimate(Packet *pkt)
{
    m_NumUpdates = pkt->ReadInt();
    m_Estimate = pkt->ReadFloat();
}

void CostEstimate::Serialize(Packet *pkt)
{
    pkt->WriteInt(m_NumUpdates);
    pkt->WriteFloat(m_Estimate);
}

uint32 CostEstimate::GetLength()
{
    return sizeof (m_NumUpdates) + sizeof (m_Estimate);
}

///////////////////////////////////////////////////////////////////////////////

ostream& operator<<(ostream& out, ReplicaPtr& ptr)
{
    out << "[" << ptr.GetNumInterests()
	<< " life=" << ptr.LifeTime()
	<< " exp=" << ptr.TimedOut() << "]";
    return out;
}

ostream& operator<<(ostream& out, InterestLink *link)
{
    out << "[{" << link->GetSource()->GetGUID() 
	<< "}->{" << link->GetTarget()->GetGUID()
	<< "} life=" << link->LifeTime()
	<< " exp=" << link->TimedOut() << "]";
    return out;
}

ostream& operator<<(ostream& out, GObjectInfo *info)
{
    out << "(GObjectInfo guid=" << info->GetGUID()
	<< " sid=" << info->GetSID()
	<< " interests=[";
    for (InterestLinkMapIter it = info->m_Interests.begin();
	 it != info->m_Interests.end(); it++) {
	if (it != info->m_Interests.begin()) out << ",";
	out << it->second;
    }
    out << "] backptrs=[";
    for (InterestLinkSetIter it = info->m_BackPtrs.begin();
	 it != info->m_BackPtrs.end(); it++) {
	if (it != info->m_BackPtrs.begin()) out << ",";
	out << *it;
    }
    out << "] registered=[";
    for (ReplicaPtrMapIter it = info->m_RegisteredReplicas.begin();
	 it != info->m_RegisteredReplicas.end(); it++) {
	if (it != info->m_RegisteredReplicas.begin()) out << ",";
	out << it->first << "->" << it->second;
    }
    out << "] lifetime=" << info->LifeTime() << " timeleft=" 
	<< info->TimeLeft() << ")";
    return out;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
