////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////
/* -*- Mode:C++; c-basic-offset:4; tab-width:4; indent-tabs-mode:t -*- */

/***************************************************************************
  DirectRouter.cpp

  Simple router for sending events directly between hosts.

begin           : Nov 6, 2002
copyright       : (C) 2002-2003 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2002-2003 Jeff Pang          ( jeffpang@cs.cmu.edu )

***************************************************************************/

#include <om/DirectRouter.h>
#include <om/Manager.h>
#include <util/Benchmark.h>
#include <om/OMMessage.h>
#include <mercury/Scheduler.h>
#include <wan-env/RealNet.h>

    DirectRouter *DirectRouter::m_Instance = NULL;

DirectRouter *DirectRouter::GetInstance(sid_t sid, Manager *handler) {
    if (DirectRouter::m_Instance == NULL) {
	DirectRouter::m_Instance = new DirectRouter(sid, handler);
    }
    return m_Instance;
}

DirectRouter *DirectRouter::GetInstance() {
    return DirectRouter::m_Instance;
}

DirectRouter::DirectRouter(sid_t sid, Manager *handler) : 
    m_Started(false), m_IP(sid), m_Manager(handler) 
{
    m_Scheduler = new WANScheduler ();
    m_Scheduler->SetNode(handler->GetMerc());
    m_Network = new RealNet(m_Scheduler, m_IP, 
			    ManagerParams::ENABLE_LOAD_INFO);
    m_BufferManager = new BufferManager();

    if (g_MeasurementParams.enabled) {
	((RealNet *)m_Network)->EnableLog();
    }

    m_Network->StartListening (PROTO_UDP);
    m_Network->StartListening (PROTO_TCP);
}

DirectRouter::~DirectRouter() {
    delete m_Network;
    delete m_BufferManager;
    delete m_Scheduler;
    m_Instance = NULL;
}

uint32 DirectRouter::GetIP() {
    return m_IP.GetIP();
}

unsigned short DirectRouter::GetPort() {
    return m_IP.GetPort();
}

void DirectRouter::SendMsg(IPEndPoint *target, MsgApp *msg, int protocol) {
    DBG << "sending msg(" << *target << "): " << msg << endl;
    // This happens when we forward messages for others...
    //ASSERT(*msg->GetSender() == m_Manager->GetSID());
    ASSERT(*target != m_Manager->GetSID());
    ASSERT(*target != SID_NONE);

    m_Network->SendMessage(msg, target, (TransportType) protocol);
}

void DirectRouter::FreeMsg(MsgApp *msg)
{
    m_Network->FreeMessage(msg);
}

void DirectRouter::RegisterInterest(Interest *) {
    Debug::die("can't register an Interest with the DirectRouter");
}

void DirectRouter::DoWork(u_long timeout) {
    m_Scheduler->ProcessTill (m_Scheduler->TimeNow());
}

MsgApp *DirectRouter::ReadMsg() {
    IPEndPoint from((uint32)0, 0);
    Message *msgp = 0;
    MsgApp *msg = 0;
    ConnStatusType status;

    do {
	status = m_Network->GetNextMessage(&from, &msgp);
	msg = (MsgApp *)msgp;

	DBG << "Got ConnectionStatus: " << g_ConnStatusStrings[status]
	    << endl;

	switch (status) {
	case CONN_NEWINCOMING:
	case CONN_OK:

	    break;
	case CONN_CLOSED:
	    DBG << "connection closed from: " << from.ToString() << endl;
	    break;
	case CONN_ERROR:
	    // XXX TODO
	    DBG << "connection error from: " << from.ToString() << endl;
	    break;
	case CONN_NOMSG:
	    break;
	default:
	    Debug::die("Hmm... got weird connection status: %d.", status);
	    break;
	}

    } while (msg == NULL && status != CONN_NOMSG);

    // Note: GetSender() may return our SID because in some race conditions
    // an object could migrate to us and in progress, we could update it
    // sending the remote update to the old owner. it will subsequently
    // forward the update back to us
    ASSERT(!msg || msg->sender != m_Manager->GetSID());
    ASSERT(!msg || msg->sender != SID_NONE);

    if (msg) DBG << "got msg: " << msg << endl;

    return msg;
}

bwidth_t DirectRouter::GetOutboundUsage(TimeVal& now)
{
    return m_Network->GetOutboundUsage(now);
}

bwidth_t DirectRouter::GetInboundUsage(TimeVal& now)
{
    return m_Network->GetInboundUsage(now);
}

///////////////////////////////////////////////////////////////////////////////
///// NOT IMPLEMENTED

void DirectRouter::SendEvent(Event *evt) {
    ASSERT(false);
}

Event *DirectRouter::ReadEvent() {
    ASSERT(false);
    return NULL;
}

void DirectRouter::LockBuffer() {
    ASSERT(false);
}

void DirectRouter::UnlockBuffer() {
    ASSERT(false);
}

void DirectRouter::Print(FILE *stream) {
    ASSERT(false);
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
