////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************

begin           : Nov 6, 2002
copyright       : (C) 2002-2005 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2002-2005 Jeffrey Pang       ( jeffpang@cs.cmu.edu )

***************************************************************************/

#ifndef __WAYPOINT_MANAGER_H__
#define __WAYPOINT_MANAGER_H__

#include <list>
#include <vector>
#include <gameapi/common.h>
#include <gameapi/GameObject.h>

#define WAYPOINT_GOAL 1

    struct WayPoint {
	uint32 id; // assigned by WayPointManager -- globally unique WP ID
	byte flags;
	Vec3 orig;
	real32 radius;
	// edges are directed!
	vector<WayPoint *> neighbors;

	WayPoint() : flags(0) {}
	WayPoint(const Vec3& orig, real32 radius) : 
	    orig(orig), radius(radius), flags(0) {}

	void SetIsGoal(bool v) {
	    if (v)
		flags |= WAYPOINT_GOAL;
	    else
		flags &= ~WAYPOINT_GOAL;
	}

	inline uint32 GetID() const { return id; }
	inline const Vec3& GetOrigin() const { return orig; }
	inline bool IsGoal() const { return (bool)(flags&WAYPOINT_GOAL); }
	inline const vector<WayPoint *>& GetNeighbors() const {
	    return neighbors;
	}

	uint32 size() const {
	    return neighbors.size();
	}

	bool Contains(const Vec3& pt) const {
	    return pt.Distance(orig) <= radius;
	}
    };

typedef vector<WayPoint *> WayPointVec;
typedef pair<WayPoint *, WayPoint *> WayPointEdge;
typedef vector<WayPointEdge> WayPointEdgeVec;

class WayPointFactory {
 public:
    virtual ~WayPointFactory () {}

    virtual void GenWayPoints(WayPointVec *toFill) const = 0;
    virtual void DestroyWayPoints(WayPointVec *wps) const = 0;
};

class SpaceWorld;

/**
 * Randomly distributed G(n,p) graph as a waypoint graph.
 */
class RandomWayPointFactory : public WayPointFactory {
 private:
    uint32 n;
    real32 p;
    real32 rad;
    SpaceWorld *w;
 public:
    /**
     * @param n number of nodes
     * @param p probability of an edge
     * @param sz radius of each wayppint
     * @param w gameworld to waypoint. Right now only supports SpaceWorld
     */
    RandomWayPointFactory(uint32 n, real32 p, real32 rad, SpaceWorld *w);

    void GenWayPoints(WayPointVec *toFill) const;
    void DestroyWayPoints(WayPointVec *wps) const;
};

/**
 * Reads in waypoint graph from a file. The format should be:
 *
 * <num_waypoints> <num_edges>
 * <waypoint_0>
 * ...
 * <waypoint_n-1>
 * <edge_0>
 * ...
 * <edge_n-1>
 *
 * where each waypoint is:
 *
 * <x> <y> <z> <radius> <is_goal>
 *
 * and each edge is:
 *
 * <start_index> <end_index>
 *
 * i.e., the index of the waypoints in the preceding waypoint list
 * (starting with 0).
 */
class FileWayPointFactory : public WayPointFactory {
 private:
    WayPointVec wps;
 public:
    FileWayPointFactory(const char *filename);
    virtual ~FileWayPointFactory();

    void GenWayPoints(WayPointVec *toFill) const;
    void DestroyWayPoints(WayPointVec *wps) const;
};

///////////////////////////////////////////////////////////////////////////////

// currently this can not be modified after construction!
class WayPointManager {
 private:
    // XXX: TODO: quad tree this (?)
    const WayPointFactory *fact;

    real32 alpha;
    vector<real64> goalProbs;
    WayPointVec goals;
    WayPointVec waypoints;
    WayPointEdgeVec edges;

    /*
    // <next_hop,dist>
    pair<sint32,real32> **shortestPaths;
    */

    real32 avgDeg;
    real32 medDeg;
 public:
    // c = constructs waypoints
    // alpha = zipf alpha parameter for probability of goals
    WayPointManager(const WayPointFactory *c, real32 alpha);
    virtual ~WayPointManager();

    const WayPoint *GetClosestWayPoint(const Vec3& pt) const;
    const WayPoint *GetClosestGoal(const Vec3& pt) const;

    const WayPoint *GetWayPointByID(uint32 id) const;
    const WayPoint *GetRandomWayPoint() const;
    // uniformly at random
    const WayPoint *GetRandomGoal() const;
    // based on roughly Zipf distribution
    const WayPoint *GetWeightedRandomGoal() const;
    const WayPointVec *GetWayPoints() const;
    const WayPointEdgeVec *GetEdges() const;

    /*
      const WayPoint *GetNextHop(const WayPoint *curr, 
      const WayPoint *target) const;
    */
    bool GetPath(list<const WayPoint *> *path, 
		 const WayPoint *from, 
		 const WayPoint *to);

    const WayPoint *GetNextGoal(const Vec3& loc,
				const list<const WayPoint *> *prevGoals);

    uint32 NumWayPoints() const;
    uint32 NumEdges() const;
    real32 AvgDegree() const;
    real32 MedianDegree() const;
};

#endif
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
