////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************

begin           : Nov 6, 2002
copyright       : (C) 2002-2005 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2002-2005 Jeffrey Pang       ( jeffpang@cs.cmu.edu )

***************************************************************************/

#include <algorithm>
#include <util/FibHeap.h>
#include "SpaceWorld.h"
#include "WayPointManager.h"
#include <util/Benchmark.h>

    RandomWayPointFactory::RandomWayPointFactory(uint32 n, real32 p, real32 rad,
						 SpaceWorld *w) :
	n(n), p(p), rad(rad), w(w) {
}

void RandomWayPointFactory::GenWayPoints(WayPointVec *toFill) const
{
    // make nodes
    for (uint32 i=0; i<n; i++) {
	WayPoint *p = new WayPoint( w->GetRandomPoint(), rad );
	p->SetIsGoal(true);
	ASSERT(w->GetExtent().Contains( p->orig ));
	toFill->push_back( p );
    }

    int edges = 0;

    // make edges
    for (uint32 i=0; i<n; i++) {
	for (uint32 j=0; j<n; j++) {
	    if (i == j) continue;
	    if (drand48() <= p) {
		//DBG << "edge [" << i << "," << j << "]" << endl;
		(*toFill)[i]->neighbors.push_back( (*toFill)[j] );
		edges++;
	    }
	}
    }

    //DBG << "num nodes: " << toFill->size() << endl;
    //DBG << "num edges: " << edges << endl;
}

void RandomWayPointFactory::DestroyWayPoints(WayPointVec *wps) const
{
    for (uint32 i=0; i<wps->size(); i++) {
	delete (*wps)[i];
    }
    wps->clear();
}

///////////////////////////////////////////////////////////////////////////////

FileWayPointFactory::FileWayPointFactory(const char *filename)
{
    char line[255];
    char *lret;
    int cret;

    FILE *fp = fopen(filename, "r");
    if (!fp) {
	Debug::die("couldn't open waypoint file %s", filename);
    }

    uint32 nodes;
    uint32 edges;

    // how many nodes and how many edges?
    lret = fgets(line, 255, fp);
    ASSERT(lret);
    cret = sscanf(line, "%d %d\n", &nodes, &edges);
    if (cret != 2) {
	Debug::die("bad 'nodes edges' line in waypoint file %s", filename);
    }
    ASSERT(nodes > 0);
    ASSERT(edges > 0);

    for (uint32 i=0; i<nodes; i++) {
	real32 x,y,z;
	real32 rad;
	int goal;

	lret = fgets(line, 255, fp);
	ASSERT(lret);
	cret = sscanf(line, "%f %f %f %f %d\n", &x,&y,&z,&rad,&goal);
	if (cret != 5) {
	    Debug::die("bad nodes line in waypoint file %s", filename);
	}

	WayPoint *p = new WayPoint(Vec3(x,y,z), rad);
	p->SetIsGoal(goal != 0);

	wps.push_back(p);
    }

    for (uint32 i=0; i<edges; i++) {
	int start;
	int end;

	lret = fgets(line, 255, fp);
	ASSERT(lret);
	cret = sscanf(line, "%d %d\n", &start, &end);
	if (cret != 2) {
	    Debug::die("bad edges line in waypoint file %s", filename);
	}

	ASSERT(start >= 0 && start < (int)wps.size());
	ASSERT(end >= 0 && end < (int)wps.size());

	wps[start]->neighbors.push_back(wps[end]);
    }

    fclose(fp);
}

FileWayPointFactory::~FileWayPointFactory()
{
    for (uint32 i=0; i<wps.size(); i++) {
	delete wps[i];
    }
}

void FileWayPointFactory::GenWayPoints(WayPointVec *toFill) const
{
    for (uint32 i=0; i<wps.size(); i++) {
	toFill->push_back(wps[i]);
    }
}

void FileWayPointFactory::DestroyWayPoints(WayPointVec *wps) const
{
    // nothing
}

///////////////////////////////////////////////////////////////////////////////

WayPointManager::WayPointManager(const WayPointFactory *c, real32 alpha) :
    fact(c), alpha(alpha)
{
    fact->GenWayPoints(&waypoints);

    vector<real32> degs;
    avgDeg = 0;

    for (uint32 i=0; i<waypoints.size(); i++) {
	waypoints[i]->id = i; // must be deterministic!
	degs.push_back( waypoints[i]->size() );
	avgDeg += waypoints[i]->size();
	for (vector<WayPoint *>::iterator it = waypoints[i]->neighbors.begin();
	     it != waypoints[i]->neighbors.end(); it++) {
	    edges.push_back( WayPointEdge(waypoints[i], *it) );
	}

	if (waypoints[i]->IsGoal()) {
	    goals.push_back(waypoints[i]);
	}
    }

    // compute the probabilities of entering each goal waypoint based on
    // a Zipf distribution

    // first calc the approx unnormalized prob based on rank 
    real64 total = 0, norm_so_far = 0;
    for (uint32 i=0; i<goals.size(); i++) {
	real64 v = pow(i+1, -alpha);
	goalProbs.push_back( v );
	total += v;
    }
    // normalize the value and split the range from [0,1]
    for (uint32 i=0; i<goals.size(); i++) {
	real64 nv = goalProbs[i]/total;
	norm_so_far += nv;
	goalProbs[i] = norm_so_far;
    }
    // now the prob of a given rank in the list of goals is equiv to
    // its range within [0,1]

    // test
    for (int i=0; i<10; i++) {
	GetWeightedRandomGoal();
    }

    avgDeg /= degs.size();
    stable_sort(degs.begin(), degs.end());
    medDeg = degs.size() > 0 ? degs[degs.size()/2] : 0;

    /* Too slow for large numbers of waypoints
    // compute shortest paths
    shortestPaths = new pair<sint32,real32> *[waypoints.size()];
    for (uint32 i=0; i<waypoints.size(); i++) {
    shortestPaths[i] = new pair<sint32,real32>[waypoints.size()];
    for (uint32 j=0; j<waypoints.size(); j++) {
    shortestPaths[i][j].first  = -1;
    shortestPaths[i][j].second = INFINITY;
    }
    }

    for (uint32 i=0; i<edges.size(); i++) {
    uint32 start = edges[i].first->id;
    uint32 end   = edges[i].second->id;

    shortestPaths[start][end].first = end;
    shortestPaths[start][end].second = 
    edges[i].first->GetOrigin().
    Distance(edges[i].second->GetOrigin());
    }

    for (uint32 k=0; k<waypoints.size(); k++) {
    for (uint32 i=0; i<waypoints.size(); i++) {
    for (uint32 j=0; j<waypoints.size(); j++) {
    real32 dist = 
    shortestPaths[i][k].second + shortestPaths[k][j].second;
    if (dist < shortestPaths[i][j].second) {
    shortestPaths[i][j].first  = k;
    shortestPaths[i][j].second = dist;
    }
    }
    }
    }
    */
}

WayPointManager::~WayPointManager()
{
    fact->DestroyWayPoints(&waypoints);

    /*
      for (uint32 i=0; i<waypoints.size(); i++) {
      delete[] shortestPaths[i];
      }
      delete[] shortestPaths;
    */
}

const WayPoint *WayPointManager::GetClosestWayPoint(const Vec3& pt) const
{
    real32 min = INFINITY;
    WayPoint *p = NULL;

    // XXX: make more efficient!
    for (uint32 i=0; i<waypoints.size(); i++) {
	real32 d = pt.Distance(waypoints[i]->orig);
	if (d < min) {
	    min = d;
	    p = waypoints[i];
	}
    }

    return p;
}

const WayPoint *WayPointManager::GetClosestGoal(const Vec3& pt) const
{
    real32 min = INFINITY;
    WayPoint *p = NULL;

    // XXX: make more efficient!
    for (uint32 i=0; i<goals.size(); i++) {
	real32 d = pt.Distance(goals[i]->orig);
	if (d < min) {
	    min = d;
	    p = goals[i];
	}
    }

    return p;
}

const WayPoint *WayPointManager::GetNextGoal(const Vec3& pt,
					     const list<const WayPoint *> *prevGoals)
{
    //return GetRandomGoal();
    return GetWeightedRandomGoal();

    /*
    // XXX TODO -- pick a goal based on previous few goals
    real32 min = INFINITY;
    WayPoint *p = NULL;

    // for now, just pick the closest goal that was not visited in
    // recent past
    set<const WayPoint *> visited(prevGoals->begin(), prevGoals->end());

    // XXX: make more efficient!
    for (uint32 i=0; i<goals.size(); i++) {
    if (visited.find(goals[i]) != visited.end()) continue;
    real32 d = pt.Distance(goals[i]->orig);
    if (d < min) {
    min = d;
    p = goals[i];
    }
    }

    return p;
    */
}

const WayPoint *WayPointManager::GetWayPointByID(uint32 id) const
{
    ASSERT(id < waypoints.size());
    return waypoints[id];
}

const WayPoint *WayPointManager::GetRandomWayPoint() const
{
    uint32 index = (int)(drand48()*waypoints.size());
    return waypoints[index];
}

const WayPoint *WayPointManager::GetRandomGoal() const
{
    uint32 index = (int)(drand48()*goals.size());
    return goals[index];
}

const WayPoint *WayPointManager::GetWeightedRandomGoal() const
{
    real64 v = drand48();
    // find the index with smallest value >= v;
    // bin search
    uint32 min = 0;
    uint32 max = goals.size()-1;
    while (true) {
	if (min == max) break;
	uint32 index = (max-min)/2 + min;
	if (goalProbs[index] < v) {
	    min = index + 1;
	} else if (index == min || v > goalProbs[index-1]) {
	    min = max = index;
	    break;
	} else {
	    max = index - 1;
	}
    }

    return goals[min];
}

const WayPointVec *WayPointManager::GetWayPoints() const
{
    return &waypoints;
}
const WayPointEdgeVec *WayPointManager::GetEdges() const
{
    return &edges;
}

/*
  const WayPoint *WayPointManager::GetNextHop(const WayPoint *curr, 
  const WayPoint *target) const
  {
  sint32 index = shortestPaths[curr->id][target->id].first;
  if (index < 0) {
  return NULL;
  } else {
  return waypoints[index];
  }
  }
*/

struct WayPointInfo {
    const WayPoint *wp;
    WayPointInfo *parent;
    real32 f, g;

    WayPointInfo() : wp(NULL), f(0), g(0), parent(NULL) {}
    WayPointInfo(const WayPoint *wp) :
	wp(wp), f(0), g(0), parent(NULL) {}
    WayPointInfo(const WayPointInfo& other) : 
	wp(other.wp), f(other.f), g(other.g), parent(other.parent)
    {}

    bool operator<(const WayPointInfo& other) {
	return this->f < other.f;
    }
    bool operator<=(const WayPointInfo& other) {
	return this->f <= other.f;
    }
    bool operator>(const WayPointInfo& other) {
	return this->f > other.f;
    }
    bool operator>=(const WayPointInfo& other) {
	return this->f >= other.f;
    }
};

struct HeapInfo {
    WayPointInfo *info;
    FibHeapElt   *elt;

    HeapInfo() : info(NULL), elt(NULL) {};
    HeapInfo(const HeapInfo& other) :
	info(other.info), elt(other.elt) {}
};

#include <util/google/sparsehash/config.h>
#include <util/google/dense_hash_map>
using GOOGLE_NAMESPACE::dense_hash_map;

// typedef map<const WayPoint *, HeapInfo, less_ptr> WayPointMap;

struct WayPointHash { 
    // awfulcast
    int operator() (const WayPoint *pw) const {
	return hash<int> () ((int) pw);
    }
};

struct WayPointEq { 
    bool operator() (const WayPoint *pa, const WayPoint *pb) const {
	return pa == pb;
    }
};

typedef dense_hash_map<const WayPoint *, HeapInfo, WayPointHash, WayPointEq> WayPointMap;
typedef FibHeap<WayPointInfo> WayPointHeap;

static ClassChunker<WayPointInfo> g_WPIChunker(102400);

bool WayPointManager::GetPath(list<const WayPoint *> *path, 
			      const WayPoint *from, 
			      const WayPoint *to)
{
    // TODO: this implementation of A* is somewhat uglified due to
    // the heap interface issues. :P

    bool done = false;

    list<WayPointInfo *> tofree;
    {
	WayPointMap ref;
	WayPointHeap open;
	WayPointMap closed;

	// these wont be valid referenced pointers, hopefully :P
	ref.set_empty_key ((WayPoint *) 0x1);
	ref.set_deleted_key ((WayPoint *) 0x2); 

	closed.set_empty_key ((WayPoint *) 0x1);
	closed.set_deleted_key ((WayPoint *) 0x2);

	WayPointInfo *wpi = g_WPIChunker.alloc();
	tofree.push_back(wpi);

	wpi->wp = from;
	ref[from].info = wpi;
	ASSERT(ref[from].info->wp);
	ref[from].elt = open.Insert(wpi);

	WayPointInfo *curr;
	while ((curr = open.ExtractMin()) != NULL) {
	    //cout << "looking at: " << curr->wp->GetOrigin() 
	    //	 <<  " neighbors: " << curr->wp->neighbors.size() << endl;
	    ASSERT(curr->wp);
	    HeapInfo hi = ref[curr->wp];
	    hi.elt = NULL;
	    ref.erase(curr->wp);

	    if (curr->wp == to) {
		done = true;
		break;
	    }

	    for (uint32 i=0; i<curr->wp->neighbors.size(); i++) {
		WayPoint *n = curr->wp->neighbors[i];
		ASSERT(n);
		WayPointInfo info(n);
		ASSERT(info.wp);
		info.parent = curr;
		info.g = curr->g + curr->wp->GetOrigin().Distance(n->GetOrigin());
		real32 h = n->GetOrigin().Distance(to->GetOrigin());
		info.f = info.g + h;

		WayPointMap::iterator ptr = ref.find(n);
		if (ptr != ref.end()) {
		    if (ptr->second.info->f <= info.f) {
			continue;
		    } else {
			ASSERT(n != from);

			//cout << "here: wp=" << n->GetOrigin() << endl;
			//cout << "here: f=" << info.f << endl;

			// ick! this code is nasty!
			WayPointInfo *wpi = g_WPIChunker.alloc();
			tofree.push_back(wpi);

			*wpi = info;
			ptr->second.info = wpi;
			ASSERT(ptr->second.info->wp);
			open.ChangeKey(ptr->second.elt, ptr->second.info);
		    }
		} else {
		    ptr = closed.find(n);
		    if (ptr != closed.end()) {
			if (ptr->second.info->f <= info.f) {
			    continue;
			}
		    }

		    // XXX This should never happen but it seems like it does
		    // so this is a hack which masks a larger bug. oh well.
		    if (n == from) continue;

		    // newly added to open list
		    WayPointInfo *wpi = g_WPIChunker.alloc();
		    tofree.push_back(wpi);

		    *wpi = info;
		    ref[n].info = wpi;
		    ASSERT(ref[n].info->wp);
		    ref[n].elt = open.Insert(ref[n].info);
		}
	    }

	    closed[curr->wp] = hi;
	}

	if (done) {
	    while (curr != NULL) {
		//cout << curr->wp->GetOrigin() << endl;
		path->push_front(curr->wp);
		curr = curr->parent;
	    }
	}
    }

    for (list<WayPointInfo *>::iterator it = tofree.begin();
	 it != tofree.end(); it++) {
	g_WPIChunker.free(*it);
    }

    return done;
}

uint32 WayPointManager::NumWayPoints() const
{
    return waypoints.size();
}

uint32 WayPointManager::NumEdges() const
{
    return edges.size();
}

real32 WayPointManager::AvgDegree() const
{
    return avgDeg;
}

real32 WayPointManager::MedianDegree() const
{
    return medDeg;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
