////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************

begin           : Nov 6, 2002
copyright       : (C) 2002-2005 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2002-2005 Jeffrey Pang       ( jeffpang@cs.cmu.edu )

***************************************************************************/

#ifndef __SIMPLE_WORLD_H__
#define __SIMPLE_WORLD_H__

#include <mercury/common.h>
#include "WayPointManager.h"
#include "EmptyWorld.h"

    class SimpleWorld : public EmptyWorld {
    private:
    WayPointManager *wpm;
    uint32 region_split[2];
    protected:
    virtual WayPointManager *GetWayPoints() { return wpm; }
    public:

    SimpleWorld(BBox dims, BBox aoi, 
		uint32 regions_per_x = 1, 
		uint32 regions_per_y = 1) : 
	EmptyWorld(dims, aoi), wpm(NULL) {
	region_split[0] = regions_per_x;
	region_split[1] = regions_per_y;
    };
    virtual ~SimpleWorld() {
	delete wpm;
    };

    const bool PublishDim(uint32 dim) {
	return dim == 0 || dim == 1; // don't publish z
    }

    void Init(WayPointFactory *f, real32 alpha) {
	wpm = new WayPointManager(f, alpha);
    }

    ///////////////////////////////////////////////////////////////////////////

    WayPointManager *GetWayPointManager() {
	return wpm;
    }
    const WayPoint *GetClosestWayPoint(const Vec3& pt) {
	return wpm->GetClosestWayPoint(pt);
    }
    const WayPoint *GetRandomWayPoint() {
	return wpm->GetRandomWayPoint();
    }
    const WayPoint *GetWayPointByID(uint32 id) {
	return wpm->GetWayPointByID(id);
    }
    bool GetPath(list<const WayPoint *> *path, 
		 const WayPoint *from, 
		 const WayPoint *to)
	{
	    return wpm->GetPath(path, from, to);
	}

    uint32 GetRegion(const Vec3& pt) {
	const BBox& dims = GetExtent();
	real32 range[2];
	for (uint32 i=0; i<2; i++)
	    range[i] = dims.max[i] - dims.min[i];

	uint32 indices[2];
	real32 p;
	for (int i = 0; i < 2; i++) {
	    p = MAX(pt[i], dims.min[i]);
	    p = MIN(p, dims.max[i]);

	    indices[i] = (uint32)(region_split[i] * 
				  (p - dims.min[i])/range[i]);
	}

	uint32 ret = indices[0] * region_split[1] + indices[1];
	ASSERT( ret < region_split[0]*region_split[1] );
	return ret;
    }

    void GetRegions(vector<uint32> *regs, const BBox& bbox) {
	uint32 imin[2], imax[2];

	const BBox& dims = GetExtent();
	real32 range[2];
	for (uint32 i=0; i<2; i++)
	    range[i] = dims.max[i] - dims.min[i];

	// compute the bounding box in terms of region indices
	real32 min, max;

	for (int i=0; i<2; i++) {
#if 0
	    min = bbox.min[i];
	    max = bbox.max[i];
	    if (bbox.min[i] < dims.min[i]) return;
	    if (bbox.max[i] > dims.max[i]) return;
#else 
	    /* perhaps there could be a tiny floating point error here. try to be 
	     * a little more relaxed about these things */
	    min = MAX(bbox.min[i], dims.min[i]);
	    max = MIN(bbox.max[i], dims.max[i]);
#endif

	    imin[i] = (uint32)(region_split[i] * 
			       (min-dims.min[i])/range[i]);
	    imax[i] = (uint32)(region_split[i] * 
			       (max-dims.min[i])/range[i]);
	}

	// convert each discrete bounding box point into a region
	for (uint32 x = imin[0]; x <= imax[0]; x++) {
	    for (uint32 y = imin[1]; y <= imax[1]; y++) {
		uint32 reg = x * region_split[1] + y;
		regs->push_back(reg);
	    }
	}
    }
};

#endif
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
