////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************

begin           : Nov 6, 2002
copyright       : (C) 2002-2005 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2002-2005 Jeffrey Pang       ( jeffpang@cs.cmu.edu )

***************************************************************************/

#include <gameapi/GameManager.h>
#include <gameapi/GameStore.h>
#include "SimpleTypeCodes.h"
#include "SimpleWorld.h"
#include "SimpleMissile.h"
#include "SimplePlayer.h"
#include "SimpleGame.h"

    const real32 SimplePlayer::SIZE   ;
const real32 SimplePlayer::HEIGHT ; // doesn't matter, we have no zaxis

// based on q3 max walk velocity
const uint32 SimplePlayer::MAX_HEALTH ;
const uint32 SimplePlayer::MAX_AMMO ;
// (100+200)/10 (suppose 1/2 time walking, 1/2 time running, no accel)
const real32 SimplePlayer::WALK_VELOCITY ;

// these probs approximately estimated from q3dm7 and q3dm14 games
const real32 SimplePlayer::PROB_RUN_AWAY ;
const real32 SimplePlayer::PROB_FIGHT ;
const real32 SimplePlayer::PROB_LEAVE ;

// if this far away from a target, try to move closer
const real32 SimplePlayer::FIGHT_FOLLOW_RADIUS ; // about 1/2 sub-size
const real32 SimplePlayer::WANDER_GOAL_DIST ; // about 1/2 sub-size
const real32 SimplePlayer::WANDER_GOAL_RADIUS ;

uint32 SimplePlayer::SUB_PREDICTION = 5000;

bool g_SimplePlayerWander = false;

static bool inited = false;
static DeltaMask initDeltaMask;
static const BBox vol(Vec3(-SimplePlayer::SIZE/2,-SimplePlayer::SIZE/2,-SimplePlayer::HEIGHT/2), Vec3(SimplePlayer::SIZE/2,SimplePlayer::SIZE/2,SimplePlayer::HEIGHT/2));

SimplePlayer::SimplePlayer(GameManager *manager) :
    SimpleMovable(manager, SIMPLE_PLAYER), nextAttack(0)
{
    SetDefaults();
    Respawn(manager->GetWorld());
}

SimplePlayer::SimplePlayer(GObjectInfoIface *info) :
    SimpleMovable(info, SIMPLE_PLAYER), nextAttack(0)
{
    SetDefaults();
}

SimplePlayer::~SimplePlayer()
{}

const char *SimplePlayer::StateToString(SimplePlayerState s)
{
    switch(s) {
    case SP_FIGHTING_AT_WAYPOINT:
	return "SP_FIGHTING_AT_WAYPOINT";
    case SP_FIGHTING_TO_WAYPOINT:
	return "SP_FIGHTING_TO_WAYPOINT";
    case SP_AT_WAYPOINT:
	return "SP_AT_WAYPOINT";
    case SP_TO_WAYPOINT:
	return "SP_TO_WAYPOINT";
    default:
	return "SP_UNKNOWN";
    }
}

///////////////////////////////////////////////////////////////////////////////

void SimplePlayer::SetDefaults()
{
    health = MAX_HEALTH;
    ammo = MAX_AMMO;
    frags = 0;
    state = SP_TO_WAYPOINT;
    nextAttack = 0;
    longTermGoal = NULL;
    goal = NULL;
    wanderGoal = Vec3::ZERO;

    target = NULL;
    path.clear();
    clientOutbound = 0;
}

// Bits:
//
// 0 - health
// 1 - ammo
// 2 - frags
// 3 - state
// 4 - nextAttack
// 5 - longTermGoal
// 6 - goal
// 7 - wanderGoal

#define NUM_MASK_BITS 8

#define SET_MASK(index) (SetDeltaMaskField( SimpleMovable::GetDeltaMaskBits() + (index) ));
#define ISSET_MASK(mask, index) ((mask).IsSet( SimpleMovable::GetDeltaMaskBits() + (index) ))

const uint32 SimplePlayer::GetDeltaMaskBits() { 
    return SimpleMovable::GetDeltaMaskBits() + NUM_MASK_BITS; 
}

const DeltaMask& SimplePlayer::GetInitDeltaMask()
{
    if (!inited) {
	initDeltaMask = SimpleMovable::GetInitDeltaMask();
	for (int i=0; i<NUM_MASK_BITS; i++) {
	    initDeltaMask.Set( SimpleMovable::GetDeltaMaskBits() + i );
	}
	inited = true;
    }

    return initDeltaMask;
}

void SimplePlayer::PackUpdate(Packet *buffer, const DeltaMask& mask)
{
    SimpleMovable::PackUpdate(buffer, mask);
    if (ISSET_MASK(mask, 0)) {
	buffer->WriteShort( health );
    }
    if (ISSET_MASK(mask, 1)) {
	buffer->WriteShort( ammo );
    }
    if (ISSET_MASK(mask, 2)) {
	buffer->WriteInt( frags );
    }
    if (ISSET_MASK(mask, 3)) {
	buffer->WriteByte( state );
    }
    if (ISSET_MASK(mask, 4)) {
	buffer->WriteShort( nextAttack );
    }
    if (ISSET_MASK(mask, 5)) {
	buffer->WriteInt(longTermGoal == NULL ? (uint32)-1 : longTermGoal->id);
    }
    if (ISSET_MASK(mask, 6)) {
	buffer->WriteInt(goal == NULL ? (uint32)-1 : goal->id);
    }
    if (ISSET_MASK(mask, 7)) {
	wanderGoal.Serialize(buffer);
    }
}

void SimplePlayer::UnpackUpdate(Packet *buffer, const DeltaMask& mask, 
				list<GameObjectRef *> *unresolved)
{
    SimpleWorld *w = 
	static_cast<SimpleWorld *>(GameManager::GetInstance()->GetWorld());

    SimpleMovable::UnpackUpdate(buffer, mask, unresolved);
    if (ISSET_MASK(mask, 0)) {
	health = buffer->ReadShort();
    }
    if (ISSET_MASK(mask, 1)) {
	ammo = buffer->ReadShort();
    }
    if (ISSET_MASK(mask, 2)) {
	frags = buffer->ReadInt();
    }
    if (ISSET_MASK(mask, 3)) {
	state = (SimplePlayerState)buffer->ReadByte();
    }
    if (ISSET_MASK(mask, 4)) {
	nextAttack = buffer->ReadShort();
    }
    if (ISSET_MASK(mask, 5)) {
	uint32 id = buffer->ReadInt();
	if (id == (uint32)-1)
	    longTermGoal = NULL;
	else
	    // XXX should fail gracefully here!
	    longTermGoal = w->GetWayPointByID(id);
    }
    if (ISSET_MASK(mask, 6)) {
	uint32 id = buffer->ReadInt();
	if (id == (uint32)-1)
	    goal = NULL;
	else
	    // XXX should fail gracefully here!
	    goal = w->GetWayPointByID(id);
    }
    if (ISSET_MASK(mask, 7)) {
	wanderGoal = Vec3(buffer);
    }
}

void SimplePlayer::PrintFields(ostream& out)
{
    SimpleMovable::PrintFields(out);
    out << " health=" << health << " ammo=" << ammo
	<< " state=" << StateToString(state)
	<< " nextAttack=" << nextAttack;
}

uint16 SimplePlayer::GetHealth() {
    return health;
}

void SimplePlayer::SetHealth(uint16 v) {
    SET_MASK( 0 );
    health = v;
}

uint16 SimplePlayer::GetAmmo() {
    return ammo;
}

void SimplePlayer::SetAmmo(uint16 v) {
    SET_MASK( 1 );
    ammo = v;
}

uint32 SimplePlayer::GetFrags() {
    return frags;
}

void SimplePlayer::SetFrags(uint32 v) {
    SET_MASK( 2 );
    frags = v;
}

void SimplePlayer::IncFrags() {
    SetFrags( GetFrags() + 1 );
}

SimplePlayerState SimplePlayer::GetState() {
    return state;
}

void SimplePlayer::SetState(SimplePlayerState s) {
    SET_MASK( 3 );
    state = s;
}

uint16 SimplePlayer::GetNextAttack() {
    return nextAttack;
}

void SimplePlayer::SetNextAttack(uint16 f) {
    SET_MASK( 4 );
    nextAttack = f;
}

const WayPoint *SimplePlayer::GetLongTermGoal()
{
    return longTermGoal;
}

void SimplePlayer::SetLongTermGoal(const WayPoint *goal)
{
    SET_MASK( 5 );
    longTermGoal = goal;
}

const WayPoint *SimplePlayer::GetGoal()
{
    return goal;
}

void SimplePlayer::SetGoal(const WayPoint *goal)
{
    SET_MASK( 6 );
    this->goal = goal;
}

const Vec3& SimplePlayer::GetWanderGoal()
{
    return wanderGoal;
}

void SimplePlayer::SetWanderGoal(const Vec3& goal)
{
    SET_MASK( 7 );
    wanderGoal = goal;
}

///////////////////////////////////////////////////////////////////////////////

uint32 SimplePlayer::GetAreaOfInterest(list<BBox> *toFill, GameWorld *w) {
    w->GetAreaOfInterest(toFill, this);

    uint32 ret = MAX(SUB_PREDICTION, 10);
#if 0
    if (g_Preferences.slowdown_factor > 1.0f) 
	ret = (uint32) (ret * g_Preferences.slowdown_factor);
#endif

    ASSERT (ret > 0);
    return ret;
}

uint32 SimplePlayer::IsInterested(const Vec3& pt, gTypeCode t)
{
    list<BBox> aoi;

    GetPredictedAreaOfInterest(&aoi, GameManager::GetInstance()->GetWorld());

    for (list<BBox>::iterator it = aoi.begin(); it != aoi.end(); it++) {
	if ((*it).Contains(pt)) {
	    switch (t) {
	    case SIMPLE_PLAYER:
		return 3 * SimpleGame::frameInterval;
	    case SIMPLE_MISSILE:
		return 3 * SimpleGame::frameInterval;
	    default:
		// take it for a little bit
		return 3 * SimpleGame::frameInterval;
	    }
	}
    }

    return 0;
}

uint32 SimplePlayer::IsInterested(const BBox& vol, gTypeCode t)
{
    list<BBox> aoi;

    GetPredictedAreaOfInterest(&aoi, GameManager::GetInstance()->GetWorld());

    for (list<BBox>::iterator it = aoi.begin(); it != aoi.end(); it++) {
	if ((*it).Overlaps(vol)) {
	    switch (t) {
	    case SIMPLE_PLAYER:
		return 10 * SimpleGame::frameInterval;
	    case SIMPLE_MISSILE:
		return 5 * SimpleGame::frameInterval;
	    default:
		// take it for a little bit
		return 3 * SimpleGame::frameInterval;
	    }
	}
    }

    return 0;
}

///////////////////////////////////////////////////////////////////////////////

void SimplePlayer::FindVisible(SimpleWorld *w)
{
    visible.clear();
    visiblePlayers = 0;
    visibleMissiles = 0;
    list<GameObject *> fill;
    w->GetVisible( &fill, this, this );
    for (list<GameObject *>::iterator it = fill.begin();
	 it != fill.end(); it++) {
	visible.push_back(GameObjectRef(*it));
	if ((*it)->GetType() == SIMPLE_PLAYER) {
	    visiblePlayers++;
	} else if ((*it)->GetType() == SIMPLE_MISSILE) {
	    visibleMissiles++;
	}
    }
}

void SimplePlayer::FindEnemy(SimpleWorld *w)
{
    // look for a new target
    for (list<GameObjectRef>::iterator it = visible.begin();
	 it != visible.end(); it++) {
	if ((*it)->GetType() == SIMPLE_PLAYER) {
	    target = static_cast<SimplePlayer *>((GameObject *)*it);
	    break;
	}
    }
}

void SimplePlayer::PickWanderGoal(SimpleWorld *w)
{
    if (!g_SimplePlayerWander)
	return;

    ASSERT( GetLongTermGoal() );
    Vec3 ngoal = w->Clip( GetLongTermGoal()->GetOrigin() + 
			  Vec3::Random().Normalize()*
			  WANDER_GOAL_DIST );
    ngoal[2] = 0;
    SetWanderGoal(ngoal);
}

bool SimplePlayer::PickLongTermGoal(SimpleWorld *w)
{
    WayPointManager *wpm = w->GetWayPointManager();

    /*
    // where we are currently at
    uint32 sz = GetGoal() == NULL ? 0 : GetGoal()->neighbors.size();
    if (sz == 0) {
    // oops, no exits from here
    SetLongTermGoal( w->GetRandomWayPoint() );
    } else {
    SetLongTermGoal( goal->neighbors[(int)(drand48()*sz)] );
    }

    // TODO: follow shortest path!
    SetGoal( longTermGoal );
    */

    if (GetGoal() == NULL) {
	SetGoal( wpm->GetClosestWayPoint(GetOrigin()) );
    }

    const WayPoint *curr = GetGoal();

    /* TODO */
    list<const WayPoint *> visited;
    if (GetLongTermGoal() != NULL)
	visited.push_back(GetLongTermGoal());
    const WayPoint *g = wpm->GetNextGoal(GetOrigin(), &visited);
    ASSERT(g);
    ASSERT(g->IsGoal());
    path.clear();

    START(PickLongTerm::GetPath);
    bool found = wpm->GetPath(&path, curr, g);
    STOP(PickLongTerm::GetPath);

    sint32 tries = 5;
    while (!found && tries-- > 0) {
	g = wpm->GetRandomGoal();
	path.clear();

	START(PickLongTerm::GetPath);
	found = wpm->GetPath(&path, curr, g);
	STOP(PickLongTerm::GetPath);
    }

    if (!found) {
	// I appear to be stuck!
	WARN << "I am stuck! " << this << " committing suicide!" << endl;
	Suicide();
	return false;
    } else {
	SetLongTermGoal( g );
	SetGoal( *path.begin() );
	return true;
    }
}

void SimplePlayer::HandleFightingTransition(SimpleWorld *w, real32 delta)
{
    bool runAway = drand48() < PROB_RUN_AWAY;

    if (!runAway) {
	if (target != NULL && !w->IsVisible(this, target->GetOrigin()))
	    target = NULL;

	if (target == NULL) {
	    FindEnemy(w);
	}
    } else {
	target = NULL;
    }

    // either ran away or didn't see anyone
    if (target == NULL) {
	switch(GetState()) {
	case SP_FIGHTING_AT_WAYPOINT:
	    SetState(SP_AT_WAYPOINT);
	    break;
	case SP_FIGHTING_TO_WAYPOINT:
	    SetState(SP_TO_WAYPOINT);
	    break;
	default:
	    WARN << "unknown SimplePlayerState: " << GetState() << endl;
	}
    }
}

void SimplePlayer::HandleAtWaypointTransition(SimpleWorld *w, real32 delta)
{
    ASSERT(goal != NULL);

    bool leave = drand48() < PROB_LEAVE;

    // see if we want to leave this waypoint
    if (leave) {
	if (!PickLongTermGoal(w))
	    return;
	SetState(SP_TO_WAYPOINT);
    }

    bool fight = drand48() < PROB_FIGHT;

    // see if we want to pick a fight
    if (fight) {
	FindEnemy(w);
    }

    // transition to state
    if (target != NULL) {
	switch (GetState()) {
	case SP_AT_WAYPOINT:
	    SetState(SP_FIGHTING_AT_WAYPOINT);
	    break;
	case SP_TO_WAYPOINT:
	    SetState(SP_FIGHTING_TO_WAYPOINT);
	    break;
	default:
	    WARN << "unknown state: " << GetState() << endl;
	}
    }
}

void SimplePlayer::HandleToWaypointTransition(SimpleWorld *w, real32 delta)
{
    ASSERT(goal != NULL);
    ASSERT(longTermGoal != NULL);

    // see if we reached our short term goal
    if (GetGoal()->Contains(GetOrigin())) {
	if (GetGoal() == GetLongTermGoal()) {
	    // reached long term goal
	    PickWanderGoal(w);
	    SetState(SP_AT_WAYPOINT);
	} else {
	    // find where we are in the path toward the goal
	    while (path.size() > 0) {
		if ( *path.begin() == GetGoal() ) {
		    // reached this part in our path, take it off
		    path.pop_front();
		    break;
		}
		path.pop_front();
	    }
	    // if we didn't find ourselves, recalculate the path
	    if (path.size() == 0) {
		bool found = w->GetPath(&path, GetGoal(), GetLongTermGoal());
		if (!found) {
		    // we got lost somehow! pick a new long term goal
		    if (!PickLongTermGoal(w))
			return;
		}
		path.pop_front(); // pop where we are already at
	    }
	    ASSERT(path.size() > 0);
	    // set short term goal to next point in path
	    SetGoal( *path.begin() );
	}
    }

    bool fight = drand48() < PROB_FIGHT;

    // see if we want to pick a fight
    if (fight) {
	FindEnemy(w);
    }

    // transition to state
    if (target != NULL) {
	switch (GetState()) {
	case SP_AT_WAYPOINT:
	    SetState(SP_FIGHTING_AT_WAYPOINT);
	    break;
	case SP_TO_WAYPOINT:
	    SetState(SP_FIGHTING_TO_WAYPOINT);
	    break;
	default:
	    WARN << "unknown state: " << GetState() << endl;
	}
    }
}

void SimplePlayer::HandleFightingThink(SimpleWorld *w, real32 delta)
{
    ASSERT(target != NULL);

    Vec3 dir;

    if (GetOrigin().Distance( target->GetOrigin() ) <= FIGHT_FOLLOW_RADIUS) {
	// close-range combat mode -- random movement
	real32 c, angle;
	Vec3  facing;
	Vec3  adjust;

	// Randomly choose a movement direction
	c = drand48();

	// +y is forward, -x is left
	if (c < 0.2)
	    adjust[0] = -1;
	else if (c < 0.4)
	    adjust[0] = 1;

	c = drand48();

	if (c < 0.2)
	    adjust[1] = 1;
	else if (c < 0.4)
	    adjust[1] = -1;

	// face the target
	facing = target->GetOrigin() - GetOrigin();

	// do the adjustment move wrt the target
	if (facing != Vec3::ZERO)
	    angle = facing.Angle(Vec3(0,1,0));
	else
	    angle = 0;
	dir = WALK_VELOCITY*adjust.RotateAroundVector(Vec3(0,0,1), angle);
    } else {
	// long range combat mode -- follow target
	dir = WALK_VELOCITY*(target->GetOrigin() - GetOrigin()).Normalize();
    }

    if ( GetVelocity().Length() < WALK_VELOCITY ||
	 fabs( dir.Angle(GetVelocity()) ) > M_PI/360 ) {
	SetVelocity( dir );
    }

    Attack(target);
}

void SimplePlayer::EvasiveAction(SimpleWorld *w, real32 delta)
{
    // close-range combat mode -- random side-step
    real32 c, angle;
    Vec3  facing;
    Vec3  adjust;

    // Randomly choose a movement direction
    c = drand48();

    // +y is forward, -x is left
    if (c < 0.2)
	adjust[0] = -1;
    else if (c < 0.4)
	adjust[0] = 1;

    // do the adjustment move wrt the target
    Vec3 aim = GetVelocity();
    if (aim.Length() == 0) {
	aim = Vec3::Random();
	aim[2] = 0;
    }

    angle = aim.Angle(Vec3(0,1,0));
    Vec3 dir = WALK_VELOCITY*( aim.Normalize() + 
			       adjust.RotateAroundVector(Vec3(0,0,1), angle) );

    if ( GetVelocity().Length() < WALK_VELOCITY ||
	 fabs( dir.Angle(GetVelocity()) ) > M_PI/360 ) {
	SetVelocity( dir );
    }
}

void SimplePlayer::HandleAtWaypointThink(SimpleWorld *w, real32 delta)
{
    if (g_SimplePlayerWander) {

	// at a waypoint and not fighting, so wander around
	if (GetOrigin().Distance(wanderGoal) < WANDER_GOAL_RADIUS) {
	    PickWanderGoal(w);
	}

	Vec3 dir = GetWanderGoal() - GetOrigin();
	dir[2] = 0;
	dir = WALK_VELOCITY*dir.Normalize();

	// only change direction if the direction changed significantly
	// (allow deviations by 0.5 degrees)
	if ( GetVelocity().Length() < WALK_VELOCITY ||
	     fabs( dir.Angle(GetVelocity()) ) > M_PI/360 ) {
	    SetVelocity( dir );
	}
    } else {
	// wandering no enabled... just sit here then

	if ( GetVelocity().Length() > 0 ) {
	    SetVelocity( Vec3::ZERO );
	}
    }
}

void SimplePlayer::HandleToWaypointThink(SimpleWorld *w, real32 delta)
{
    // moving toward a short term goal
    ASSERT(goal != NULL);
    ASSERT(longTermGoal != NULL);

    Vec3 dir = goal->orig - GetOrigin();
    dir[2] = 0;
    dir = WALK_VELOCITY*dir.Normalize();

    // only change direction if the direction changed significantly
    // (allow deviations by 0.5 degrees)
    if ( GetVelocity().Length() < WALK_VELOCITY ||
	 fabs( dir.Angle(GetVelocity()) ) > M_PI/360 ) {
	SetVelocity( dir );
    }
}

void SimplePlayer::PerformPhysics(SimpleWorld *w, real32 delta)
{
    if (GetVelocity().Length() == 0)
	return;

    Vec3 new_origin = GetOrigin() + delta*GetVelocity();
    ASSERT ( !isnan(new_origin[0]) &&
	     !isnan(new_origin[1]) &&
	     !isnan(new_origin[2]) );

    Vec3 hit_pt;
    GameObject *hit_obj = NULL;

#if 0
    // with collission detection
    bool h = w->Fly(GetVolume(), GetOrigin(), new_origin, 
		    hit_pt, hit_obj, this);
#else
    // without collision detection
    bool h = w->FlyThrough(GetOrigin(), new_origin, hit_pt);
#endif

    if (h && !hit_obj) {
	ASSERT ( !isnan(hit_pt[0]) &&
		 !isnan(hit_pt[1]) &&
		 !isnan(hit_pt[2]) );
	new_origin = (hit_pt - 0.01*GetVelocity().Normalize());
	ASSERT ( !isnan(new_origin[0]) &&
		 !isnan(new_origin[1]) &&
		 !isnan(new_origin[2]) );
    }

    //if (GetOrigin() == new_origin) {
    //	INFO << "didn't move! hit=" << h << " hit_pt=" << hit_pt << endl;
    //}

    if (w->GetExtent().Contains(new_origin))
	SetOrigin(new_origin);
    //else
    //	INFO << "was outside world!" << endl;
}

void SimplePlayer::RunThink(GameManager *manager, real32 delta)
{
    SimpleWorld *w = static_cast<SimpleWorld *>( manager->GetWorld() );

    // find visible players
    FindVisible(w);

    // attack cool down
    if (nextAttack > 0) {
	nextAttack--;
    }

    // transition to another state if required
    switch (GetState()) {
    case SP_FIGHTING_AT_WAYPOINT:
    case SP_FIGHTING_TO_WAYPOINT:
	HandleFightingTransition(w, delta);
	break;
    case SP_AT_WAYPOINT:
	HandleAtWaypointTransition(w, delta);
	break;
    case SP_TO_WAYPOINT:
	HandleToWaypointTransition(w, delta);
	break;
    default:
	WARN << "unknown SimplePlayerState: " << GetState() << endl;
	ASSERT(0);
    }

    // perform what we need todo in this state
    switch (GetState()) {
    case SP_FIGHTING_AT_WAYPOINT:
    case SP_FIGHTING_TO_WAYPOINT:
	HandleFightingThink(w, delta);
	break;
    case SP_AT_WAYPOINT:
	HandleAtWaypointThink(w, delta);
	if (visibleMissiles > 0)
	    EvasiveAction(w, delta);
	break;
    case SP_TO_WAYPOINT:
	HandleToWaypointThink(w, delta);
	if (visibleMissiles > 0)
	    EvasiveAction(w, delta);
	break;
    default:
	WARN << "unknown SimplePlayerState: " << GetState() << endl;
	ASSERT(0);
    }

    //INFO << GetGUID() << " state: " << GetState() << " vel: " << GetVelocity() << endl;

    PerformPhysics(w, delta);

}

///////////////////////////////////////////////////////////////////////////////

void SimplePlayer::DoDamage(GameObject *source,
			    DynamicGameObject *inflictor, 
			    uint32 dmg)
{
    sint32 h = health;
    h -= dmg;
    if (h <= 0) {
	if (source && source->GetType() == SIMPLE_PLAYER) {
	    SimplePlayer *p = static_cast<SimplePlayer *>(source);
	    p->IncFrags();
	} else {
	    if (!source)
		WARN << GetGUID() << " was damaged by no source by " 
		     << inflictor << endl;
	}
	// dead!
	Respawn(GameManager::GetInstance()->GetWorld());
    } else {
	SetHealth((uint16)h);
    }
}

void SimplePlayer::Suicide()
{
    // lose a point for killing yourself!
    SetFrags( MAX( GetFrags()-1, 0 ) );
    Respawn(GameManager::GetInstance()->GetWorld());
}

void SimplePlayer::Respawn(GameWorld *w)
{
    SimpleWorld *e = static_cast<SimpleWorld *>( w );

    // XXX FIXME: if this point is too close to the edge, we will be stuck
    // half outside the world! (ok for now -- no collision detection)
    Vec3 start = e->GetWayPointManager()->GetRandomWayPoint()->GetOrigin();
    start[2] = 0;
    SetOrigin(start);
    SetVelocity(Vec3::ZERO);
    SetHealth(MAX_HEALTH);

    SetGoal( NULL );
    PickLongTermGoal(e);

    SetState(SP_TO_WAYPOINT);
}

void SimplePlayer::Attack(SimplePlayer *target)
{
    if (nextAttack > 0) {
	return;
    }

    Vec3 tgt = target->GetOrigin();
    // skew shots a little
    for (uint32 i=0; i<2; i++)
	tgt[i] += drand48()*32;

    Vec3 dir = tgt - GetOrigin();

    // in some wierd circumstances, two guys are in the same spot...
    // don't shoot in this case since we will hit ourself!
    if (dir == Vec3::ZERO)
	return;

    Vec3 odir = dir;
    dir = dir.Normalize();
    if (dir.Length() <= 0) {
	WARN << "Direction vector " << odir 
	     << " became 0-length (" << dir << ") after normalizing!" << endl;
	return;
    }

    Vec3 start = GetOrigin() + (10+32)*dir; // make sure start outside us

    // missile would have started outside world!
    if (!GameManager::GetInstance()->GetWorld()->GetExtent().Contains(start))
	return;

    SimpleMissile *m = new SimpleMissile(GameManager::GetInstance());
    m->SetOrigin(start); // make sure this starts outside us!
    m->SetOwner(this);
    m->SetDirection(dir);

    //DB(1) << "obj:     " << this << endl;
    //DB(1) << "target:  " << target << endl;
    //DB(1) << "missile: " << m << endl;

    GameManager::GetInstance()->GetStore()->Add( m );

    nextAttack = 5;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
