////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************

begin           : Nov 6, 2002
copyright       : (C) 2002-2005 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2002-2005 Jeffrey Pang       ( jeffpang@cs.cmu.edu )

***************************************************************************/

#include "SimpleTypeCodes.h"
#include "SimpleMovable.h"
#include "EmptyWorld.h"
#include "SimpleGame.h"
#include "SimpleWorld.h"
#include <gameapi/GameManager.h>

    static bool inited = false;
static DeltaMask initDeltaMask;

SimpleMovable::SimpleMovable(GameManager *manager, gTypeCode t) :
    DynamicGameObject(t, manager), deleted(false)
{
    SetPubInterval(SimpleGame::pubInterval);
}

SimpleMovable::SimpleMovable(GObjectInfoIface *info, gTypeCode t) :
    DynamicGameObject(t, info), deleted(false)
{
    SetPubInterval(SimpleGame::pubInterval);
}

SimpleMovable::~SimpleMovable()
{}

void SimpleMovable::FillEvents(EventList *reg, EventList *unreg, EventSet *curr)
{
    if (SimpleGame::publishAsDHTRegions) {
	TimeVal now = GameManager::GetInstance()->ThisFrameTime();
	if (m_LastPubTime + m_PubInterval >= now) {
	    return;
	}
	m_LastPubTime = now;

	SimpleGame  *g = 
	    static_cast<SimpleGame *>(GameManager::GetInstance()->GetModule());
	SimpleWorld *w = 
	    static_cast<SimpleWorld *>(GameManager::GetInstance()->GetWorld());

	// must convert the events to DHT keyed events
	if (curr) {
	    for (EventSetIter it = curr->begin(); it != curr->end(); it++) {
		unreg->push_back(*it);
	    }
	}

	OMEvent *ev = 
	    GameManager::GetInstance()->GetManager()->CreateEvent(GetGUID());
	real32 key_val = g->GetHasher()->Hash( w->GetRegion(m_Origin) );
	// XXX HACK -- This only works for this hacky case
	ASSERT( g->GetHasher()->GetKeyDim() == 0 );
	uint32 key_dim = g->GetHasher()->GetKeyDim();
	Manager *m = GameManager::GetInstance()->GetManager();
	BBox ext = w->GetExtent();
	Value v = m->ConvertToValue(key_dim, 
				    key_val, 
				    ext.min[key_dim], 
				    ext.max[key_dim]);
	ev->AddTuple( Tuple( key_dim, v ) );

	m_Origin.FillEvent(ev);

	//INFO << "publishing: " << ev << endl;

	ev->SetLifeTime( m_PubInterval );
	reg->push_back(ev);
    } else {
	DynamicGameObject::FillEvents(reg, unreg, curr);
    }
}

void SimpleMovable::GetPredictedAreaOfInterest(list<BBox> *toFill, 
					       GameWorld *world)
{
    const PredictiveParams *params = &SimpleGame::predictionParams;

    DynamicGameObject::GetAreaOfInterest(toFill, world);

    // XXX Incorporate this code with PredictiveInterestFilter
    real32 fudge_ext = MAX( 0,
			    // the amount of "fudge" we have to add is the
			    // velocity of this object + the velocity of any
			    // other object moving toward one another
			    // times the amount of time that might elapse
			    // between an object changing position and us
			    // being notified of that change (= inverse of
			    // the rate of publications + how long it takes
			    // the publication to get delivered to us)
			    // - how far the publications themselves are
			    // predicted (since they themselves are fudged)
			    (params->maxVelocity/*+m_ObjVelocity*/)*
			    (params->pubInterval+params->maxDiscoveryLat)/
			    params->timeUnit - params->locRadius );

    // expand the bbox by the fudge to get the predicted bbox
    for (list<BBox>::iterator it = toFill->begin(); 
	 it != toFill->end(); it++) {
	*it = world->Clip( BBox((*it).min - fudge_ext,
				(*it).max + fudge_ext) );
    }
}

uint32 SimpleMovable::IsInterested(const Vec3& pt, gTypeCode t)
{
    return 0;
}

uint32 SimpleMovable::IsInterested(const BBox& vol, gTypeCode t)
{
    return 0;
}

void SimpleMovable::RunThink(GameManager *manager, real32 delta)
{
    //SetOrigin(GetOrigin() + delta*velocity);
    Vec3 new_origin = GetOrigin() + delta*velocity;
    EmptyWorld *w = static_cast<EmptyWorld *>( manager->GetWorld() );

    Vec3 hit_pt;
    GameObject *hit_obj = NULL;

    bool h = w->Fly(GetVolume(), GetOrigin(), new_origin, 
		    hit_pt, hit_obj, this);

    if (h) {
	new_origin = (hit_pt - 0.01*velocity.Normalize());
    }
    if (w->GetExtent().Contains(new_origin))
	SetOrigin(new_origin);
}

const DeltaMask& SimpleMovable::GetInitDeltaMask()
{
    if (!inited) {
	initDeltaMask = DynamicGameObject::GetInitDeltaMask();
	for (int i=0; i<3; i++) {
	    initDeltaMask.Set( DynamicGameObject::GetDeltaMaskBits() + 1 );
	}
	inited = true;
    }
    return initDeltaMask;
}

void SimpleMovable::PackUpdate(Packet *buffer, const DeltaMask& mask)
{
    DynamicGameObject::PackUpdate(buffer, mask);
    for (int i=0; i<3; i++) {
	if (mask.IsSet(DynamicGameObject::GetDeltaMaskBits()+i)) {
	    buffer->WriteFloat(velocity[i]);
	}
    }
}

void SimpleMovable::UnpackUpdate(Packet *buffer, const DeltaMask& mask, 
				 list<GameObjectRef *> *unresolved)
{
    DynamicGameObject::UnpackUpdate(buffer, mask, unresolved);
    for (int i=0; i<3; i++) {
	if (mask.IsSet(DynamicGameObject::GetDeltaMaskBits()+i)) {
	    velocity[i] = buffer->ReadFloat();
	}
    }
}

uint32 SimpleMovable::GetRegion()
{
    SimpleWorld *w = 
	static_cast<SimpleWorld *>(GameManager::GetInstance()->GetWorld());

    return w->GetRegion( GetOrigin() );
}

void SimpleMovable::PrintFields(ostream& out)
{
    DynamicGameObject::PrintFields(out);
    out << " velocity=" << velocity;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
