////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

#ifndef QUAD_TREE
#define QUAD_TREE

#include <algorithm>
#include <gameapi/common.h>

// XXX XXX XXX XXX DOESN'T WORK YET!!! XXX XXX XXX
// 
// class T should implement "const Vec3& GetOrigin()"
template<class T>
class QuadTree
{
 private:
    uint32 d1, d2;
    real32 fill;
    uint32 size, k;
    BBox extent;
    bool isLeaf;
    vector<T *> objs;
    QuadTree * parent;
    QuadTree * subTree[4];

    real32 Distance(const Vec3& a, const Vec3& b) const;
    T *FindLocalClosest(const Vec3& pt) const;
    void Merge();
    void ConsiderSplit();
 public:
    QuadTree(const BBox& extent, uint32 k=16, real32 fill=0.5, 
	     uint32 d1=0, uint32 d2=0);
    ~QuadTree();

    T *FindNearestNeighbor(const Vec3& pt) const;

    void Insert(T *obj);
    bool Delete(T *obj); // must satisfy == for deletion
};

///////////////////////////////////////////////////////////////////////////////

template<class T>
QuadTree<T>::QuadTree(const BBox& extent, uint32 k, real32 fill, 
		      uint32 d1, uint32 d2) :
    extent(extent), k(k), fill(fill), 
     d1(d1), d2(d2), parent(NULL), isLeaf(true)
{
    for (int i=0; i<4; i++)
	subTree[i] = NULL;
}

template<class T>
QuadTree<T>::~QuadTree()
{
    for (int i=0; i<4; i++)
	delete subTree[i];
}

template<class T>
real32 QuadTree<T>::Distance(const Vec3& a, const Vec3& b) const
{
    return sqrt( (a[d1]-b[d1])*(a[d1]-b[d1]) + 
		 (a[d2]-b[d2])*(a[d2]-b[d2]) );
}

template<class T>
T *QuadTree<T>::FindLocalClosest(const Vec3& pt) const
{
    ASSERT(isLeaf);

    real32 min = INFINITY;
    T *ret = NULL;

    for (uint32 i=0; i<objs.size(); i++) {
	real32 dist = Distance(pt, objs[i]->GetOrigin());
	if (dist < min) {
	    min = dist;
	    ret = objs[i];
	}
    }

    return ret;
}

template<class T>
struct _qdist {
    QuadTree<T> *t;
    real32 dist;
};

template<class T>
bool _qdist_less(const _qdist<T>& a, const _qdist<T>&b) {
    return a.dist < b.dist;
}

template<class T>
T *QuadTree<T>::FindNearestNeighbor(const Vec3& pt) const
{
    if (isLeaf) {
	return FindLocalClosest(pt);
    }

    _qdist<T> order[4];

    for (int i=0; i<4; i++) {
	order[i].t = subTree[i];

	real32 p1 = pt[d1];
	real32 p2 = pt[d2];

	// find closest point
	bool in1 = subTree[i]->extent.min[d1] < p1 && 
	    p1 <= subTree[i]->extent.max[d1];
	bool in2 = subTree[i]->extent.min[d2] < p2 && 
	    p2 <= subTree[i]->extent.max[d2];
	if (in1 && in2) {
	    // contains point;
	    order[i].dist = 0;
	} else if (in1) {
	    // closest face is on dim1
	    order[i].dist = MIN( fabs(p1 - subTree[i]->extent.min[d1]),
				 fabs(p1 - subTree[i]->extent.max[d1]) );
	} else if (in2) {
	    // closest face is on dim2
	    order[i].dist = MIN( fabs(p1 - subTree[i]->extent.min[d2]),
				 fabs(p1 - subTree[i]->extent.max[d2]) );
	} else {
	    order[i].dist = INFINITY;
	    // closest point is a corner
	    for (uint32 x=0; x<2; x++) {
		real32 dim1 = x==0 ? subTree[i]->extent.min[d1] : 
		    subTree[i]->extent.max[d1];
		for (uint32 y=0; y<2; y++) {
		    real32 dim2 = x==0 ? subTree[i]->extent.min[d2] : 
			subTree[i]->extent.max[d2];
		    real32 dist = Distance(pt, Vec3(dim1, dim2, pt[2]));
		    order[i].dist = MIN(dist, order[i].dist);
		}
	    }
	}
    }

    stable_sort(&order[0], &order[4], _qdist_less<T>);

    real32 min = INFINITY;
    T *obj = NULL;

    // now we know the order of subtrees to visit and their respective
    // min distances to the point... traverse!
    for (uint32 i=0; i<4; i++) {
	if (min < order[i].dist)
	    break;

	obj = order[i].t->FindNearestNeighbor(pt);
    }

    return obj;
}

template<class T>
void QuadTree<T>::ConsiderSplit()
{
    if (!isLeaf || size <= k)
	return;

    // XXX TODO: split into even sized pieces by obj weight
    real32 mid1 = extent.min[d1] + (extent.max[d1] - extent.min[d1])/2;
    real32 mid2 = extent.min[d2] + (extent.max[d2] - extent.min[d2])/2;

    for (uint32 i=0; i<2; i++) {
	real32 min1 = i==0 ? extent.min[d1] : mid1;
	real32 max1 = i==0 ? mid1 : extent.max[d1];
	for (uint32 j=0; j<2; j++) {
	    real32 min2 = j==0 ? extent.min[d2] : mid2;
	    real32 max2 = j==0 ? mid2 : extent.max[d2];

	    Vec3 min = extent.min;
	    Vec3 max = extent.max;
	    min[d1] = min1;
	    min[d1] = min2;
	    max[d2] = max1;
	    max[d2] = max2;

	    subTree[i*2+j] = new QuadTree( BBox(min, max), k, fill, d1, d2 );
	    subTree[i*2+j]->parent = this;
	}
    }

    isLeaf = false;

    for (uint32 i = 0; i<objs.size(); i++) {
	for (uint32 j=0; j<4; j++) {
	    if (subTree[j]->extent.Contains(objs[i]->GetOrigin())) {
		subTree[j]->objs.push_back(objs[i]);
	    }
	}
    }
    objs.clear();

    // hopefully will never go down here, but could end up with bad split
    // if distribution is skewed in euclidean space
    for (uint32 i = 0; i<4; i++) {
	subTree[i]->ConsiderSplit();
    }
}

template<class T>
void QuadTree<T>::Merge()
{
    ASSERT(size <= fill*k);
    ASSERT(!isLeaf);

    for (uint32 i=0; i<4; i++) {
	ASSERT(subTree[i]->isLeaf);
	for (uint32 j=0; j<subTree[i].objs.size(); j++) {
	    objs.push_back(subTree[i].objs[j]);
	}
	delete subTree[i];
	subTree[i] = NULL;
    }
    ASSERT(objs.size() == size);

    isLeaf = true;
}

template<class T>
void QuadTree<T>::Insert(T *obj)
{
    if (!isLeaf) {
	for (uint32 i=0; i<4; i++) {
	    if (subTree[i]->extent.Contains(obj->GetOrigin())) {
		subTree[i]->Insert(obj);
		size++;
	    }
	}
	return;
    }

    objs.push_back(obj);
    size++;    
    ConsiderSplit();
}

template<class T>
bool QuadTree<T>::Delete(T *obj)
{
    if (!isLeaf) {
	for (uint32 i=0; i<4; i++) {
	    if (subTree[i]->extent.Contains(obj->GetOrigin())) {
		bool deleted = subTree[i]->Delete(obj);
		if (deleted) {
		    if (deleted) {
			size--;
			if (size <= fill*k) {
			    // node shrunk enough to remerge
			    Merge();
			}
		    }
		}
	    }
	    return;
	}
    }

    // at leaf, look for and delete obj
    uint32 i;
    for (i=0; i<objs.size(); i++) {
	if (objs[i] == obj) {
	    break;
	}
    }
    if (i < objs.size()) {
	// shift everything back 1
	for ( ; i<objs.size()-1; i++) {
	    objs[i] = objs[i+1];
	}
	// delete last slot
	objs.pop_back();
	size--;

	return true;
    }

    return false;
}

#endif
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
