////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************

begin           : Nov 6, 2002
copyright       : (C) 2002-2005 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2002-2005 Jeffrey Pang       ( jeffpang@cs.cmu.edu )

***************************************************************************/

#ifndef __MERGING_INTEREST_FILTER_H__
#define __MERGING_INTEREST_FILTER_H__

#include <util/refcnt.h>
#include <gameapi/GameInterestFactory.h>

    class TimedBBox : public BBox, public TimedStruct
{
 public:
    TimedBBox() : BBox(), TimedStruct((uint32)0) {}
    TimedBBox(const BBox& box, uint32 ttl) : BBox(box), TimedStruct(ttl) {}
};

class MergedBBox : public BBox
{
    uint32 m_MaxTTL;
    real32 m_CompositeVolumeTTLProduct;
    list<AnnotatedBBox *> m_Boxes;
 public:
    MergedBBox() : BBox(), m_CompositeVolumeTTLProduct(0), m_MaxTTL(0) {}
    virtual ~MergedBBox () {}
    void Merge(AnnotatedBBox *box) {
	if (m_Boxes.size() == 0) {
	    for (uint32 i=0; i<3; i++) {
		min[i] = box->GetBBox()->min[i];
		max[i] = box->GetBBox()->max[i];
	    }
	} else {
	    for (uint32 i=0; i<3; i++) {
		min[i] = MIN(min[i], box->GetBBox()->min[i]);
		max[i] = MAX(max[i], box->GetBBox()->max[i]);
	    }
	}
	m_Boxes.push_back(box);

	// The composite volume is the volume we subscribe to over
	// a period if time; this allows us to take into account both
	// volume and time to live in our expansion factor.
	//
	// XXX:
	// Note that we count overlap of composite volume as "double"
	// the actual volume. This actually overestimates the composite
	// volume since some of it is overlapped. But the "cost" of
	// overlap is not quite free since two subs that overlap will
	// still have to both be routed and stored. Matches however,
	// will only be sent ~once for each...
	m_CompositeVolumeTTLProduct += 
	    box->GetBBox()->Volume()*box->GetTTL();
	m_MaxTTL = MAX(m_MaxTTL, box->GetTTL());
    }
    list<AnnotatedBBox *> *GetBoxes() {
	return &m_Boxes;
    }
    real32 CompositeVolumeTTLProduct() {
	return m_CompositeVolumeTTLProduct;
    }
    uint32 MaxTTL() {
	return m_MaxTTL;
    }
};

/**
 * This is a global interest filter which merges interests that
 * overlap enough. It keeps track of old interests also, so if
 * they are filtered out after this stage, this filter may not
 * function properly.
 */
class MergingInterestFilter : public InterestFilter {
 private:

    // how much we are allowed to let the volume in the bboxes expand by
    real32 m_ExpansionFactor;

    // old bboxes we sent out into the system that have not expired
    list<TimedBBox> m_History;

    // expire any interest that only has this much time left
    // (idea: most new subs have ttls longer than this, so no point
    // trying to match against these)
    static const uint32 EXPIRED_TIME_LEFT = 1000;

    void ExpireOldInterests();
    void FilterOutCoveredInterests(AreaOfInterest *out,
				   AreaOfInterest *in);
    void SplitBBox_PickSeeds(AnnotatedBBox *& seed1,
			     AnnotatedBBox *& seed2,
			     list<AnnotatedBBox *> *boxes,
			     ref<MergedBBox> orig);
    void SplitBBox(list< ref<MergedBBox> > *out, 
		   ref<MergedBBox> in);
    void RefineBBox(list< ref<MergedBBox> > *out, 
		    ref<MergedBBox> in);
    void MergeInterests(AreaOfInterest *out,
			AreaOfInterest *in);
    void SaveNewInterests(AreaOfInterest *out);

    void CopyThrough(AreaOfInterest *out,
		     AreaOfInterest *in);

 public:

    MergingInterestFilter(real32 expansionFactor);
    virtual ~MergingInterestFilter();

    AreaOfInterest *Filter(AreaOfInterest *in);
    void Delete(AreaOfInterest *aoi);
};

#endif
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
