////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

#include "MergingInterestFilter.h"
#include "SimpleGame.h"

MergingInterestFilter::MergingInterestFilter(real32 expansionFactor) : 
    m_ExpansionFactor(expansionFactor) {}

MergingInterestFilter::~MergingInterestFilter() {}

void MergingInterestFilter::ExpireOldInterests()
{
    // XXX TODO: make more efficient?
    for (list<TimedBBox>::iterator it = m_History.begin();
	 it != m_History.end(); /* !!! */) {
	list<TimedBBox>::iterator oit = it;
	oit++;

	if ((*it).TimeLeft() < EXPIRED_TIME_LEFT) {
	    m_History.erase(it);
	}
	it = oit;
    }
}

void MergingInterestFilter::FilterOutCoveredInterests(AreaOfInterest *out,
						      AreaOfInterest *in)
{
    // XXX: Make this more efficient! This is O(m*n) right now
    for (AreaOfInterestIter it = in->begin(); it != in->end(); it++) {
	AnnotatedBBox *inbox = *it;
	bool covered = false;

	for (list<TimedBBox>::iterator hist = m_History.begin();
	     hist != m_History.end(); hist++) {
	    const TimedBBox& oldbox = *hist;

	    // XXX: Is there an efficient way to check if multiple
	    // old bboxes cover a new one?
	    if ( oldbox.TimeLeft() >= inbox->GetTTL() &&
		 oldbox.Contains(*inbox->GetBBox()) ) {
		//INFO << *inbox->GetBBox() << " covered by: "
		//     << oldbox << endl;

		covered = true;
		break;
	    }
	}

	if (!covered) {
	    out->push_back(inbox);
	}
    }
}

void MergingInterestFilter::CopyThrough(AreaOfInterest *out,
					AreaOfInterest *in)
{
    for (AreaOfInterestIter it = in->begin(); it != in->end(); it++) {
	AnnotatedBBox *inbox = *it;
	out->push_back(new AnnotatedBBox(*inbox));
    }
}

void MergingInterestFilter::SplitBBox_PickSeeds(AnnotatedBBox *& seed1,
						AnnotatedBBox *& seed2,
						list<AnnotatedBBox *> *boxes,
						ref<MergedBBox> orig)
{
    AnnotatedBBox *maxLow[3];
    AnnotatedBBox *minHigh[3];
    bzero(&maxLow, sizeof(maxLow));
    bzero(minHigh, sizeof(minHigh));

    ASSERT(boxes->size() > 1);

    // find the two boxes with greatest separation along some dimension
    for (list<AnnotatedBBox *>::iterator it = boxes->begin();
	 it != boxes->end(); it++) {
	AnnotatedBBox *box = *it;

	for (int i=0; i<3; i++) {
	    if (maxLow[i] == NULL) {
		maxLow[i] = box;
	    } else {
		if (maxLow[i]->GetBBox()->min[i] < box->GetBBox()->min[i])
		    maxLow[i] = box;
	    }

	    if (minHigh[i] == NULL) {
		minHigh[i] = box;
	    } else {
		if (minHigh[i]->GetBBox()->max[i] > box->GetBBox()->max[i])
		    minHigh[i] = box;
	    }
	}
    }

    // from Ashwin:
    // ugh; complication. undocumented in Gutt84!
    // happens when there's a rectangle which is 
    // enclosed entirely by all other rectangles.
    // 
    // i dont have a good solution. will just choose 
    // a random rectangle in this case.
    for (uint32 i=0; i<3; i++) {
	if (maxLow[i] == minHigh[i]) {
	    for (list<AnnotatedBBox *>::iterator it = boxes->begin();
		 it != boxes->end(); it++) {
		if (*it == maxLow[i])
		    continue;
		minHigh[i] = *it;
		break;
	    }
	}
	ASSERT(maxLow[i] != minHigh[i]);
    }

    sint32 dim = -1;
    real32 min_sep = 0;

    for (int i=0; i<3; i++) {
	real32 normalized_sep = 
	    (maxLow[i]->GetBBox()->min[i] - 
	     minHigh[i]->GetBBox()->max[i]) / (orig->max[i]-orig->min[i]);
	if (dim == -1 || normalized_sep > min_sep) {
	    dim = i;
	    min_sep = normalized_sep;
	}
    }

    ASSERT(dim != -1);
    seed1 = maxLow[dim];
    seed2 = minHigh[dim];
}

void MergingInterestFilter::SplitBBox(list< ref<MergedBBox> > *out, 
				      ref<MergedBBox> in)
{
    // split the incoming bbox into two pieces formed by its
    // composites that minimizes the coverage of each sub-part. This
    // algorithm uses the simple linear-time splitting algorithm from
    // [Guttman84]. At some point we might want to to the more
    // sophisticated linear time algorithm from [Ang/Tan97]

    ref<MergedBBox> B1 = new refcounted<MergedBBox>();
    ref<MergedBBox> B2 = new refcounted<MergedBBox>();

    list<AnnotatedBBox *> *boxes = in->GetBoxes();

    AnnotatedBBox *seed1 = NULL, *seed2 = NULL;
    SplitBBox_PickSeeds(seed1, seed2, boxes, in);
    ASSERT(seed1);
    ASSERT(seed2);
    ASSERT(seed1 != seed2);
    B1->Merge(seed1);
    B2->Merge(seed2);

    for (list<AnnotatedBBox *>::iterator it = boxes->begin();
	 it != boxes->end(); it++) {
	AnnotatedBBox *next = *it;
	if (next == seed1 || next == seed2) continue;

	real32 vol1 = B1->GetMerged(*next->GetBBox()).Volume();
	real32 vol2 = B2->GetMerged(*next->GetBBox()).Volume();
	if (vol1 < vol2) {
	    B1->Merge(next);
	}
#ifndef NSDI2006_MERGING_INTEREST_BUG 
	else {
	    B2->Merge(next);
	}
#endif
    }

    out->push_back(B1);
    out->push_back(B2);
}

void MergingInterestFilter::RefineBBox(list< ref<MergedBBox> > *out, 
				       ref<MergedBBox> in)
{
    // This algorithm runs in worst case O(n\lg n)
    real32 totalVol = in->Volume();
    uint32 totalTTL = in->MaxTTL();
    real32 compositeProduct = in->CompositeVolumeTTLProduct();

    if (totalVol*totalTTL <= m_ExpansionFactor*compositeProduct) {
	// box is not too big, let's keep it
	out->push_back(in);
	return;
    }
    //INFO << "first: " << *(*in->GetBoxes()->begin())->GetBBox() << endl;
    //INFO << "this:  " << *in << endl;
    ASSERT(in->GetBoxes()->size() > 1);

    // box is too big, split it and try again on each piece.
    list< ref<MergedBBox> > split;
    SplitBBox(&split, in);

    for (list< ref<MergedBBox> >::iterator it = split.begin();
	 it != split.end(); it++) {
	RefineBBox(out, *it);
    }
}

void MergingInterestFilter::MergeInterests(AreaOfInterest *out,
					   AreaOfInterest *in)
{
    // make the largest merged box
    ref<MergedBBox> input = new refcounted<MergedBBox>();
    for (AreaOfInterestIter it = in->begin(); it != in->end(); it++) {
	input->Merge(*it);
    }

    // refine it
    list< ref<MergedBBox> > output;
    RefineBBox(&output, input);

    for (list< ref<MergedBBox> >::iterator it = output.begin();
	 it != output.end(); it++) {
	ref<MergedBBox> merged = *it;

	//INFO << "merged box: " << *(BBox *)merged << endl;

	AnnotatedBBox *obox = new AnnotatedBBox( *merged,
						 GUID_NONE,
						 merged->MaxTTL() );

	list<AnnotatedBBox *> *composite = merged->GetBoxes();
	for (list<AnnotatedBBox *>::iterator comp = composite->begin();
	     comp != composite->end(); comp++) {
	    obox->AddGUIDs( (*comp)->GetGUIDs() );
	}
	out->push_back(obox);
    }
}

void MergingInterestFilter::SaveNewInterests(AreaOfInterest *boxes)
{
    for (list<AnnotatedBBox *>::iterator it = boxes->begin();
	 it != boxes->end(); it++) {
	AnnotatedBBox *box = *it;

	m_History.push_back( TimedBBox(*box->GetBBox(), box->GetTTL()) );
    }
}

AreaOfInterest *MergingInterestFilter::Filter(AreaOfInterest *in)
{
    //INFO << "in=" << in->size() << endl;

    // get rid of historical interests that have expired
    ExpireOldInterests();

    // filter out all new interests that are fully covered by old ones
    AreaOfInterest inFiltered;
    FilterOutCoveredInterests(&inFiltered, in);

    //INFO << "filtered=" << inFiltered.size() << endl;

    // merge all the new interests that can be merged together into one blob
    AreaOfInterest *out = new AreaOfInterest();;

    if (inFiltered.size() == 0)
	return out;

    // XXX HACK: what we really should do is separate the saving of
    // old interests and the merging of interests into different filters
    // and only use the former for the DHT
    if (SimpleGame::publishAsDHTRegions) {
	CopyThrough(out, &inFiltered);
    } else {
	MergeInterests(out, &inFiltered);
    }

    /*
      for(AreaOfInterestIter it = inFiltered.begin();
      it !=inFiltered.end(); it++) {
      out->push_back(new AnnotatedBBox(**it));
      }
    */

    ASSERT(out->size() > 0);

    // save new historical interests
    SaveNewInterests(out);

    //INFO << "out=" << out->size() << endl;

    return out;
}

void MergingInterestFilter::Delete(AreaOfInterest *aoi)
{
    for (AreaOfInterestIter it = aoi->begin(); it != aoi->end(); it++) {
	delete *it;
    }
    delete aoi;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
