////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************

begin           : Nov 6, 2002
copyright       : (C) 2002-2005 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2002-2005 Jeffrey Pang       ( jeffpang@cs.cmu.edu )

***************************************************************************/

#include <mercury/common.h>
#include <fstream>
#include <gameapi/GameManager.h>
#include <gameapi/GameModule.h>
#include "FileDataSource.h"

    void FDS_Loader::Run()
{
    while (1) {
	FDS_Info next = todo.Remove();
	if (next.type == FDS_Info::LOAD) {
	    string filename = dir + "/" + next.guid.ToString();
	    ifstream f(filename.c_str());
	    int len;
	    f.read(reinterpret_cast<char *>(&len), sizeof(len));
	    Packet pkt(len);
	    f.read(reinterpret_cast<char *>(pkt.GetBuffer ()), len);
	    GameObject *obj;
	    if (f.fail()) {
		obj = NULL;
	    } else {
		GameManager *m = GameManager::GetInstance();
		obj = m->GetModule()->ConstructObject(&pkt);
	    }
	    f.close();
	    FDS_Info info(obj == NULL ? FDS_Info::ERROR : FDS_Info::OK, 
			  obj, next.tok);
	    done.Insert(info);
	} else {
	    string filename = dir + "/" + next.obj->GetGUID().ToString();
	    ofstream f(filename.c_str());
	    int len = next.obj->GetLength();
	    Packet pkt(len);
	    next.obj->Serialize(&pkt);
	    f.write(reinterpret_cast<char *>(&len), sizeof(len));
	    f.write(reinterpret_cast<char *>(pkt.GetBuffer ()), len);
	    FDS_Info::FDS_Type t;
	    if (f.fail()) {
		t = FDS_Info::ERROR;
	    } else {
		t = FDS_Info::OK;
	    }
	    f.close();
	    FDS_Info info(t, NULL, next.tok);
	}

	// separated this out so that we can synchronize it in another thread
	// if desired...
	DoWork();
    }
}

void FDS_Loader::DoWork()
{
    while (done.Size() > 0) {
	FDS_Info next = done.Remove();
	ASSERT(next.tok);
	if (next.obj) {
	    src->lock.Acquire();
	    src->cache[next.obj->GetGUID()] = next.obj;
	    src->pending_loads.erase(next.obj->GetGUID());
	    src->lock.Release();
	}
	next.tok->Signal(next.type == FDS_Info::OK ? GAME_OK : GAME_ERROR);

	src->lock.Acquire();
	if (src->todelete.find(next.tok) != src->todelete.end()) {
	    src->todelete.erase(next.tok);
	    delete next.tok;
	}
	src->waiting.erase(next.tok);
	src->lock.Release();
    }
}

FileDataSource::FileDataSource(const char *dir, 
			       const GUID& min, 
			       const GUID& max) :
    min(min), max(max)
{
    loader = new FDS_Loader(dir, this);
    loader->Start();
}

FileDataSource::~FileDataSource() {
    delete loader;
}

bool FileDataSource::CanLoad(const GUID& globalID)
{
    return true;
    // XXX TODO
    //return globalID >= min && globalID <= max;
}

bool FileDataSource::CanStore(GameObject *obj)
{
    return true;
    // XXX TODO
    //return globalID >= min && globalID <= max;
}

EventToken *FileDataSource::Preload(const GUID& globalID, uint32 prio)
{
    EventToken *tok = new EventToken();

    // XXX assuming that we won't call preload on guid twice while in progress!
    lock.Acquire();
    if (cache.find(globalID) != cache.end()) {
	tok->Signal(GAME_OK);
    } else {
	waiting.insert(tok);
	pending_loads[globalID] = tok;
	FDS_Info info(FDS_Info::LOAD, globalID, tok);
	loader->todo.Insert(info);
    }
    lock.Release();

    return tok;
}

GameObject *FileDataSource::Load(const GUID& globalID, uint32 prio)
{
    lock.Acquire();
    if (pending_loads.find(globalID) != pending_loads.end()) {
	EventToken *tok = pending_loads[globalID];
	lock.Release();
	tok->Wait();
	lock.Acquire();
    }
    if (cache.find(globalID) == cache.end()) {
	EventToken tok;
	FDS_Info info(FDS_Info::LOAD, globalID, &tok);
	loader->todo.Insert(info);
	lock.Release();
	tok.Wait();
	lock.Acquire();
	if (tok.GetError() != GAME_OK) {
	    lock.Release();
	    return NULL;
	}
    }

    GameObjectCacheIter p = cache.find(globalID);
    ASSERT(p != cache.end());
    GameObject *ret = p->second;
    cache.erase(p);
    lock.Release();

    return ret;
}

EventToken *FileDataSource::Store(GameObject *obj, uint32 prio)
{
    EventToken *tok = new EventToken();
    waiting.insert(tok);
    FDS_Info info(FDS_Info::STORE, obj, tok);
    loader->todo.Insert(info);
    return tok;
}

void FileDataSource::FreeToken(EventToken *tok)
{
    lock.Acquire();
    if (waiting.find(tok) == waiting.end()) {
	delete tok;
    } else {
	todelete.insert(tok);
    }
    lock.Release();
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
