////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************
  GObject.h

  Basic GameObject.

begin           : Nov 6, 2002
copyright       : (C) 2002-2005 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2002-2005 Jeffrey Pang       ( jeffpang@cs.cmu.edu )

***************************************************************************/

#ifndef __GAME_OBJECT_H__
#define __GAME_OBJECT_H__

#include <mercury/common.h>
#include <util/debug.h>
#include <gameapi/common.h>
#include <om/GObject.h>
#include <gameapi/GameWorld.h>

    class GameManager;
class GameObjectRef;
class GameObject;
class StaticGameObject;
class DynamicGameObject;

///////////////////////////////////////////////////////////////////////////////

typedef map<GameObjectRef *, GameObject *, less_ptr> RefMap;
typedef RefMap::iterator RefMapIter;
typedef multimap<GameObject *, GameObjectRef *, less_ptr> BackRefMap;
typedef BackRefMap::iterator BackRefMapIter;

/**
 * Instead of directly pointing to other objects, should use
 * subclasses of this instead so that distributed objects will be
 * resolved correctly. This reference class does two things:
 *
 * (1) When an object is deleted, all references
 *     (of this class) to it automatically get set to NULL.
 *
 * (2) When an object pointer is serialized or deserialized, it 
 *     will be resolved appropriately from remote hosts.
 */
class GameObjectRef : public GObjectRef, public Serializable {
 private:
    friend class GameObject;
    friend class DynamicGameObject;

    static RefMap m_Refs;
    static BackRefMap m_BackRefs;

 protected:
    /**
     * Register a reference. When this object is removed from the
     * store, all references to it will automatically be set to NULL.
     */
    static void RegisterRef(GameObjectRef *ptr);
    static void DeleteRef(GameObjectRef *ptr);
    static void DeleteObjRefs(GameObject *obj);

    // these are just cached values -- they may be stale
    GUID guid;
    SID  owner;
    // this is authoritative, if non-NULL
    GameObject *target;

    virtual void Resolve(GameObject *obj);
    GameObject *GetObject() { return target; }
    void Nullify() {
	guid = GUID_NONE;
	owner = SID_NONE;
	target = NULL;
    }
 public:
    static const GameObjectRef Null;

    GameObjectRef() : guid(GUID_NONE), owner(SID_NONE), target(NULL) {}
    GameObjectRef(GameObject *target);
    GameObjectRef(const GameObjectRef& other);
    GameObjectRef(Packet *pkt);
    virtual ~GameObjectRef();
    GameObjectRef& operator=(const GameObjectRef& other);
    inline bool operator==(const GameObjectRef& other) {
	if (target || other.target) {
	    return target == other.target;
	} else {
	    return guid == other.guid;
	}
    }
    inline bool operator!=(const GameObjectRef& other) {
	return !( *this == other ); 
    }

    bool IsNull();
    void SetNull();
    GUID GetTarget();
    SID  GetOwner();
    bool IsResolved();

    inline GameObject *operator->() {
	ASSERT(IsResolved());
	return target;
    }
    inline operator GameObject *() {
	ASSERT(IsResolved());
	return target;
    }

    void Serialize(Packet *pkt);
    uint32 GetLength();
    void Print(FILE *) { ASSERT(0); }
};

typedef set<GameObjectRef *> GameObjectRefSet;
typedef GameObjectRefSet::iterator GameObjectRefSetIter;

/**
 * This is a useful utility wrapper for game pointers to use the applications
 * self-defined GameObject classes. For example, if you have a Missle subclass
 * then a pointer to a Missile would be gRef<Missile>.
 *
 * XXX FIXME: Right now this class is inherently unsafe under serialization
 * and pointer resolution because there is no check that the resolved object
 * is of the correct sub-type!
 */
template<class T>
class gRef : public GameObjectRef {
    friend class GameObject;
 protected:
    virtual void Resolve(GameObject *obj) {
	// XXX FIXME: not type safe! should be able to throw an exception
	// or something here so the application doesn't just crash!
	GameObjectRef::Resolve( (GameObject *)dynamic_cast<T *>(obj) );
    }
 public:
    gRef() : GameObjectRef() {}
    gRef(T *target) : GameObjectRef(target) {}
    gRef(const gRef<T>& other) : GameObjectRef(other) {}
    gRef(Packet *pkt) : GameObjectRef(pkt) {}
    gRef<T>& operator=(const gRef<T>& other) {
	// XXX How do I call "GameObjectRef::=(other)" ???
	target = other.target;
	guid = other.guid;
	owner = other.owner;
	if (this != &other) {
	    GameObjectRef::RegisterRef(this);
	}
	return *this;
    }
    gRef<T>& operator=(T *other) {
	if (!other) {
	    SetNull();
	} else {
	    target = other;
	}
	GameObjectRef::RegisterRef(this);
	return *this;
    }
    inline bool operator==(const T *other) {
	return target == other;
    }
    friend inline bool operator==(const T *other, const gRef<T>& t) {
	return t.target == other;
    }
    inline bool operator!=(const T *other) {
	return target != other;
    }
    friend inline bool operator!=(const T *other, const gRef<T>& t) {
	return t.target != other;
    }

    inline T *operator->() {
	ASSERT(IsResolved());
	return static_cast<T *>(target);
    }
    inline operator T *() {
	ASSERT(IsResolved());
	return static_cast<T *>(target);
    }
};

///////////////////////////////////////////////////////////////////////////////

// Type-code for objects for application-level classification and casting
typedef uint16 gTypeCode;
// Typecode indicating we don't know the type of the object
const gTypeCode GAMETYPE_UNKNOWN = 0xFFFF;

/**
 * Base class for all types of game objects, static or dynamic. We just need
 * this so that all objects in the game have a globally unique ID.
 */
class GameObject : public Serializable {
 private:
    friend class StaticGameObject;
    friend class DynamicGameObject;

    GameObject() {} // prevent apps from directly inheriting from this
 protected:
    ///////////////////////////////////////////////////////////////////////////
    // SHARED ATTRIBUTES

    /**
     * The "type" of this class. E.G., used by the application to aid
     * in casting. if you delegate to DynamicGameObject for
     * (un)packing, be sure to set the delta mask fields if this is
     * modified! (Though this field should never be modified really)
     */
    gTypeCode m_TypeCode;

    /**
     * Where this object is currently centered -- if you delegate to
     * DynamicGameObject for (un)packing, be sure to set the delta
     * mask fields if this is modified!
     */
    Vec3 m_Origin;
    /**
     * Dimensions of this object around (0,0,0) (i.e., if m_Origin were
     * translated to (0,0,0)).
     *
     * This field is NOT automatically serialized when you delegate to
     * this DynamicGameObject! You must serialize it explicitly if you want to.
     */
    BBox m_Volume;
 public:
    inline gTypeCode GetType() { 
	return m_TypeCode;
    }
    virtual GUID GetGUID() = 0;

    /**
     * Get this object's origin. This is a copy.
     */
    inline Vec3 GetOrigin() {
	return m_Origin;
    }
    inline const BBox& GetVolume() {
	return m_Volume;
    }

    virtual ~GameObject();

    /**
     * Set this object's origin. This function is virtual since the dynamic
     * object sub-class has to mark the delta mask appropriately to serialize
     * this.
     */
    virtual void SetOrigin(const Vec3& new_origin) {
	ASSERT( !isnan(new_origin[0]) );
	ASSERT( !isnan(new_origin[1]) );
	ASSERT( !isnan(new_origin[2]) );
	m_Origin = new_origin;
    }

    bool IsDynamic() { return IsDynamicGUID(GetGUID()); }
    bool IsStatic() { return IsStaticGUID(GetGUID()); }

    GameObject(Packet *pkt) {
	m_TypeCode = (gTypeCode)pkt->ReadInt();
    }

    void Serialize(Packet *pkt) {
	pkt->WriteInt(GetType());
    }

    uint32 GetLength() {
	return 4;
    }

    void Print(FILE *) {
	return;
    }
};

///////////////////////////////////////////////////////////////////////////////

/**
 * A static object. This is an object that is "static" like a chunk of
 * the map etc. that can not change during the course of a game. There
 * should be at most one instance of each id on each node.
 *
 * XXX: need mechanisms for updating static objects, i.e., patches, etc.
 * maybe some versioning info?
 */
class StaticGameObject : public GameObject {
 private:
    GUID globalID;
 public:
    StaticGameObject(GUID id) : globalID(id) {
	ASSERT(IsStaticGUID(GetGUID()));
    }
    StaticGameObject(Packet *pkt) : GameObject(pkt), globalID(pkt) {
    }
    virtual ~StaticGameObject() {}

    void Serialize(Packet *pkt) {
	GameObject::Serialize(pkt);
	globalID.Serialize(pkt);
    }

    uint32 GetLength() {
	return GameObject::GetLength() + 4;
    }

    GUID GetGUID() { return globalID; }
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Useful wrapper around GObject interface that implements basic routines
 * common to most games. Important for subclasses to read the notes below.
 *
 * <b>DeltaMask</b> - fields 0,1,2,3 are reserved for m_TypeCode and
 * m_Origin, unless the subclass overrides all the delta encoding
 * functions (rather than delegating to them).
 *
 * <b>m_Origin</b> - if (x,y,z) is modified, then you must set DeltaMask
 * fields (0,1,2) respectively in order for the changes to be propagated.
 * A safe way to modify the origin is by using the SetOrigin() method.
 */
class DynamicGameObject : public GObject, public GameObject {
 private:

    static const DeltaMask m_InitialDeltaMask;

    GameObjectRefSet m_Pointers;

 protected:

    uint32 m_PubInterval;
    TimeVal m_LastPubTime;

    // XXX This interface is imperfect. Should be transparent

    /**
     * All distributed pointers to other objects should be registered
     * with this method. The easiest thing todo (for class member
     * references) is just to register them all upon object construction.
     * E.G., If I have:
     *
     * class Foo : public DynamicGameObject {
     *   gRef<Bar> obj1;
     *   gRef<Goo> obj2;
     *
     *   ...
     *
     * Then the constructor can just do:
     *
     *   RegisterRef(&obj1);
     *   RegisterRef(&obj2);
     *
     * And the pointers will remain valid no matter what they point to.
     * The only time you will have to dynamically register or unregister
     * references is if, for example, you have a list or other dynamic
     * datastructure containing a variable number of references.
     */
    void RegisterRef(GameObjectRef *ptr);

    /**
     * When a distributed pointer is deleted, unregister is with this
     * method.
     */
    void UnregisterRef(GameObjectRef *ptr);

    /**
     * Same as unregister(old_ptr), register(new_ptr).
     */
    void ChangeRef(GameObjectRef *old_ptr, GameObjectRef *new_ptr);

    void SetPubInterval(uint32 pubInterval) {
	m_PubInterval = pubInterval;
    }

 public:

    /**
     * Construct a new object locally
     *
     * @param manager the instance of the GameManager.
     */
    DynamicGameObject(gTypeCode type, GameManager *manager);

    /**
     * Construct a replica object from the network
     */
    DynamicGameObject(gTypeCode type, GObjectInfoIface *info);

    virtual ~DynamicGameObject();

    ///////////////////////////////////////////////////////////////////////////
    // GAME API
    //

    GUID GetGUID() { return GObject::GetGUID(); }

    /**
     * Number of delta mask bits used by this class (so sub classes can
     * use bits starting from this index.
     */
    virtual const uint32 GetDeltaMaskBits() { return 4; }

    /**
     * Set this object's origin. This will modify the delta mask appropriately.
     */
    virtual void SetOrigin(const Vec3& new_origin) {
	for (int i=0; i<3; i++) {
	    ASSERT( !isnan(new_origin[i]) );
	    if (m_Origin[i] != new_origin[i]) {
		SetDeltaMaskField(i+1);
		m_Origin[i] = new_origin[i];
	    }
	}
    }

    /**
     * Obtain what this object is currently is interested in. This is a list
     * of bounding boxes. By default, delegate to the game world to figure
     * out what this object is interested in. Return the TTL on the sub
     * (or rely on the InterestFilters to set the TTL).
     *
     * XXX: Should this take into account the discovery delay and
     * publication delay? or push that off to a post-filtering step. I
     * think it would be simplest if we take into account the
     * discovery delay here since only the application knows how much
     * "extra" slack X milliseconds corresponds to.
     */
    virtual uint32 GetAreaOfInterest(list<BBox> *toFill, GameWorld *world) {
	return world->GetAreaOfInterest(toFill, this);
    }

#ifdef SAVE_PREDICTED_AOI
 private:
    list<BBox> m_PredictedAoi;
 public:
    void ClearPredictedAreaOfInterest() {
	m_PredictedAoi.clear();
    }
    // For debugging and visualization
    void AddPredictedAreaOfInterest(const BBox& aoi) {
	m_PredictedAoi.push_back(aoi);
    }

    list<BBox> *GetPredictedAreaOfInterest() {
	return &m_PredictedAoi;
    }
#endif

    /**
     * Return >0 if this object is (or may be) interested in objects
     * at this point. The value returned should be an estimate of how
     * long we expect to remain interested in milliseconds (before
     * having to check again).
     */
    virtual uint32 IsInterested(const Vec3& pt, 
				gTypeCode t = GAMETYPE_UNKNOWN) = 0;

    /**
     * Return >0 if this object is (or may be) interested in objects
     * filling this volume. The value returned should be an estimate
     * of how long we expect to remain interested in milliseconds
     * (before having to check again).
     */
    virtual uint32 IsInterested(const BBox& vol,
				gTypeCode t = GAMETYPE_UNKNOWN) = 0;

    /**
     * This object's think function.
     */
    virtual void RunThink(GameManager *manager, real32 delta) = 0;

    ///////////////////////////////////////////////////////////////////////////
    // COLYSEUS API
    //

    virtual bool IsNew() { return false; }
    virtual void AttachedTo(GUIDList *others) { }
    virtual void AttachTo(GUIDList *others) { }

    virtual void FillEvents(EventList *reg,
			    EventList *unreg,
			    EventSet  *curr);
    virtual uint32 InterestTime(OMEvent *ev, InterestMap *curr);
    virtual uint32 InterestTime(GObject *obj, InterestMap *curr);

    ///////////////////////////////////////////////////////////////////////////
    // DELTA ENCODING AND UPDATES

    virtual const DeltaMask& GetInitDeltaMask();

    /**
     * We assume that the first bit in the delta mask is assigned to the
     * m_TypeCode field of the object so that if it is set, the first
     * uint32 in the buffer will be the object's gTypeCode. This is used
     * by the construction routine in GameManager.
     */
    virtual void PackUpdate(Packet *buffer, const DeltaMask& mask);
    virtual void UnpackUpdate(Packet *buffer, const DeltaMask& mask, 
			      list<GameObjectRef *> *unresolved);
    virtual void GetAllRefs(GObjectRefList *tofill);
    virtual void GetRefs(GObjectRefList *tofill);
    virtual void ResolveRef(GObjectRef *ref, GObject *obj);

    /**
     * Don't implement this! Backward compat...
     */
    virtual void UnpackUpdate(Packet *buffer, const DeltaMask& mask, 
			      SIDMap *unresolved);

    ///////////////////////////////////////////////////////////////////////////
    // MIGRATION API (OPTIONAL)

    virtual bool IsMigratable() { return true; }
    virtual uint32 BeginMigration(sid_t target) { return 1000; }
    virtual void CancelMigration(sid_t target) { return; }
    virtual void CommitMigration(sid_t target) { return; }

    ///////////////////////////////////////////////////////////////////////////
    // DEBUGGING

    virtual void PrintFields(ostream& out);
};

#endif
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
