////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************
  GObject.h

  Basic GameObject.

begin           : Nov 6, 2002
copyright       : (C) 2002-2005 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2002-2005 Jeffrey Pang       ( jeffpang@cs.cmu.edu )

***************************************************************************/

#include "GameObject.h"
#include "GameManager.h"
#include "GameStore.h"
#include "GameDatabase.h"

    ///////////////////////////////////////////////////////////////////////////////

    RefMap GameObjectRef::m_Refs;
BackRefMap GameObjectRef::m_BackRefs;

void GameObjectRef::DeleteRef(GameObjectRef *ptr)
{
    //DB(1) << "called: " << ptr << endl;
    RefMapIter p = m_Refs.find(ptr);

    if (p != m_Refs.end()) {
	GameObject *old = p->second;
	m_Refs.erase(p);
	BackRefMapIter beg = m_BackRefs.lower_bound(old);
	BackRefMapIter end = m_BackRefs.upper_bound(old);
	for ( ; beg != end; beg++) {
	    if (beg->second == ptr) {
		m_BackRefs.erase( beg );
		break;
	    }
	}
    }
}

void GameObjectRef::RegisterRef(GameObjectRef *ptr)
{
    //DB(1) << "called: " << ptr << endl;
    // delete the old one
    DeleteRef(ptr);

    // add the new one
    if ( ptr->GetObject() ) {
	m_Refs.insert( pair<GameObjectRef *,GameObject *>(ptr, ptr->GetObject()) );
	m_BackRefs.insert( pair<GameObject *,GameObjectRef *>(ptr->GetObject(), ptr) );
    }
}

void GameObjectRef::DeleteObjRefs(GameObject *obj)
{
    //DB(1) << "called: " << obj << endl;
    BackRefMapIter beg = m_BackRefs.lower_bound(obj);
    BackRefMapIter end = m_BackRefs.upper_bound(obj);
    for (BackRefMapIter it = beg ; it != end; it++) {
	DB(1) << " del: " << it->second << endl;
	it->second->Nullify();
	m_Refs.erase( it->second );
    }
    m_BackRefs.erase(beg, end);
}

///////////////////////////////////////////////////////////////////////////////

const GameObjectRef::GameObjectRef Null();

void GameObjectRef::Resolve(GameObject *obj) {
    ASSERT(!IsResolved());
    target = obj;
    if (target) {
	GameObjectRef::RegisterRef(this);
    } else {
	SetNull(); // if resolved with NULL, the target was deleted!
    }
}

GameObjectRef::GameObjectRef(GameObject *target) : 
    target(target), guid(target ? target->GetGUID() : GUID_NONE), 
    owner(target && target->IsDynamic() ? g_LocalSID : SID_NONE) {

    if (target) {
	GameObjectRef::RegisterRef(this);
    }
}

GameObjectRef::GameObjectRef(const GameObjectRef& other) :
    target(other.target), guid(other.guid), owner(other.owner) {
    if (this != &other) {
	GameObjectRef::RegisterRef(this);
    }
}

GameObjectRef& GameObjectRef::operator=(const GameObjectRef& other) {
    target = other.target;
    guid = other.guid;
    owner = other.owner;
    if (this != &other) {
	GameObjectRef::RegisterRef(this);
    }
    return *this;
}

GameObjectRef::~GameObjectRef() {
    GameObjectRef::DeleteRef(this);
}

GUID GameObjectRef::GetTarget() {
    if (target) {
	return target->GetGUID();
    } else {
	return guid;
    }
}

SID GameObjectRef::GetOwner() {
    if (target) {
	if (target->IsDynamic()) {
	    return static_cast<DynamicGameObject *>(target)->GetSID();
	} else {
	    return SID_NONE;
	}
    } else {
	return owner;
    }
}

bool GameObjectRef::IsNull() {
    return guid == GUID_NONE && target == NULL;
}

void GameObjectRef::SetNull() {
    Nullify();
    GameObjectRef::DeleteRef(this);
}

bool GameObjectRef::IsResolved() {
    return IsNull() || target != NULL;
}

GameObjectRef::GameObjectRef(Packet *pkt) {
    GameManager  *m = GameManager::GetInstance();
    GameStore    *s = m->GetStore();
    GameDatabase *d = m->GetDatabase();

    guid = GUID(pkt);
    if (guid == GUID_NONE) {
	Nullify();
    } else if (IsStaticGUID(guid)) {

	owner = SID_NONE;
	GameObject *o = s->Find(guid);
	if (o == NULL) {
	    // XXX: What if we don't want to block here?
	    o = d->Load(guid);
	    // XXX: don't crash
	    ASSERT(o);
	    s->Add(o);
	}
	target = o;
	if (target) {
	    GameObjectRef::RegisterRef(this);
	}
	return;

    } else {

	owner = SID(pkt);
	GameObject *o = s->Find(guid);
	// XXX What about the pending store?
	if (o != NULL) {
	    target = o;
	    GameObjectRef::RegisterRef(this);
	} else {
	    target = NULL;
	}
	return;

    }
}

void GameObjectRef::Serialize(Packet *pkt) {
    // Can't check that this is true if serializing to a client
    //ASSERT(IsResolved());

    if (IsNull()) {
	GUID_NONE.Serialize(pkt);
    } else {
	if (IsResolved()) {
	    GUID guid = target->GetGUID();
	    guid.Serialize(pkt);
	    if (IsDynamicGUID(guid)) {
		SID sid = static_cast<DynamicGameObject *>(target)->GetSID();
		sid.Serialize(pkt);
	    }
	} else {
	    // assume that if we get here, we are serializing this ptr
	    // to a client, not to another server -- that should never
	    // happen!
	    ASSERT(IsDynamicGUID(guid));
	    ASSERT(owner != SID_NONE);
	    guid.Serialize(pkt);
	    owner.Serialize(pkt);
	}
    }
}

uint32 GameObjectRef::GetLength() {
    if (IsNull()) {
	return GUID_NONE.GetLength();
    } else {
	return guid.GetLength() + owner.GetLength();
    }
}

///////////////////////////////////////////////////////////////////////////////

GameObject::~GameObject() {
    GameObjectRef::DeleteObjRefs(this);
}

///////////////////////////////////////////////////////////////////////////////

DynamicGameObject::DynamicGameObject(gTypeCode type, GameManager *manager) : 
    GObject(manager->CreateGUID()), m_PubInterval(0), m_LastPubTime(TIME_NONE)
{
    ASSERT(IsDynamicGUID(GetGUID()));
    m_TypeCode = type;
}

DynamicGameObject::DynamicGameObject(gTypeCode type, GObjectInfoIface *info) : 
    GObject(info), m_PubInterval(0), m_LastPubTime(TIME_NONE)
{
    m_TypeCode = type;
}

DynamicGameObject::~DynamicGameObject()
{
}

///////////////////////////////////////////////////////////////////////////////

void DynamicGameObject::FillEvents(EventList *reg, EventList *unreg, EventSet *curr)
{
    // default impl eliminates all old pubs and publishes origin
    TimeVal now = GameManager::GetInstance()->ThisFrameTime();
    if (m_LastPubTime + m_PubInterval >= now) {
	return;
    }
    m_LastPubTime = now;

    m_LastPubTime = now;

    if (curr) {
	for (EventSetIter it = curr->begin(); it != curr->end(); it++) {
	    unreg->push_back(*it);
	}
    }

    OMEvent *ev = 
	GameManager::GetInstance()->m_Manager->CreateEvent(GetGUID());
    m_Origin.FillEvent(ev);
    ev->SetLifeTime( m_PubInterval );
    reg->push_back(ev);
}

uint32 DynamicGameObject::InterestTime(OMEvent *ev, InterestMap *curr)
{
    // delegate to more intuitive game-world functions
    if (ev->IsPointEvent()) {
	Vec3 pt(ev);
	return IsInterested(pt);
    } else {
	BBox vol(ev);
	return IsInterested(vol);
    }
}

uint32 DynamicGameObject::InterestTime(GObject *obj, InterestMap *curr)
{
    // delegate to more intuitive game-world functions
    DynamicGameObject *gobj = (DynamicGameObject *)obj;
    BBox vol(gobj->m_Origin + gobj->m_Volume.min,
	     gobj->m_Origin + gobj->m_Volume.max);
    return IsInterested(vol, gobj->GetType());
}

///////////////////////////////////////////////////////////////////////////////

const DeltaMask DynamicGameObject::m_InitialDeltaMask(1,1,1,1);

const DeltaMask& DynamicGameObject::GetInitDeltaMask()
{
    return m_InitialDeltaMask;
}

void DynamicGameObject::PackUpdate(Packet *buffer, const DeltaMask& mask)
{
    if (mask.IsSet(0)) {
	buffer->WriteInt(m_TypeCode);
    }
    for (int i=0; i<3; i++) {
	if (mask.IsSet(i+1)) {
	    // XXX - need to change this is we use length other than float
	    buffer->WriteFloat(m_Origin[i]);
	}
    }
}

void DynamicGameObject::UnpackUpdate(Packet *buffer, const DeltaMask& mask, 
				     list<GameObjectRef *> *unresolved)
{
    // Ignore the first bit -- it should be for the typecode which is
    // set during construction.
    if (mask.IsSet(0)) {
	gTypeCode type = buffer->ReadInt();
	ASSERT(type == m_TypeCode);
    }
    for (int i=0; i<3; i++) {
	if (mask.IsSet(i+1)) {
	    // XXX - need to change this is we use length other than float
	    m_Origin[i] = buffer->ReadFloat();
	    ASSERT( !isnan(m_Origin[i]) );
	}
    }
}

// backward compatability -- at some point we should push this interface
// change down to the OM layer
void DynamicGameObject::UnpackUpdate(Packet *buffer, const DeltaMask& mask, 
				     SIDMap *unresolved)
{
    list<GameObjectRef *> refs;
    UnpackUpdate(buffer, mask, &refs);
    for (list<GameObjectRef *>::iterator it = refs.begin(); 
	 it != refs.end(); it++) {
	GameObjectRef *ref = *it;
	(*unresolved)[ref->GetTarget()] = ref->GetOwner();
    }
}

///////////////////////////////////////////////////////////////////////////////

void DynamicGameObject::RegisterRef(GameObjectRef *ptr)
{
    m_Pointers.insert(ptr);
}

void DynamicGameObject::UnregisterRef(GameObjectRef *ptr)
{
    m_Pointers.erase(ptr);
}

void DynamicGameObject::ChangeRef(GameObjectRef *old_ptr, 
				  GameObjectRef *new_ptr)
{
    UnregisterRef(old_ptr);
    RegisterRef(new_ptr);
}

void DynamicGameObject::GetAllRefs(GObjectRefList *tofill)
{
    for (GameObjectRefSetIter it = m_Pointers.begin(); 
	 it != m_Pointers.end(); it++) {
	if ( !(*it)->IsNull() )
	    tofill->push_back(*it);
    }
}

void DynamicGameObject::GetRefs(GObjectRefList *tofill)
{
    for (GameObjectRefSetIter it = m_Pointers.begin(); 
	 it != m_Pointers.end(); it++) {
	if ( !(*it)->IsNull() && !(*it)->IsResolved() ) {
	    tofill->push_back(*it);
	}
    }
}

void DynamicGameObject::ResolveRef(GObjectRef *ref, GObject *obj)
{
    GameObjectRef *gref = static_cast<GameObjectRef *>(ref);
    DynamicGameObject *gobj = static_cast<DynamicGameObject *>(obj);

    gref->Resolve(gobj);
}

///////////////////////////////////////////////////////////////////////////////

void DynamicGameObject::PrintFields(ostream& out)
{
    out << "orig=" << m_Origin;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
