////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************
  GameManager.h

  Top-level interface to the game application.

begin           : Nov 6, 2002
copyright       : (C) 2002-2005 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2002-2005 Jeffrey Pang       ( jeffpang@cs.cmu.edu )

***************************************************************************/

#ifndef __GAME_MANAGER_H__
#define __GAME_MANAGER_H__

#include <gameapi/common.h>
#include <gameapi/GameInterestFactory.h>
#include <util/TimeVal.h>
#include <om/Manager.h>
#include <Mercury.h>
#include <gameapi/GameVisualizer.h>

    class GameObject;
class GameModule;
class GameStore;
class GameWorld;
class GameDatabase;
class GameDataSource;
class GameClientHandler;

typedef map<GUID, list<InterestFilter *> *, less_GUID> FilterMap;
typedef FilterMap::iterator FilterMapIter;

/**
 * The GameManager serves as the entry point into the game: it is a composition
 * of several modular components of the game:
 *
 * <ul>
 * <li> GameModule - the actual game logic. This is similar to the interface
 *      most FPS's export to mods.
 * <li> GameWorld - interface to the game's static geometry (e.g., a bsp or
 *      tile-based map).
 * <li> GameClientHandler - an interface to a module that manages client
 *      connections.
 * <li> GameDataSource(s) - a list of persistent stores from which the game
 *      can load data.
 * </ul>
 *
 * The process of starting up the game is as follows:
 *
 * <ol>
 * <li> <tt>GameManager *gm = GameManager::GetInstance(...);</tt>
 * <li> <tt>gm->Init(...);</tt>
 * <li> <tt>gm->Run();</tt>
 * </ol>
 */
class GameManager : public GameAdaptor {
 private:
    friend class DynamicGameObject;

    // singleton instance of GameManager
    static GameManager *m_Instance;

    /**
     * Gameplay logic.
     */
    GameModule *m_Module;
    /**
     * Dynamic gameplay objects.
     */
    GameStore *m_Store;
    /**
     * Static gameplay world geometry/tiles.
     */
    GameWorld *m_World;
    /**
     * Data sources for loading and storing persistent gameplay data.
     */
    GameDatabase *m_Database;
    /**
     * Module that handles client connections.
     */
    GameClientHandler *m_ClientHandler;
    /**
     * Module that filters object interests.
     */
    GameInterestFactory *m_InterestFactory;

    FilterMap    m_ObjectFilters;
    FilterChain *m_GlobalFilter;

    /**
     * Interface to Colyseus.
     */
    Manager *m_Manager;

    // visualization
    GameVisualizer m_Visualizer;

    // Local Game State
    uint32  m_FrameNumber;
    TimeVal m_LastFrame;
    TimeVal m_ThisFrame;
    TimeVal m_NextFrame;

    GameManager(int *pArgc, char *argv[], 
		OptionType appOptions[] = NULL,
		bool printOptions = true);

    bool ProcessInterestFilters(FilterChain *filters,
				AreaOfInterest *toFill,
				AreaOfInterest *orig);

    void LogStoreStats();
    void LogInterestStats();
    void LogObjectDeltaStats();
 protected:

    GUID CreateGUID();

 public:
    // XXX Shouldn't really expose this, but common.cpp needs it :P
    Manager *GetManager() { return m_Manager; }

    virtual ~GameManager();

    static GameManager *GetInstance(int *pArgc, char *argv[], 
				    OptionType appOptions[] = NULL,
				    bool printOptions = true);

    inline static GameManager *GetInstance() {
	ASSERT(m_Instance);
	return m_Instance;
    }

    inline GameModule   *GetModule()   { return m_Module; }
    inline GameStore    *GetStore()    { return m_Store; }
    inline GameWorld    *GetWorld()    { return m_World; }
    inline GameDatabase *GetDatabase() { return m_Database; }

    inline GameVisualizer *GetVisualizer() { return &m_Visualizer; }

    inline uint32 GetFrameNumber() {
	return m_FrameNumber;
    }
    inline TimeVal CurrentTime() {
	static TimeVal v;
	OS::GetCurrentTime(&v);
	return v;
    }
    inline const TimeVal& LastFrameTime() { return m_LastFrame; }
    inline const TimeVal& ThisFrameTime() { return m_ThisFrame; }
    inline const TimeVal& NextFrameTime() { return m_NextFrame; }
    inline uint32 FrameTimeElapsed() {
	return CurrentTime() - ThisFrameTime();
    }
    inline uint32 FrameTimeRemaining() {
	return NextFrameTime() - CurrentTime();
    }

    /**
     * Initialize the game world and data sources.
     */
    void Init(GameModule *module,
	      GameWorld  *world,
	      GameClientHandler *chandler = NULL,
	      GameInterestFactory *ifactory = NULL,
	      list<GameDataSource *> *sources = NULL,
	      bool dummy = false /* for testing purposes only */);

    /**
     * Run the game loop. This function only terminates when the game
     * is shutdown.
     */
    void Run();

    /**
     * Used currently for debug. makes sure Manager and mercury 
     * underneath are idling for some time...
     **/
    void Idle(uint32 timeoutMillis);

    ///// GameAdaptor interface

    uint32 GetSendInterval();
    ObjectStore *GetObjectStore();
    ObjectStore *GetPendingStore();
    void FillInterests(InterestList *reg,
		       InterestList *unreg,
		       GUIDMap *reg_assoc,
		       InterestMap *curr);
    GObject *Construct(GObjectInfoIface *info,
		       Packet *pkt, 
		       const DeltaMask& mask,
		       SIDMap *unresolved);
    void Destroy(GUID guid);
    void InstallTerminalHandlers(InputHandlers *tofill);
};

#endif
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
