////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

/**************************************************************************
  GameDatabase.h

begin           : Nov 6, 2002
copyright       : (C) 2002-2005 Ashwin R. Bharambe ( ashu@cs.cmu.edu   )
(C) 2002-2005 Jeffrey Pang       ( jeffpang@cs.cmu.edu )

***************************************************************************/

#ifndef __GAME_DATABASE_H__
#define __GAME_DATABASE_H__

#include <list>
#include <mercury/ID_common.h>
#include <gameapi/common.h>
#include <util/EventToken.h>

    class GameObject;

/**
 * Interface for sources from which we can load global objects from.
 * Implementations are things like backend databases, local files, a DHT
 * etc.
 *
 * XXX: Right now this only supports a put/get interface. Some data sources
 * will inevitably want to use a richer set of queries...
 */
class GameDataSource {
 public:
    virtual ~GameDataSource () {} 

    /**
     * Return true if this data source can (or should be able to,
     * dependent on transient failures) load a given object.
     */
    virtual bool CanLoad(const GUID& globalID) = 0;

    /**
     * Return true if this data source can (or should be able to)
     * store this object persistently in it.
     */
    virtual bool CanStore(GameObject *obj) = 0;

    /**
     * Begin loading the specified object. Return a token that will be signaled
     * when the loading is complete.
     *
     * @param globalID the id of the object to load. Should be statically
     * addressable.
     * @param prio priority of this load request. Implementations may or
     * may not support this feature and semantics are implementation
     * dependent. For conformity, we will assume a general range from
     * [0..1000] and that higher values have higher priority (should be
     * completed sooner).
     *
     * @return a token that will be signaled when the fetch is completed,
     * or NULL if we can not load the specified object. The token should be
     * signaled with an error code of 0 if successful, and non-0 otherwise.
     */
    virtual EventToken *Preload(const GUID& globalID, uint32 prio = 0) = 0;

    /**
     * Load the specified object from the datasource itself if not already
     * loaded, or from the cache if preloaded. Block until the object is
     * loaded. See Preload for more info.
     *
     * @return the loaded object or NULL if we could not load it.
     */
    virtual GameObject *Load(const GUID& globalID, uint32 prio = 0) = 0;

    /**
     * Store the specified object as persistent at its GUID location.
     * Signal when it is complete. The token should be signaled with
     * an error code of 0 if successful, and non-0 otherwise.
     */
    virtual EventToken *Store(GameObject *obj, uint32 prio = 0) = 0;

    /**
     * Free a token that was given by this data source.
     */
    virtual void FreeToken(EventToken *tok) = 0;
};

///////////////////////////////////////////////////////////////////////////////

typedef list<GameDataSource *> DataSourceList;
typedef DataSourceList::iterator DataSourceListIter;

typedef map<EventToken *, GameDataSource *> TokenMap;
typedef TokenMap::iterator TokenMapIter;

/**
 * The database is the interface through which we load static game objects.
 * 
 * XXX: We should push this down --- the OM's fetching routine should just be
 * a type of datasource where it receives SID hints as to where objects are
 * located. That way we can also fetch pointers lazily.
 */
class GameDatabase {
 private:
    static GameDatabase *m_Instance;

    // our data sources to load from, in order
    DataSourceList m_Sources;
    TokenMap m_TokenMap;

    GameDatabase();
 public:

    static GameDatabase *GetInstance();

    ~GameDatabase();

    void AddDataSource(GameDataSource *src);
    EventToken *Preload(GUID globalID, uint32 prio = 0);
    GameObject *Load(GUID globalID, uint32 prio = 0);
    EventToken *Store(GameObject *obj, uint32 prio = 0);
    void FreeToken(EventToken *tok);
};

#endif
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
