////////////////////////////////////////////////////////////////////////////////
// Mercury and Colyseus Software Distribution 
// 
// Copyright (C) 2004-2005 Ashwin Bharambe (ashu@cs.cmu.edu)
//               2004-2005 Jeffrey Pang    (jeffpang@cs.cmu.edu)
//                    2004 Mukesh Agrawal  (mukesh@cs.cmu.edu)
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2, or (at
// your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA
////////////////////////////////////////////////////////////////////////////////

using namespace std;

#include <iostream>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <gameapi/common.h>
#include <gameapi/GameManager.h>
#include <gameapi/GameDatabase.h>
#include <gameapi/GameStore.h>
#include <gameapi/impl/SimpleGame.h>
#include <gameapi/impl/SimplePlayer.h>
#include <gameapi/impl/SimpleMissile.h>
#include <gameapi/impl/EmptyWorld.h>
#include <gameapi/impl/SimpleInterestFactory.h>
#include <gameapi/impl/SimpleWorld.h>
#include <gameapi/impl/SimpleClientHandler.h>
#include <gameapi/impl/WayPointManager.h>

static uint32 g_RandSeed;
static bool g_DisableColyseus;
static char g_MapDir[255];
static uint32 g_NumBots;
static real32 g_MapWidth;
static real32 g_MapHeight;
static real32 g_MapZAxis;
static real32 g_AoiSize;
static real32 g_WayPointParetoAlpha;
static uint32 g_RegionsPerX;
static uint32 g_RegionsPerY;

static uint32 g_SubPredictionPlayer;
static uint32 g_SubPredictionMissile;
static uint32 g_PubInterval;
static uint32 g_MaxDisLat;

static uint32 g_IdleTime;

static OptionType g_Options[] =
    {
	{ '/', "seed", OPT_INT, 
	  "random seed", 
	  &g_RandSeed, "42", NULL },
	{ '/', "no-colyseus", OPT_BOOL|OPT_NOARG, 
	  "disable colyseus (run game locally only)", 
	  &g_DisableColyseus, "0", (void *)"1" },
	// XXX: Find some way to push this down one layer. App should not
	// need to know about this!
	{ '/', "dht", OPT_BOOL|OPT_NOARG, 
	  "publish/subscribe using DHT region keys instead of range queries (requires --regions-per-side)", 
	  &SimpleGame::publishAsDHTRegions, "0", (void *)"1" },
	{ '/', "bots", OPT_INT, 
	  "number of bots to start on this server", 
	  &g_NumBots, "4", NULL },
	{ '/', "aoisize", OPT_FLT, 
	  "width and height of area of interest (area = aoisize^2)", 
	  &g_AoiSize, "650", NULL },
	{ '/', "map", OPT_STR,
	  "map directory", 
	  g_MapDir, "", NULL},
	{ '/', "regions-per-x", OPT_INT, 
	  "how to split up the xrange into regions", 
	  &g_RegionsPerX, "1", NULL },
	{ '/', "regions-per-y", OPT_INT, 
	  "how to split up the yrange into regions", 
	  &g_RegionsPerY, "1", NULL },
	{ '/', "waypoint-alpha", OPT_FLT,
	  "waypoint popularity pareto parameter (Pr[X=x]=x^-a)", 
	  &g_WayPointParetoAlpha, "0.1", NULL},
	{ '/', "bot-wander", OPT_BOOL|OPT_NOARG, 
	  "enable bot random wandering at waypoints", 
	  &g_SimplePlayerWander, "0", (void *)"1" },
	{ '/', "maxdislat", OPT_INT, 
	  "max discovery latency for prediction fudge (msec)", 
	  &g_MaxDisLat, "300", NULL },
	{ '/', "pub-interval", OPT_INT, 
	  "how often an object should publish (msec)", 
	  &g_PubInterval, "1000", NULL },
	{ '/', "sub-prediction-player", OPT_INT, 
	  "how many msec to predict player subscriptions", 
	  &g_SubPredictionPlayer, "5000", NULL },
	{ '/', "sub-prediction-missile", OPT_INT, 
	  "how many msec to predict missile subscriptions", 
	  &g_SubPredictionMissile, "500", NULL },
	{ '/', "idletime", OPT_INT,
	  "time to idle after hitting timelimit",
	  &g_IdleTime, "120", NULL },
	{ 0, 0, 0, 0, 0, 0, 0 }
    };

static void load_mapdims(const char *filename)
{
    char line[255];
    FILE *f = fopen(filename, "r");
    if (f == NULL) {
	Debug::die("can't open mapdims file: %s", filename);
    }

    real32 x, y, z;

    fgets(line, 255, f);
    int ret = sscanf(line, "%f %f %f", &x, &y, &z);
    ASSERT(ret == 3);

    fclose(f);

    g_MapWidth  = x;
    g_MapHeight = y;
    g_MapZAxis  = z;
}

int main(int argc, char **argv)
{

    GameManager *m = GameManager::GetInstance(&argc, argv, g_Options);

    srand(g_RandSeed);
    srand48(g_RandSeed);

    SimpleGame::SetMaxDisLat(g_MaxDisLat);
    SimpleGame::SetPubInterval(g_PubInterval);
    SimplePlayer::SetSubPrediction(g_SubPredictionPlayer);
    SimpleMissile::SetSubPrediction(g_SubPredictionMissile);

    if (!strcmp(g_MapDir, "")) {
	Debug::die("testgame requires the --map option");
    }

    load_mapdims( (string(g_MapDir) + "/dims").c_str() );

    BBox extent( Vec3(0,0,-g_MapZAxis/2),
		 Vec3(g_MapWidth,g_MapHeight,g_MapZAxis/2) );
    BBox aoi( Vec3(-g_AoiSize/2,-g_AoiSize/2,-20), 
	      Vec3(g_AoiSize/2,g_AoiSize/2,20) );

    SimpleGame          *g = new SimpleGame();
    SimpleWorld         *w = new SimpleWorld(extent, aoi, 
					     g_RegionsPerX,
					     g_RegionsPerY);
    SimpleClientHandler *c = new SimpleClientHandler();

    FileWayPointFactory wpf( (string(g_MapDir) + "/waypoints").c_str() );
    w->Init(&wpf, g_WayPointParetoAlpha);
    SimpleInterestFactory *f = new SimpleInterestFactory();

    DBG_INIT(&g_LocalSID);

    INFO << "world=" << w->GetExtent() << endl;

    m->Init(g, w, c, f, NULL, g_DisableColyseus);

    GameStore *s = m->GetStore();

    for (uint32 i=0; i<g_NumBots; i++) {
	TDB (-1) << " initing bot " << i << endl;

	SimplePlayer *o1 = new SimplePlayer(m);

	Vec3 start = w->GetRandomPoint();
	start[2] = 0;
	o1->SetOrigin(start);

	s->Add(o1);
    }

    Benchmark::print ();
    Benchmark::restart ();

    m->Run();

    cerr << "* idling for " << g_IdleTime << " seconds " << endl;
    m->Idle(g_IdleTime * 1000);
    cerr << "exiting gracefully" << endl;
    return 0;
}
// vim: set sw=4 sts=4 ts=8 noet: 
// Local Variables:
// Mode: c++
// c-basic-offset: 4
// tab-width: 8
// indent-tabs-mode: t
// End:
