/**
 * 
 */
package edu.cmu.cs.ark.compuframes;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;

import com.martiansoftware.jsap.JSAPResult;

import edu.cmu.cs.ark.yc.config.AppConfig;

/**
 * @author Yanchuan Sim
 * @version 0.1
 * @since 0.1
 */
public class AppSettings
{
  public String NAME;
  public File DATA_DIR, OUTPUT_DIR, RESUME_DIR, SAMPLE_DIR;
  public File TERMS_FILE, IDEOLOGY_FILE;
  public double ALPHA, GAMMA, EMISSION_IDEO, EMISSION_OTHER;
  public int SAVE_INTERVAL;
  public int ITERATIONS, SAMPLE_COUNT, SAMPLE_INTERVALS, BURNIN_SAMPLES, BURNIN_ITERS, MSTEP_ITERS;

  private boolean valid = false;
  private boolean resume = false;
  private JSAPResult jsap_results;

  public AppSettings(AppConfig app_config, String[] args)
  {
    jsap_results = app_config.parse(args);

    if (!jsap_results.success())
    {
      valid = false;
      return;
    }

    if (jsap_results.contains("resume-dir"))
    {
      RESUME_DIR = jsap_results.getFile("resume-dir");
      resume = true;

      try
      {
        System.err.format("[Init] Resuming system with data loaded from <%s>. Note: Other arguments/options provided will be ignored.\n", RESUME_DIR.getPath());

        jsap_results = app_config.parse(AppConfig.getArgumentsFromFile(new File(FilenameUtils.concat(RESUME_DIR.getAbsolutePath(), "settings.txt"))));
        valid = useJSAPResults(jsap_results);
      }
      catch (IOException e)
      {
        jsap_results.addException("resume-dir", e);
      }
    }
    else
      valid = useJSAPResults(jsap_results);
  }

  /**
   * Populate this object with system options contained in a {@link JSAPResult}.
   * 
   * @param results
   *          {@link JSAPResult} object containing the system options.
   * @return <i>True</i> if the options are valid.
   */
  private boolean useJSAPResults(JSAPResult results)
  {
    Map<String, Exception> settings_exceptions = new HashMap<String, Exception>();

    if (!isResume())
    {
      if (!results.contains("data-dir"))
        settings_exceptions.put("data-dir", new Exception("Argument is required."));
      if (!results.contains("output-dir"))
        settings_exceptions.put("output-dir", new Exception("Argument is required."));

      if (!results.contains("terms-file"))
        settings_exceptions.put("terms-file", new Exception("Argument is required."));
      if (!results.contains("ideology-file"))
        settings_exceptions.put("ideology-file", new Exception("Argument is required."));
    }

    if (settings_exceptions.size() > 0)
    {
      for (Entry<String, Exception> e : settings_exceptions.entrySet())
        results.addException(e.getKey(), e.getValue());
      return false;
    }

    NAME = results.getString("name", "");

    ALPHA = results.getDouble("alpha");
    GAMMA = results.getDouble("gamma");
    EMISSION_IDEO = results.getDouble("emission-ideo");
    EMISSION_OTHER = results.getDouble("emission-other");

    SAVE_INTERVAL = results.getInt("save-interval");
    ITERATIONS = results.getInt("iterations");
    SAMPLE_COUNT = results.getInt("sample-count");
    SAMPLE_INTERVALS = results.getInt("sample-interval");
    BURNIN_SAMPLES = results.getInt("burnin-samples");
    BURNIN_ITERS = results.getInt("burnin-iters");
    MSTEP_ITERS = results.getInt("mstep-iters");

    if (isResume())
    {
      OUTPUT_DIR = RESUME_DIR;
      SAMPLE_DIR = new File(FilenameUtils.concat(OUTPUT_DIR.getAbsolutePath(), "samples"));
      DATA_DIR = results.getFile("data-dir");
      TERMS_FILE = new File(FilenameUtils.concat(RESUME_DIR.getPath(), "terms.txt"));
      IDEOLOGY_FILE = new File(FilenameUtils.concat(RESUME_DIR.getPath(), "ideology.txt"));
    }
    else
    {
      DATA_DIR = results.getFile("data-dir");
      OUTPUT_DIR = results.getFile("output-dir");
      TERMS_FILE = results.getFile("terms-file");
      IDEOLOGY_FILE = results.getFile("ideology-file");
      SAMPLE_DIR = new File(FilenameUtils.concat(OUTPUT_DIR.getAbsolutePath(), "samples"));

      if (!TERMS_FILE.canRead())
        settings_exceptions.put("terms-file", new Exception("File is unreadable."));
      if (!IDEOLOGY_FILE.canRead())
        settings_exceptions.put("ideology-file", new Exception("File is unreadable."));

      if (!OUTPUT_DIR.canRead())
        settings_exceptions.put("output-dir", new Exception("Directory is not readable."));
      if (!OUTPUT_DIR.canWrite())
        settings_exceptions.put("output-dir", new Exception("Directory is not writable."));

      if (!DATA_DIR.canRead())
        settings_exceptions.put("data-dir", new Exception("Directory is not readable."));
    }

    if (SAMPLE_COUNT < BURNIN_SAMPLES)
      settings_exceptions.put("burnin-samples", new Exception("Value must be smaller than sample counts."));

    if (ITERATIONS < BURNIN_ITERS)
      settings_exceptions.put("burnin-iters", new Exception("Value must be smaller than total iterations."));

    if (settings_exceptions.size() > 0)
    {
      for (Entry<String, Exception> e : settings_exceptions.entrySet())
        results.addException(e.getKey(), e.getValue());
      return false;
    }

    if (NAME.isEmpty())
      NAME = OUTPUT_DIR.getName();

    return true;
  }

  /**
   * @return <i>True</i> if the settings are valid.
   */
  public final boolean isValid()
  {
    return valid;
  }

  /**
   * @return <i>True</i> if this is a resume.
   */
  public final boolean isResume()
  {
    return resume;
  }

  public final JSAPResult getJSAPResults()
  {
    return jsap_results;
  }

  /**
   * Displays settings information to <code>ps</code>.
   * 
   * @param ps
   *          {@link PrintStream} to display the app settings.
   */
  public void displaySettings(PrintStream ps)
  {
    ps.format("[Init] Settings for <%s>\n", NAME);
    ps.format("[Init] --------------%s-\n", StringUtils.repeat('-', NAME.length()));
    ps.format("[Init] Output directory                     = %s\n", OUTPUT_DIR.getPath());
    ps.format("[Init] Data directory                       = %s\n", DATA_DIR.getPath());
    ps.format("[Init] Terms file                           = %s\n", TERMS_FILE.getPath());
    ps.format("[Init] Ideology file                        = %s\n", IDEOLOGY_FILE.getPath());
    ps.format("[Init] Transitions Dirichlet hyperparameter = %f\n", ALPHA);
    ps.format("[Init] Restarting  hyperparameter           = %f\n", GAMMA);
    ps.format("[Init] Emissions prior hyperparameter       = %f, %f\n", EMISSION_IDEO, EMISSION_OTHER);
    ps.format("[Init] Iterations(-Burnin) x Burnin/Samples/Interval = %d(-%d) x (%d/%d/%d)\n", ITERATIONS, BURNIN_ITERS, BURNIN_SAMPLES, SAMPLE_COUNT, SAMPLE_INTERVALS);
    ps.format("[Init] M-steps iteration                    = %d\n", MSTEP_ITERS);
    ps.format("[Init] -----------------------------------------\n");
  }
}
