
functor NamelessFn (structure Lambda : LAMBDA) :> NAMELESS where L = Lambda = struct

    structure A = AbsPcf
    structure L = Lambda

    type identifier = A.identifier

    type info = LambdaInfo.info

    (* for contexts, we use the following trick:
       whenever we add a variable to the context, we record
       with it the size of the context at the time (so x => |Gamma|)
       when we need to pull it out later, the context will be
       G = (Gamma,x -> |Gamma|,Gamma'), so the proper deBruijn index for x
       is |Gamma'| = |G| - (|Gamma| + 1).
       The final trick is, to speed up lookups, we don't store the context
       as a list, but rather as some kind of clever data structure.  So
       we can lookup x quickly, and do a constant amount of work to 
       figure out its index. *)
    structure Context = ListMapFn (struct
				       type ord_key = identifier
				       val compare = String.compare
                                   end);
    type context = int * (int Context.map)

    val empty_context = (0, Context.empty)

    exception UnboundVariable of (identifier * info)

    fun lookup (z,m) (nm,info) = 
	case Context.find (m,nm) of
	    NONE => raise (UnboundVariable (nm, info))
	  | SOME i => z - (i + 1)


    fun extend (z,m) nm = let
	val m' = Context.insert(m,nm,z)
    in
	(1+z,m')
    end

    fun sizeof_context (z,_) = z

    fun nameless_typ tyctx (tp,info) =
	case tp of
	    A.Unit => L.Unit
	  | A.Bool => L.Bool
	  | A.Nat => L.Nat
	  | A.Arrow (tp1, tp2) => L.Arrow (nameless_typ tyctx tp1, nameless_typ tyctx tp2)
	  | A.Rec (s, t) => let
		val tyctx' = extend tyctx s
	    in
		L.Rec (s, nameless_typ tyctx' t)
	    end
	  | A.TyVar s => L.TyVar (lookup tyctx (s,info), sizeof_context tyctx)

    fun unfold_let ([], body) = body
      | unfold_let ((id,tp, t)::bs, body) = 
	(A.App ((A.Abs (id, tp, unfold_let (bs, body)), LambdaInfo.dummyinfo),
		t),
	 LambdaInfo.dummyinfo)

    fun nameless ctx = let
	val look = lookup ctx
	val sz = sizeof_context ctx
	val ex = extend ctx
	fun nmls (t,info) =
	    case t of
		(A.Var nm) => (L.Var (look (nm,info), sz), info)
	      | A.UnitVal => (L.UnitVal, info)
	      | A.True => (L.True,info)
	      | A.False => (L.False, info)
	      | (A.IfThenElse (t1,t2,t3)) => (L.IfThenElse (nmls t1, nmls t2, nmls t3), info)
	      | A.Zero => (L.Zero, info)
	      | A.Succ t' => (L.Succ (nmls t'), info)
	      | A.Pred t' => (L.Pred (nmls t'), info)
	      | A.IsZero t' => (L.IsZero (nmls t'), info)
	      | A.Abs (id,tp,t') => let
		    val ctx' = ex id
		in
		    (L.Abs (id, nameless_typ empty_context tp, nameless ctx' t'), info)
		end
	      | A.App (t1,t2) =>
		(L.App (nmls t1, nmls t2), info)
	      | A.Roll (tp,t) =>
		(L.Roll (nameless_typ empty_context tp, nmls t), info)
	      | A.Unroll t =>
		(L.Unroll (nmls t), info)
	      | A.Let (binds, t) =>
		nmls (unfold_let (binds,t))
    in
	nmls
    end


end