(* you should really read sources.cm first, since that will give you
   some idea where everything is.  The important functions in here are
   Main.run and Main.run_str which take a file name, or a string to
   parse, respectively, parse it, convert to lambda terms by removing
   names, call your typechecker, call your evaluator, add the names
   back in, and pretty print the result. *)
(* everything is probably most readable in an editor that's about 120
   columns wide, btw.*)
structure Main =
   struct

      exception MainError


      fun parse fname =
          let val ins = TextIO.openIn fname
              val e = Parse.parse ins
		  handle Parse.ParseError (s,loc) => 
			 (print (s ^ (Location.toString loc)) ;
			  raise MainError)
          in
              TextIO.closeIn ins;
              (*Print.flush ();*)
              e
          end

      fun parse_str s = Parse.parse_str s
	  handle Parse.ParseError (s,loc) => 
		 (print (s ^ (Location.toString loc)) ;
		  raise MainError)

      structure Nameless = NamelessFn (structure Lambda = Lambda);

      structure Named = NamedFn (structure Lambda = Lambda);

      (* a little utility function to make debugging easier.
	 if you don't get interesting tracebacks, make
	 sure Compiler.Control.trackExn is assigned true
         when you compile your modules.  That is, do:
         Compiler.Control.trackExn := true ;
         CM.make ();
         at least once. *)
      fun traceback oops = 
	  (print "  exception's history:\n" ;
	   List.app (fn x => (print ("\t" ^ x); print "\n"))
		    (SMLofNJ.exnHistory oops) )
		  

      fun nameless abs = Nameless.nameless Nameless.empty_context abs
	  handle Nameless.UnboundVariable (id, info) =>
		 (print ("Unbound variable " ^ id ^ " while removing names" ^
			 (Location.toString info) ^ "\n") ;
		  raise MainError)
	       | x =>
		 (print "Unknown exception, while removing names, propagating...\n" ;
		  raise x)
		      
      fun tc t = Lambda.typecheck Lambda.empty_context t
	  handle (oops as Lambda.TypeError (msg, info)) =>
		 (print ("Type error: " ^ msg ^ ", while typechecking" ^
			 (Location.toString info) ^ "\n") ;
		  print "  exception's history:\n" ;
		  List.app (fn x => (print ("\t" ^ x) ; print "\n")) (SMLofNJ.exnHistory oops) ;
		  raise MainError)
	       | (Lambda.Unimplemented s) =>
		 (print ("you didn't implement " ^ s ^ "\n"); raise MainError)
	       | x => (print "Unknown exception, while typechecking, propagating...\n" ;
		       raise x)

      fun eval t = Lambda.eval t
	  handle (oops as Lambda.NoRuleApplies) =>
		 (print ("No reduction rule applies.  But well-typed terms do not go wrong..." ^
			 "either there's a bug, or the term is ill-typed\n") ;
		  traceback oops ;
		  raise MainError)
	       | x => (print "Unknown exception during evaluation, propagating...\n" ;
		       raise x)

      fun name t = Named.named Named.empty_context t
	  handle Named.InconsistentVariable (i,m,n,info) =>
		 (* the info here may be bogus since we could use this function after evaluation *)
		 (print ("Inconsistent Variable " ^ (Int.toString i) ^
			 " while adding names: expected context size " ^ (Int.toString n) ^
			 ", real context size " ^ (Int.toString m) ^
			 (Location.toString info) ^ "\n") ;
		  raise MainError)
	       | x => (print "Unknown exception while adding names, propagating...\n" ;
		       raise x)

      fun pretty e = Pretty.pretty () e
		     
      fun run fname = let
	  val term = (nameless o parse) fname
	  val tp = tc term
      in
	  (pretty o name o eval) term
      end
	  
      fun run_str s = let
	  val term = (nameless o parse_str) s
	  val tp = tc term
      in
	  (pretty o name o eval) term
      end 
   end


fun shallow () = (Compiler.Control.Print.printDepth := 5;
		  Compiler.Control.Print.printLength := 12 )
fun deep () = (Compiler.Control.Print.printDepth := 200 ;
	       Compiler.Control.Print.printLength := 400 )
