E_M4WARNING

/*
 * 	ECO: Efficient Collective Operations
 * 	Beta release 0.1b
 * 	Bruce Lowekamp and Adam Beguelin	
 * 	School of Computer Science
 * 	Carnegie Mellon University
 * 	Pittsburgh, PA 15213
 * 
 * 	(C) 1996 All Rights Reserved
 * 
 * NOTICE:
 * 
 *  Permission to use, copy, modify, and distribute this software and
 *  its documentation for any purpose and without fee is hereby granted
 *  provided that the above copyright notice appear in all copies and
 *  that both the copyright notice and this permission notice appear in
 *  supporting documentation.
 * 
 *  Neither Carnegie Mellon University nor the Authors make any
 *  representations about the suitability of this software for any
 *  purpose.  This software is provided ``as is'' without express or
 *  implied warranty.
 * 
 *  This research is sponsored in part by the Department of Defense
 *  Advanced Research Projects Agency and the National Science
 *  Foundation.
 */

/* eco method = 0 --- normal
                 1 --- star
                 2 --- tree */

ifdef(`E_EXTERNSTDINCLUDES',`extern "C"{')
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
ifdef(`E_EXTERNSTDINCLUDES',`}')

ifdef(`E_CPLUSPLUS',
`extern "C" {')
#include <pvm3.h>
ifdef(`E_CPLUSPLUS',
`}')

#include "eco.h"

#define MaxStr 256
#define TREE_K 2

/* network database info */
static char*** subnet_by_name=NULL;
static int* subnet_size=NULL;
double** inter_subnet_times=NULL;
static int subnet_count;

static int have_database = FALSE;

static int eco_method = 0; /* default to optimized format */

/**************** eco's global variables ***********************/
int* ECO_tree_parent; /* parent in tree communication pattern rooted at []*/
int* ECO_tree_child_count; /* number of children in tree comm pattern */
int** ECO_tree_children; /* list of children for each rooted tree */
int ECO_myid; /* index of this process in group list */
int ECO_p; /* number of processes in group */
int* ECO_tids; /* list of tids in group list */
int ECO_opt_tree;
int ECO_ALL;
int** ECO_tree_desc_info;
int** ECO_tree_desc_max;
int ECO_cur_mtag=0x600;
int* subnet_map; /* tids[i] is in subnet subnet_map[i] */
ECO_user_function eco_oper_list[ECO_MAX_FUNCS];

/**************** end global variables ************************/

/* local subroutines */
static char *pvmhostname(struct pvmhostinfo *hostp, int ti_host);
static int eco_database_init (int p_i, int p, int* tids, char* data_filename);
static int eco_subnet_init(int p, int* tids);
static int eco_other_pattern_init(int p_i, int p, int* tids);


#undef min
#undef max
#define max(a,b)  ((a)>(b)?(a):(b))
#define min(a,b)  ((a)<(b)?(a):(b))


#define MAX_P 10000

/* eco_init (IN int p_i, IN int p, IN/OUT int tids[], char data_filename[], */
/* 	IN int topology) */

/* p_i: pvm_mytid() = tids[p_i]  0<=p_i<p */
/* p: number of tasks participating in collective communication */
/* tids: array of tids.  returned in new order if topology is used */
/* data_filename: network database filename (used by p_i=0) */
/* 	defaults to a compile time value if NULL */
/* topology: topology to provide for local communication.  values: */
/* 		ECO_DEFAULT_TOPOLOGY */
/* 		ECO_RING_TOPOLOGY */

/* eco_init is used to read in the database and generate patterns for */
/* collective communication.  It must be called by all tasks.  If */
/* topology is set to something other than ECO_DEFAULT_TOPOLOGY (0), tids */
/* may be returned in a different order. */
int eco_init(int* p_i, int p, int* tids, char* data_filename,
	     int topology, void* topology_struct)
{
  int i;

  /*printf ("calling eco_init %i,%i\n", *p_i,p);
  fflush(NULL);*/

  /* initialize variables */
  for(i=0;i<ECO_MAX_FUNCS;i++)
    eco_oper_list[i] = NULL;

    if(eco_database_init(*p_i,p,tids,data_filename))
      return 1;
    
    if(eco_subnet_init(p, tids))
	return 1;


    switch(topology){
    case ECO_DEFAULT_TOPOLOGY:
      break;
    case ECO_RING_TOPOLOGY:
      eco_ring_topology(tids,tids,p);
      for((*p_i)=0;tids[*p_i] != pvm_mytid();(*p_i)++);
      break;
    default:
      fprintf (stderr, "unknown topology\n");
      return 1;
    }

    ECO_p = p;
    ECO_myid = *p_i;
    /* ECO_ALL communication is sort of faked by putting the patterns
       at index p, thus ECO_ALL is a variable, not a constant */
    ECO_ALL = p;
    /* copy tids information so it is available to eco */
    ECO_tids = (int *) malloc ((p+1) * sizeof(int));
    for(i=0;i<ECO_p;i++)
      ECO_tids[i] = tids[i];

    if(eco_method==0){
      if(eco_pattern_init(*p_i,p,tids))
	return 1;
    }
    else {
      if(eco_other_pattern_init(*p_i,p,tids))
	return 1;
    }


  return 0;
}



/* eco_database_init */
/* Prototype: int eco_database_init(int p_i, int p, int* tids, 
                                     char* data_filename) */
/* Inshort: initializes eco network database */
/* Description: */
/*     This procedure reads in the database file and sets up the global */
/*  variables.  Only the parent (p_i=0) reads the file directly. */
/* Parameters: */
/*     p_i           :  pvm_mytid == tids[p_i]   0 <= p_i < p */
/*     p             : number of tasks participating in collective comm. */
/*                     if p=0(or 1), then read in file, but don't transmit */
/*     tids          : array of tids */
/*     data_filename : network database filename (used by p_i = 0) */
static int eco_database_init (int p_i, int p, int* tids, char* data_filename)
{
  char hostname_buffer[MaxStr];
  int i,j,k;
  FILE *net_data;
  
  /* first section of the code is when p_i = 0, so the file is read */
  /* in and then broadcast to all hosts */
  if(p_i == 0) {

    if(!have_database) {
      if(data_filename == 0)
	net_data = fopen(DEFAULT_NETWORK_DATA_FILENAME, "r");
      else
	net_data = fopen(data_filename, "r");
      if (net_data == NULL) {
	perror("eco_init: Unable to open network data file");
	abort();
	return 1;
      }
      
      while(fgets(hostname_buffer, MaxStr, net_data)==hostname_buffer)
	if(strcmp("---------\n", hostname_buffer)==0)
	  break;
      if(feof(net_data)||(ferror(net_data))) {
	fprintf(stderr, 
		"eco_init: unable to find starting row\n");
	return 1;
      }
      
      
      if(fscanf(net_data, "subnets: %i", &subnet_count)!= 1) {
	fprintf(stderr, "eco_init: subnets: not found\n");
	return 1;
      }
      
      subnet_by_name = (char ***) malloc(subnet_count * sizeof(char **));
      inter_subnet_times = (double **) malloc(subnet_count * sizeof(double*));
      subnet_size = (int *) malloc (subnet_count * sizeof(int));
      for(i=0;i<subnet_count;i++)
	inter_subnet_times[i] = (double *) 
	  malloc (subnet_count * sizeof(double));
      
      for(i=0;i<subnet_count;i++) {
	if(fscanf(net_data, "%i", subnet_size+i) != 1) {
	  fprintf (stderr, "eco_init: subnet %i size not found\n", i);
	  return 1;
	}
	subnet_by_name[i] = (char **) malloc(subnet_size[i] * sizeof(char*));
	
	for(j=0;j<subnet_size[i];j++) {
	  if(fscanf(net_data, "%s", hostname_buffer) != 1) {
	    fprintf (stderr, "eco_init: hostname problem on subnet %i, host %i.\n", 
		     i, j);
	    return 1;
	  }
	  subnet_by_name[i][j] = (char *) malloc ((strlen(hostname_buffer)+1)*
						 sizeof(char));
	  strcpy(subnet_by_name[i][j], hostname_buffer);
	}
      }
      
      for(i=0;i<subnet_count;i++)
	for(j=i+1;j<subnet_count;j++)
	  if(fscanf(net_data, "%lf",&(inter_subnet_times[i][j])) != 1) {
	    fprintf (stderr, "eco_init: subnet size problem %i,%i\n",
		     i, j);
	    return 1;
	  }


      fscanf(net_data, " method: %i", &eco_method);
      /*fprintf(stderr, "eco_init: using method %i\n", 
	      eco_method);*/
      
      fclose(net_data);
    
      have_database = TRUE;
    }

    /* transmit database to other tasks */
    if(p>1) {
      pvm_initsend(PvmDataDefault);
      pvm_pkint(&subnet_count, 1, 1);
      pvm_pkint(subnet_size, subnet_count, 1);
      
      for(i=0;i<subnet_count;i++)
	for(j=0;j<subnet_size[i];j++) {

	  k = strlen(subnet_by_name[i][j]);
	  pvm_pkint(&k,1,1);
	  pvm_pkstr(subnet_by_name[i][j]);
	}

      for(i=0;i<subnet_count;i++)
	for(j=i+1;j<subnet_count;j++)
	  pvm_pkdouble(&inter_subnet_times[i][j], 1, 1);
      pvm_pkint(&eco_method,1,1);
      pvm_mcast(tids+1, p-1, ECO_NET_DATA);
    }
  }
  /* not parent --- receive data file
     NOTE: setting p=0 for non-parent won't help anything */
  else {
    pvm_recv(tids[0], ECO_NET_DATA);

    pvm_upkint(&subnet_count, 1, 1);
    
    subnet_by_name = (char ***) malloc(subnet_count * sizeof(char **));
    inter_subnet_times = (double **) malloc(subnet_count * sizeof(double*));
    subnet_size = (int *) malloc (subnet_count * sizeof(int));
    for(i=0;i<subnet_count;i++)
      inter_subnet_times[i] = (double *) 
	malloc (subnet_count * sizeof(double));

    pvm_upkint(subnet_size, subnet_count, 1);


    for(i=0;i<subnet_count;i++) {
      subnet_by_name[i] = (char **) malloc (subnet_size[i] * sizeof(char*));
      for(j=0;j<subnet_size[i];j++) {
	pvm_upkint(&k,1,1);
	subnet_by_name[i][j] = (char *) malloc((k+1)*sizeof(char));
	
	pvm_upkstr(subnet_by_name[i][j]);
      }
    }

    for(i=0;i<subnet_count;i++)
      for(j=i+1;j<subnet_count;j++)
	pvm_upkdouble(&inter_subnet_times[i][j],1,1);
    pvm_upkint(&eco_method,1,1);
    have_database = TRUE;
  }

  return 0;
}


/* eco_subnet_init */
/* Prototype: int eco_subnet_init(int p, int* tids) */
/* Inshort: determines subnet membership */
/* Description: */
/*    This procedure searches for each host in the database and */
/*  sets subnet_map[tid_index] to that subnet */
/* Parameters: */
/*     p             : number of tasks participating in collective comm. */
/*     tids          : array of tids */
static int eco_subnet_init(int p, int* tids)
{
  int i,j,k;
  struct pvmtaskinfo *taskp; 
  struct pvmhostinfo *hostp; /* information on the virtual machine */

  subnet_map = (int *) malloc (p * sizeof(int));

  for(i=0;i<p;i++)
    subnet_map[i] = MAX_P;

  pvm_config(&i, &j, &hostp);

  /*if(p==0) printf ("\ncurrent host ordering:\n");*/
  for (i=0;i< p;i++) {
    char *cur_host_name;
    pvm_tasks(tids[i], &j, &taskp);
    cur_host_name = pvmhostname(hostp, taskp[0].ti_host);
    
    for(j=0;(j<subnet_count) && subnet_map[i]==MAX_P;j++)
      for(k=0;(k<subnet_size[j])&&subnet_map[i]==MAX_P;k++)
	if(strncmp(cur_host_name, subnet_by_name[j][k],
		   min(strchr(cur_host_name, '.')==NULL?strlen(cur_host_name):
		       strchr(cur_host_name,'.')-cur_host_name,
		       strchr(subnet_by_name[j][k], '.')==NULL?
		       strlen(subnet_by_name[j][k]):
		       strchr(subnet_by_name[j][k],'.')-subnet_by_name[j][k]))
	   == 0)
	  subnet_map[i] = j;

    /*if(p==0) printf ("%s (%i)\n", cur_host_name, subnet_map[i]);*/

  }

  /*if(p==0) printf ("\n");*/

  return 0;
}    


/* eco_pattern_init */
/* Prototype: int eco_pattern_init(int p_i, int p, int* tids) */
/* Inshort: set up communication patterns for common operations */
/* Description: */
/*    This routine sets up the communication trees which are used for  */
/*  collective operations */
  /* overview of pattern method */
  /* - the tid list is sorted by subnet */
  /* - determine the range of each subnet in the tid array */
  /* - loop for each host as root */
  /*    if root is in my subnet  */
  /*        if i'm root  */
  /*           set children to the first from each subnet and all of */
  /*           my subnet  */
  /*        not root */
  /*           parent is root */
  /*    not in my subnet */
  /*        if i'm first in my subnet */
  /*           parent is root and children are rest of subnet */
  /*        not first */
  /*           parent is first in subnet */
static int eco_other_pattern_init(int p_i, int p, int* tids)
{
  int i,j, root;
  int base_tree_c_c, assign_index;
  int subnet_start;
  int desc_id;
  int ECO_p_i;

  ECO_p = p;
  ECO_myid = p_i;
  /* ECO_ALL communication is sort of faked by putting the patterns
     at index p, thus ECO_ALL is a variable, not a constant */
  ECO_ALL = p;
  /* copy tids information so it is available to eco */
  ECO_tids = (int *) malloc ((p+1) * sizeof(int));
  for(i=0;i<ECO_p;i++)
    ECO_tids[i] = tids[i];
  /* - the tid list is sorted by subnet */
  for(i=0;i<p-1;i++)
    for(j=0;j<p-i-1;j++)
      if (subnet_map[j] > subnet_map[j+1]) {
	int temp_subnet_map, temp_tid;
	temp_subnet_map = subnet_map[j];
	temp_tid = ECO_tids[j];
	subnet_map[j] = subnet_map[j+1];
	ECO_tids[j] = ECO_tids[j+1];
	subnet_map[j+1] = temp_subnet_map;
	ECO_tids[j+1] = temp_tid;
      }


  /* allocate space for communication patterns */
  ECO_tree_parent = (int*) malloc ((p+1) * sizeof(int));
  ECO_tree_child_count = (int*) malloc ((p+1) * sizeof(int));
  ECO_tree_children = (int**) malloc ((p+1) * sizeof(int*));
  ECO_tree_desc_info = (int**) malloc ((p+1)*sizeof(int*));
  for(i=0;i<p+1;i++)
    ECO_tree_desc_info[i] = (int*) malloc (p*sizeof(int));
  ECO_tree_desc_max = (int**) malloc ((p+1)*sizeof(int*));
  
	    
 

  /* - determine the range of each subnet in the tid array */
  if(eco_method == 0) {
    /* first determine subnet ranges*/
    int subnet_start, subnet_end;
    int subnet_iam;
    for(subnet_start=ECO_p_i;subnet_start>0&&
	  subnet_map[subnet_start-1]==subnet_map[subnet_start];
	subnet_start--);
    for(subnet_end=ECO_p_i;(subnet_end+1)<p&&
	  subnet_map[subnet_end+1]==subnet_map[subnet_end];
	subnet_end++);
    subnet_iam = ECO_p_i-subnet_start;

    /* - loop for each host as root */
    for(root=0;root<p;root++)
      /*    if root is not in my subnet */
      /* root isn't in my subnet---so just do the normal thing*/
      if((root < subnet_start) || (root > subnet_end)) {
	/*        if i'm first in my subnet */
	/*           parent is root and children are rest of subnet */
#if 0
	if(ECO_p_i == subnet_start) {
	  ECO_tree_parent[root] = ECO_tids[root];
	  if((ECO_tree_child_count[root] = subnet_end-subnet_start)> 0) {
	    if((ECO_tree_children[root] = (int *) malloc ((subnet_end-subnet_start)
						      * sizeof (int)))==0)
	      {
		fprintf (stderr, "malloc returned 0\n");
		abort();
	      }
	    
	    assign_index = 0;
	    for(i=subnet_start+1;i<=subnet_end;i++)
	      ECO_tree_children[root][assign_index++] = ECO_tids[i];
	  }
	}
	else {
	  /*        not first */
	  /*           parent is first in subnet */
	  ECO_tree_parent[root] = ECO_tids[subnet_start];
	  ECO_tree_child_count[root] = 0;
#endif
	  if(ECO_p_i==subnet_start)
	    ECO_tree_parent[root]=ECO_tids[root];
	  else
	    ECO_tree_parent[root] = ECO_tids[subnet_start+((subnet_iam-1)/TREE_K)];
	  if(subnet_start+subnet_iam*TREE_K+TREE_K <= subnet_end)
	    ECO_tree_child_count[root] = TREE_K;
	  else
	    ECO_tree_child_count[root] =
	      TREE_K-(subnet_start+subnet_iam*TREE_K+TREE_K - subnet_end);
	  if(ECO_tree_child_count[root]<0)
	    ECO_tree_child_count[root] = 0;
	  if(ECO_tree_child_count[root] > 0) {
	    ECO_tree_children[root] = (int*) malloc
	      (ECO_tree_child_count[root]*sizeof(int));
	    for(i=0;i<ECO_tree_child_count[root];i++)
	      ECO_tree_children[root][i] =
		ECO_tids[subnet_start+TREE_K*subnet_iam+i+1];
	  }
	}

      else /* root is in my subnet */
	if (root == ECO_p_i) {
	  /*        if i'm root  */
	  /*           set children to the first from each subnet and all of */
	  /*           my subnet  */

	  ECO_tree_parent[root] = -1;
	  ECO_tree_child_count[root] = 
	    (subnet_end-subnet_start>=TREE_K)?TREE_K:subnet_end-subnet_start;
	  if(subnet_map[ECO_p_i] != subnet_map[0])
	    ECO_tree_child_count[root]++;
	  for(i=1;i<p;i++)
	    if((subnet_map[i-1]!=subnet_map[i]) &&
	       (subnet_map[i] != subnet_map[ECO_p_i]))
	      ECO_tree_child_count[root]++;
	  ECO_tree_children[root] = (int *)
	    malloc (ECO_tree_child_count[root] * sizeof(int));
	  assign_index = 0;
	  if(subnet_map[ECO_p_i] != subnet_map[0])
	    ECO_tree_children[root][assign_index++] =
	      ECO_tids[0];
	  for(i=1;i<p;i++)
	    if((subnet_map[i-1]!=subnet_map[i]) &&
	       (subnet_map[i] != subnet_map[ECO_p_i]))
	      ECO_tree_children[root][assign_index++] =
		ECO_tids[i];
	  j=root+1;
	  if(j>subnet_end) j=subnet_start;
	  for(i=0;(i<TREE_K)&&(j!=root);i++) {
	    ECO_tree_children[root][assign_index++] = ECO_tids[j];
	    if(++j>subnet_end) j=subnet_start;
	  }
	}
	else {
	  /*        not root */
	  /*           parent is root */
#if 0
	  ECO_tree_parent[root] = ECO_tids[root];
	  ECO_tree_child_count[root] = 0;
#endif
	  int subnet_size = subnet_end-subnet_start+1;
	  int fake_tids[subnet_size];
	  int fake_subnet_iam;

	  for(i=0,j=root;i<subnet_size;i++,(j+1>subnet_end?j=subnet_start:
					    j++))
	    fake_tids[i]=ECO_tids[j];
	  for(fake_subnet_iam=1;fake_tids[fake_subnet_iam]!=
		ECO_tids[ECO_p_i];fake_subnet_iam++);

	  ECO_tree_parent[root] = fake_tids[(fake_subnet_iam-1)/TREE_K];
	  if(fake_subnet_iam*TREE_K+TREE_K < subnet_size)
	    ECO_tree_child_count[root] = TREE_K;
	  else
	    ECO_tree_child_count[root] =
	      TREE_K-(fake_subnet_iam*TREE_K+TREE_K - subnet_size+1);
	  if(ECO_tree_child_count[root]<0)
	    ECO_tree_child_count[root] = 0;
	  if(ECO_tree_child_count[root] > 0) {
	    ECO_tree_children[root] = (int*) malloc
	      (ECO_tree_child_count[root]*sizeof(int));
	    for(i=0;i<ECO_tree_child_count[root];i++)
	      ECO_tree_children[root][i] =
		fake_tids[TREE_K*fake_subnet_iam+i+1];
	  }
	  
	  

	}
  }
  else if (eco_method==1) {
    for(root=0;root<p;root++) {
      if(ECO_myid == root) {
	ECO_tree_parent[root] = -1;
	ECO_tree_child_count[root] = p-1;
	ECO_tree_children[root] = (int*) malloc((p-1)*sizeof(int));
	assign_index = 0;
	for(i=0;i<p;i++)
	  if(i!=root) 
	    ECO_tree_children[root][assign_index++] = ECO_tids[i];
      }
      else {
	ECO_tree_child_count[root] = 0;
	ECO_tree_parent[root] = ECO_tids[root];
      }
    }
  }
  else if (eco_method == 2) {
    for(root=0;root<p;root++) {
      int fake_p_i = ((ECO_p_i-root)%p)+((ECO_p_i<root)?p:0);


      if(ECO_p_i!= root)
	ECO_tree_parent[root] = ECO_tids[(((fake_p_i-1)/TREE_K)+root)%p];
      else
	ECO_tree_parent[root] = -1;

      ECO_tree_children[root] = (int*) malloc (TREE_K*sizeof(int));
      ECO_tree_child_count[root] = 0;
      
      for(i=1;i<=TREE_K;i++)
	if((TREE_K*fake_p_i+i) < p)
	  ECO_tree_children[root][ECO_tree_child_count[root]++] =
	    ECO_tids[((TREE_K*fake_p_i+i)+root)%p];
    }
  }

  /* copy information from root=0 to root=ALL */
  ECO_tree_parent[ECO_ALL] = ECO_tree_parent[0];
  ECO_tree_child_count[ECO_ALL] = ECO_tree_child_count[0];
  ECO_tree_children[ECO_ALL] = (int*) malloc (ECO_tree_child_count[ECO_ALL] *
					   sizeof(int));
  for(i=0;i<ECO_tree_child_count[ECO_ALL];i++)
    ECO_tree_children[ECO_ALL][i] = ECO_tree_children[0][i];

  /* transfer descendant information */
  for(root=0;root<p;root++) {
    if(ECO_tree_parent[root] != -1)
      pvm_recv(ECO_tree_parent[root], ECO_BEGIN_DESC);
    pvm_initsend(PvmDataDefault);
    for(i=0;i<ECO_tree_child_count[root];i++)
      pvm_send(ECO_tree_children[root][i], ECO_BEGIN_DESC);

    /* initialize all pointers to default that links are above us */
    for(i=0;i<p;i++)
      ECO_tree_desc_info[root][i] = ECO_tree_child_count[root];

    for(i=0;i<ECO_tree_child_count[root];i++) {
      pvm_recv(ECO_tree_children[root][i], ECO_DESC_INFO);

      pvm_upkint(&desc_id, 1, 1);
      while (desc_id != -1) {
	ECO_tree_desc_info[root][desc_id] = i;
	pvm_upkint(&desc_id, 1, 1);
      }
    }

    if(ECO_tree_parent[root] != -1) {
      pvm_initsend(PvmDataDefault);
      for(i=0;i<p;i++)
	if((i== ECO_myid) ||
	   (ECO_tree_desc_info[root][i] != ECO_tree_child_count[root]))
	  pvm_pkint(&i, 1, 1);
      i=-1;
      pvm_pkint(&i,1,1);

      pvm_send(ECO_tree_parent[root], ECO_DESC_INFO);
    }

    /* send finish code */
    if(ECO_tree_parent[root] != -1)
      pvm_recv(ECO_tree_parent[root], ECO_DESC_FINISH);
    pvm_initsend(PvmDataDefault);
    for(i=0;i<ECO_tree_child_count[root];i++)
      pvm_send(ECO_tree_children[root][i], ECO_DESC_FINISH);
  }


  for(root=0;root<p;root++){
    ECO_tree_desc_max[root] = (int*) malloc((ECO_tree_child_count[root]+1)*
					sizeof(int));
    for(i=0;i<=ECO_tree_child_count[root];i++)
      ECO_tree_desc_max[root][i] = -1;
      
    for(j=0;j<p;j++)
      if(ECO_tree_desc_max[root][ECO_tree_desc_info[root][j]] < j)
	ECO_tree_desc_max[root][ECO_tree_desc_info[root][j]] = j;
  }

      

  for(i=0;i<p;i++)
    ECO_tree_desc_info[p][i] = ECO_tree_desc_info[0][i];
  ECO_tree_desc_max[p] =
    (int*) malloc((ECO_tree_child_count[0]+1)*sizeof(int));
  for(i=0;i<=ECO_tree_child_count[0];i++)
    ECO_tree_desc_max[p][i] = ECO_tree_desc_max[0][i];


#ifdef PATTERN_PRINT
  printf ("iam=%x\n", pvm_mytid());
  for(root=0;root<p;root++) {
    printf ("parent= %x\n", ECO_tree_parent[root]);
    printf ("c_c = %i\n", ECO_tree_child_count[root]);
    for(i=0;i<ECO_tree_child_count[root];i++)
      printf ("child %i:%x\n", i,ECO_tree_children[root][i]);
    for(i=0;i<p;i++)
      printf ("%1i", ECO_tree_desc_info[root][i]);
    printf ("\n");
    for(i=0;i<=ECO_tree_child_count[root];i++)
      printf ("%3i", ECO_tree_desc_max[root][i]);
    printf ("\n");
  }
  fflush(NULL);
#endif

  
  return 0;
}
      


/* eco_ring_topology */
/* Prototype: void eco_ring_topology(int* tids_in, int* tids_out, const int p) */
/* Inshort: returns tids in a pseudo-optimal ring topology */
/* Description: */
/*    This procedure reorders the tasks in tids_in so they are in a  */
/*   pseudo-optimal (communication time) ring order in tids_out. */
/*   tids_in and tids_out may be identical */
/* Parameters: */
/*   tids_in     : input list of tids */
/*   tids_out    : output list in pseudo-optimal ring ordering */
/*   p           : number of tasks */
void eco_ring_topology(int* tids_in, int* tids_out, int p)
{
  int i,j,k;

  struct pvmtaskinfo *taskp; 
  struct pvmhostinfo *hostp; /* information on the virtual machine */

  pvm_config(&i, &j, &hostp);

  if (tids_in != tids_out)
    memcpy(tids_out, tids_in, p * sizeof(int));
  
  /* arrange the subnets in an arbitrary order*/
  for(i=1;i<p-1;i++)
    for(j=1;j<p-i;j++)
      if (subnet_map[j] > subnet_map[j+1]) {
	int temp_subnet_map, temp_tid;
	temp_subnet_map = subnet_map[j];
	temp_tid = tids_out[j];
	subnet_map[j] = subnet_map[j+1];
	tids_out[j] = tids_out[j+1];
	subnet_map[j+1] = temp_subnet_map;
	tids_out[j+1] = temp_tid;
      }

  /* now rotate the list so tid 0 is in the right place */
  if(subnet_map[0] > subnet_map[1]) {
    i = 1;
    while(i<p && subnet_map[0] != subnet_map[i])
      i++;

    if(i<p) {
      int copy_buffer[p];

      for(j=1;j<i;j++)
	copy_buffer[j] = subnet_map[j];
      for(j=i;j<p;j++)
	subnet_map[j-i+1] = subnet_map[j];
      for(j=1;j<i;j++)
	subnet_map[j+(p-i)] = copy_buffer[j];

      for(j=1;j<i;j++)
	copy_buffer[j] = tids_out[j];
      for(j=i;j<p;j++)
	tids_out[j-i+1] = tids_out[j];
      for(j=1;j<i;j++)
	tids_out[j+(p-i)] = copy_buffer[j];
    }
  }

#ifdef PATTERN_PRINT
  printf("\nnew host ordering:\n");
  for (i=0;i< p;i++) {
    char *cur_host_name;
    pvm_tasks(tids_out[i], &j, &taskp);
    cur_host_name = pvmhostname(hostp, taskp[0].ti_host);
    printf ("%s (%i)\n", cur_host_name, subnet_map[i]);
  }
  printf ("\n");
#endif

}
      


/* eco_barrier() */

/* waits until all processes reach call before completing */
int eco_barrier(void)
{
  int i;

  ECO_NEW_MTAG;

  for (i=0;i<ECO_tree_child_count[ECO_opt_tree];i++)
    pvm_recv(ECO_tree_children[ECO_opt_tree][i], ECO_BARRIER);


  pvm_initsend(PvmDataDefault);

  if(ECO_tree_parent[ECO_opt_tree] != -1) {
    pvm_send(ECO_tree_parent[ECO_opt_tree], ECO_BARRIER);
    pvm_recv(ECO_tree_parent[ECO_opt_tree], ECO_BARRIER);
  }

  for(i=0;i<ECO_tree_child_count[ECO_opt_tree];i++)
    pvm_send(ECO_tree_children[ECO_opt_tree][i], ECO_BARRIER);

  return 0;
}



/* eco_bcast(IN int root) */

/* Broadcasts the current send buffer from the root node to all other */
/* nodes.  All processes must call this or other processes may not */
/* receive the message.  Message is received into a new receive buffer, */
/* as with pvm_recv. */
int eco_bcast(int root)
{
  int i;
  int cc;

  ECO_NEW_MTAG;

  if(ECO_myid == root) {
    for(i=0;i<ECO_tree_child_count[root];i++)
      cc=pvm_send(ECO_tree_children[root][i], ECO_BCAST_MSG);
  }
  else {
    cc=pvm_recv(ECO_tree_parent[root], ECO_BCAST_MSG);
    cc=pvm_setsbuf(pvm_getrbuf());
    for(i=0;i<ECO_tree_child_count[root];i++)
      cc=pvm_send(ECO_tree_children[root][i], ECO_BCAST_MSG);
  }
  cc=pvm_setrbuf(pvm_getsbuf());

  return 0;
}  


/* eco_register_function(function) */

/* registers function as a binary operation for reduce and scan.  returns */
/* the (integer) identifier to be passed to reduce and scan */

/* The function should be of type */

/* void ECO_user_function(void* In, void* InOut, int length, int* info); */

/* and should implement the operation */

/* for(i=0;i<length;i++) */
/*    InOut[i] = InOut[i] (+) In[i]; */

/* info is reserved for returning status, although it is currently not used. */
/* (+) must be both associative and commutative */
int eco_register_function( ECO_user_function operation)
{
  int i=10;

  while(eco_oper_list[i] != NULL)
    i++;

  eco_oper_list[i] = operation;
  return i;
}


/* eco_delete_function(int function_id) */

/* deletes the function previously registered */
int eco_delete_function(int function_id)
{
  eco_oper_list[function_id] = NULL;

  return 0;
}

/* return a pointer to the hostname with pvmd tid ti_host */
static char *pvmhostname(struct pvmhostinfo *hostp, int ti_host)
{
  int hostindex;

  for (hostindex = 0; hostp[hostindex].hi_tid!=ti_host;hostindex++);

  return (hostp[hostindex].hi_name);
}

