/** @file levels.c
 *  @brief Map level generator.
 *
 *  @author Michael Ashley-Rollman (mpa)
 *  @bug No known bugs.
 */

#include "levels.h"

#include <malloc.h>
#include <rand.h>
#include <stdlib.h>

drivelevel_t * levels[MAX_LEVELS];  
int numlevels = 0;

#define GO_LEFT 0
#define GO_RIGHT 1
#define STAY_PUT 2

/** @brief Creates a random level map
  *
  * This function creates a level of the specified height 
  * and returns the struct encapsulating it.
  *
  * This function is documented here as it is not intended to be 
  * exposed outside the levels.c file and is not declared in levels.h
  *
  * @param height The desired height of the level
  * @param max_left_offset The maximum offset allowed in the left direction
  * @param max_right_offset The maximum offset allowed in the right direction
  * @return A pointer to the level struct or NULL on error
  */
drivelevel_t *create_level(int height, int max_left_offset, int max_right_offset)
{
  drivelevel_t *new_level = malloc(sizeof(drivelevel_t));
  int *offsets = malloc(height * sizeof(int));

  int i;
  int dir = STAY_PUT;
  
  if (new_level == NULL || offsets == NULL)
  {
    free(new_level);
    free(offsets);

    return NULL;
  }

  new_level->height = height;
  new_level->map[0] = 0;
  
  for (i = 1; i < height; i++)
  {
    switch(genrand() & 7) {
      case GO_LEFT:
        dir = GO_LEFT;
        break;

      case GO_RIGHT:
        dir = GO_RIGHT;
        break;

      case STAY_PUT:
        dir = STAY_PUT;
        break;
    }
     
    switch (dir) {
   
      case GO_LEFT:
        new_level->map[i] = new_level->map[i-1] - 1;
        if (new_level->map[i] < max_left_offset)
          new_level->map[i] = max_left_offset;
        break;
        
      case GO_RIGHT:
        new_level->map[i] = new_level->map[i-1] + 1;
        if (new_level->map[i] > max_right_offset)
          new_level->map[i] = max_right_offset;
        break;
        
      case STAY_PUT:
        new_level->map[i] = new_level->map[i-1];
        break;
    }
  }

  return new_level;
}


void seed_levels (int seed)
{
  sgenrand (seed);
}
  
int init_levels(int num_levels, int height, int max_left_offset, int max_right_offset)
{
  drivelevel_t **temp_levels;
  int i;

  if (num_levels > MAX_LEVELS) 
    return -1;
  
  temp_levels = malloc(sizeof(drivelevel_t *) * num_levels);
  
  if (temp_levels == NULL)
    return -1;

  for(i = 0; i < num_levels; i++) {
    temp_levels[i] = create_level(height, max_left_offset, max_right_offset);

    if (temp_levels[i] == NULL) {
      for(; i >= 0; i--)
        free(temp_levels[i]);

      free(temp_levels);

      return -1;
    }
  }

  for (i = 0; i < numlevels; i++)
    free(levels[i]);

  numlevels = num_levels;

  for (i = 0; i < num_levels; i++)
    levels[i] = temp_levels[i];

  free(temp_levels);

  return 0;
}
