/*
 * disk.c
 * Steve Muckle
 * 15-412 Project Set
 *
 * Hammered on by Zach Anderson
 * and Dave Eckhardt for 15-410.
 *
 * An earlier version of this
 * driver contained code to prompt
 * the "user" for permission to
 * write to sector 0 or into an
 * extended partition, etc.
 * All gone.
 *
 * This code is NOT a good example
 * of what we'd like you to write,
 * but it's probably better for you
 * to have it now than later.
 *
 * Also, it is entirely possible it
 * contains a bug or two...luckily,
 * your OS debugging skills should
 * now be excellent.  Let us know.
 *
 */

/* --- Includes --- */
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <keyboard.h>
#include <interrupts.h>
#include <x86/proc_reg.h>
#include <x86/seg.h>
#include <x86/pio.h>
#include <kerndebug.h>
#include <common.h>

#include <disk.h>

/* --- Global Variables -- */

/* 
 * Selects which disk will be used for data transfers. 
 */
int disk_select = 0;

/* 
 * Disk information structures (filled in by disk_init). 
 */
disk_t ide0;
disk_t ide1;

/* 
 * Keep track of what kind of disk op we're doing, 
 * where the transfer is from/to, and how many blocks 
 * there are involved. 
 */
volatile int ide_busy = 0;
char* io_buf = NULL;
int block_cnt = 0;

/* 
 * Disk completion callback functions. 
 */

static void disk_pcheck_callback(void);

disk_callback_t callbacks[2];
/*
 * Pointers to the current callback functions.
 */
disk_callback_t current_read_callback;
disk_callback_t current_write_callback;

static int setup_address( int lba_addr, int blocks );
static void dump_status();
static void dump_error();
static int partition_problem(void);

static int initcode = ERROR;

/*
 * int get_num_sectors(int drive)
 * Returns the number of sectors on the drive.
 */
int get_num_sectors() 
{
  return ide0.blks;
}

/*
 * int disk_init()
 * Installs the disk interrupt handler via disk_install(),
 * waits until the disks are ready, and finds out how big
 * they are (filling in the ide0 and ide1 data structures).
 */
int
disk_init(disk_callback_t read_callback, disk_callback_t write_callback)
{
    unsigned char error_code;
    int i;
    short sector_buf_s[256];

    if (initcode == SUCCESS) 
        return SUCCESS;

    printf("disk_init(): waiting for DRIVE_READY...");

    /* 
     * The IDE controller clears the DRIVE READY bit upon power up.
     * We need to wait for this bit to clear. Also wait for heads
     * to settle over track (DRIVE SEEK COMPLETE). 
     */
    while( inb_p( IDE_STATUS_REGISTER ) !=
           (IDE_STATUS_DRIVE_READY | IDE_STATUS_DRIVE_SEEK_COMPLETE))
        ;

    printf("done.\n");

    /* 
     * Probe hard disks to make sure they are working correctly. 
     */
    outb_p( IDE_COMMAND_REGISTER, IDE_COMMAND_DIAGNOSTIC );
    while( (inb_p( IDE_STATUS_REGISTER ) & IDE_STATUS_DRIVE_BUSY ) )
        ;

    /* 
     * Read error code from drive 0. 
     */
    error_code = inb_p(IDE_ERROR_REGISTER);

    /* 
     * Check error code. The top bit of drive 0's error code 
     * indicates whether there is an error on drive 1. 
     */
    if ((error_code & 0xF) != 0x1 && error_code) {
    
        /* Drive 0 has responded with some sort of hardware error. */
        lprintf_kern("Drive 0 has a hardware error!");
        switch(error_code & 0xF) {
            case 0x02:
                lprintf_kern("Drive 0 reported a formatter device error.");
            break;
            case 0x03:
                lprintf_kern("Drive 0 reported a sector buffer error.");
            break;
            case 0x04:
                lprintf_kern("Drive 0 reported a ECC circuitry error.");
            break;
            case 0x05:
                lprintf_kern("Drive 0 reported a controlling microprocessor error.");
            break;
            default:
                lprintf_kern("Drive 0 reported an unknown error. 0x%x", error_code);
        }
        return ERROR;
    }

    if (error_code & 0x80) {
        /* 
         * Drive 1 has responded with some sort of hardware error. 
         */
        lprintf_kern("Drive 1 has a hardware error!\n");
        return ERROR;
    } 
  
    /* 
     * Get drive parameters for drive 0. 
     */
    outb_p( IDE_DRIVE_HEAD_REGISTER, 0xe0 );

    /* 
     * Enable disk interrupts. 
     */
    outb_p( IDE_DEVICE_CONTROL_REGISTER, 0x80 );
    ide_busy = IDE_BUSY_DIAGNOSTIC;
    outb_p( IDE_COMMAND_REGISTER, IDE_COMMAND_IDENTIFY_DRIVE );

    /* 
     * For now, poll for the status. 
     */

    while(ide_busy)
        ;

    /* 
     * Copy bytes into sector buffer. 
     */
    for( i=0; i<256; i++ ) 
        sector_buf_s[i] = inw_p(IDE_DATA_REGISTER);

    /* 
     * Get the drive parameters. 
     */
    ide0.cyl = sector_buf_s[IDE_IDENTIFY_NUM_CYLINDERS];
    ide0.heads = sector_buf_s[IDE_IDENTIFY_NUM_HEADS];
    ide0.spt = sector_buf_s[IDE_IDENTIFY_NUM_SECTORS_TRACK];
    ide0.blks = ide0.cyl * ide0.heads * ide0.spt;

#ifdef ENABLE_SECOND_DISK
    /* 
     * Get drive parameters for drive 1. 
     */
    outb(0xF0,IDE_DRIVE_HEAD_REGISTER);
    outb(0x8,IDE_DEVICE_CONTROL_REGISTER);
    ide_busy = IDE_BUSY_DIAGNOSTIC;
    outb(IDE_COMMAND_IDENTIFY_DRIVE, IDE_COMMAND_REGISTER);

    /* 
     * For now, poll for the status. 
     */
    while(ide_busy)
        ;

    /* 
     * Copy bytes into sector buffer. 
     */
    for( i=0; i<256; i++ )
        sector_buf_s[i] = inw(IDE_DATA_REGISTER);

    /* 
     * Get the drive parameters. 
     */
    ide1.cyl = sector_buf_s[IDE_IDENTIFY_NUM_CYLINDERS];
    ide1.heads = sector_buf_s[IDE_IDENTIFY_NUM_HEADS];
    ide1.spt = sector_buf_s[IDE_IDENTIFY_NUM_SECTORS_TRACK];
    ide1.blks = ide1.cyl * ide1.heads * ide1.spt;
#endif

    dump_status();

    initcode = SUCCESS; /* so partition_problem() can do I/O */

    if (partition_problem()) {
	int i = 20;
	while (i-- > 0)
		printf("Safety abort: this disk contains a valid partition table!\n");
	initcode = ERROR;
    } else {
        /* turn on the user */
        current_read_callback = callbacks[0] = read_callback;
        current_write_callback = callbacks[1] = write_callback;
    }

    return (initcode);
}

/*
 * void select_drive(int drive)
 * Selects the drive, 0 or 1, that will be used for future
 * reads and writes.
 */
void 
select_drive( int drive ) {
#ifdef ENABLE_SECOND_DISK
    if( drive != 0 && drive != 1 )
        return;
    else
        disk_select = drive;
#endif
}

/* 
 * int setup_address(int lba_addr, int blocks)
 * Sends LBA address, drive number, and the number of sectors
 * to transfer to IDE controller. 
 */
static int
setup_address( int lba_addr, int blocks ) {
    char temp;

    /* 
     * We want to transfer 1 sector. 
     */
    outb_p( IDE_SECTOR_COUNT_REGISTER, blocks );

    /* 
     * Make sure LBA address is within bounds. 
     */
    if (disk_select) {
        if (lba_addr < 0 || lba_addr >= ide1.blks) {
            lprintf_kern( "Invalid write attempt to disk 1: block %d/%d\n",
	                  lba_addr,ide1.blks );
            return ERROR;
        }
    } 
    else {
        if (lba_addr < 0 || lba_addr >= ide0.blks) {
            lprintf_kern( "Invalid write attempt to disk 0: block %d/%d\n",
	                  lba_addr,ide0.blks );
            return ERROR;
        }
    }

    /* 
     * LBA address bits...
     * Sector number register: bits 0-7.
     * Cylinder low register: bits 8-15.
     * Cylinder high register: bits 16-23.
     * Drive head register (bits 0-3): bits 24-27.
     *
     * Set up LBA address and drive number in registers. 
     */
    outb_p( IDE_SECTOR_NUMBER_REGISTER, lba_addr & 0xff );
    outb_p( IDE_CYLINDER_LOW_REGISTER, (lba_addr & 0xff00) >> 8 );
    outb_p( IDE_CYLINDER_HIGH_REGISTER, (lba_addr & 0xff0000) >> 16 );

    temp = (lba_addr & 0xF000000)>>24;
    if (disk_select) 
        temp |= 0xF0;
    else 
        temp |= 0xE0;

    outb_p( IDE_DRIVE_HEAD_REGISTER, temp );

    return SUCCESS;
}

/*
 * void disk_write_sector(int lba_addr, char* source, int num)
 *                      
 * Initiates an IDE operation to write the specified sectorss
 * to the disk. The sectors will be taken sequentially from the
 * address source.
 */
void 
disk_write_sector( int lba_addr, char* source, int num ) {
    int i;

    if (!num || num < 0) 
        panic("disk_write_sector(): bad count %d!\n", num);

    if( initcode != SUCCESS )
        panic("disk_write_sector(): must disk_init() successfully first!\n");

    if (ide_busy != IDE_NOT_BUSY) {
        lprintf_kern("disk_write_sector called with an outstanding IDE request!\n");
        panic("disk_write_sector called with an outstanding IDE request!\n");
    }

    /* 
     * Setup address registers. 
     */
    if (setup_address(lba_addr,num) == ERROR) 
        return;

    /* 
     * Set ide_busy. 
     */
    ide_busy = IDE_BUSY_WRITE;

    /* 
     * Send the write command code. 
     */
    outb_p( IDE_COMMAND_REGISTER, IDE_COMMAND_WRITE_SECTORS );

    /* 
     * Must wait for DRQ to be set. Eventually we might want to break out
     * and do useful work, then come back to this (there is no interrupt
     * after this). 
     */
    while( !(inb_p( IDE_STATUS_REGISTER ) & 0x8))
        ;

    /* 
     * Write the first sector into the buffer. 
     */
    io_buf = source + SECTOR_SIZE;
    block_cnt = num;
    for( i=0; i<256; i++ )
        outw_p( IDE_DATA_REGISTER, ((short *)source)[i] );  
}

/*
 * void disk_read_sector(int lba_addr, char* source, int num)
 *                  
 * Initiates an IDE operation to read the specified sectors
 * from the disk. The sectors will be stored sequentially starting
 * at the address source.
 */
void
disk_read_sector( int lba_addr, char* dest, int num ) {

    if( !num || num < 0 ) 
        panic("disk_read_sector(): bad count %d!\n", num);

    if( initcode != SUCCESS )
        panic("disk_read_sector(): must disk_init() successfully first!\n");

    if( ide_busy != IDE_NOT_BUSY ) {
        lprintf_kern("disk_read_sector called with an outstanding IDE request!\n");
        panic("disk_read_sector called with an outstanding IDE request!\n");
    }


    /* 
     * Setup address registers. 
     */
    if( setup_address( lba_addr,num ) == -1 ) 
        return;

    /* 
     * Set ide_busy. 
     */
    ide_busy = IDE_BUSY_READ;

    /* 
     * Save destination so interrupt handler knows where to copy
     * the data. 
     */
    io_buf = dest;

    /* 
     * Set block_cnt.
     */
    block_cnt = num;

    /* 
     * Send the read command code. 
     */
    outb_p( IDE_COMMAND_REGISTER, IDE_COMMAND_READ_SECTORS );
}

static void
disk_dismiss(void)
{
    outb_p( 0xA0, 0x60 | (0x7 & 14) ); /* this mess is steve's doings */
    outb_p( 0x20, 0x62 );
}

/*
 * void disk_handler()
 * Handles disk interrupts by reading in/writing out data 
 * and manipulating process queues.
 */
void 
disk_handler() {
    int i;
  
    /* 
     * Read status register to clear interrupt. 
     */
    inb_p(IDE_STATUS_REGISTER);

    switch(ide_busy) {
    case IDE_BUSY_READ:
      
        /* 
         * A pending read has completed. 
         */
        block_cnt--;
        if (!io_buf) {
            lprintf_kern("disk_handler: read with null io_buf");
            ide_busy = IDE_NOT_BUSY;
            return;
        }

        /* 
         * Save the data. 
         */
        for( i=0; i<256; i++ ) 
            ((short*)io_buf)[i] = inw_p(IDE_DATA_REGISTER);

        if (block_cnt) {
            /* 
             * If we're not done set the read pointer and wait for the next
	     * sector. 
             */
            io_buf += SECTOR_SIZE;
        } 
        else {

            /* 
             * If we're done just close up shop. 
             */
            io_buf = NULL;
            ide_busy = IDE_NOT_BUSY;

            disable_interrupts();
            /* Ack PIC *before* we potentially switch to another process!!! */
            disk_dismiss();
            current_read_callback();
        }
    break;
    case IDE_BUSY_WRITE:

        /* 
         * A pending write has completed. 
         */
        block_cnt--;
        if (block_cnt) {
            /* 
             * If we have more sectors to write then write another one. 
             */
            for( i=0; i<256; i++ )
                outw_p( IDE_DATA_REGISTER, ((short *)io_buf)[i] );
            io_buf += SECTOR_SIZE;
        } 
        else {
 
            /* If we're done just close up shop. */
            io_buf = NULL;
            ide_busy = IDE_NOT_BUSY;

            disable_interrupts();
            /* Ack PIC *before* we potentially switch to another process!!! */
            disk_dismiss();
            current_write_callback();

        }

    break;
    default:
        ide_busy = IDE_NOT_BUSY;
        disk_dismiss();
    }

}

/**
 * A call back that just returns...
 * For use in the partition table checker.
 *
 */
void
disk_null_callback() 
{
    enable_interrupts();
}

/**
 * Dumps the status of the disk
 * to the console and simics console.
 *
 */
static void
dump_status()
{
    unsigned int stat;

    stat = inb( IDE_STATUS_REGISTER );

    printf( "hd status = 0x%x\n", stat );
    lprintf_kern( "hd status = 0x%x", stat );

    if( stat & IDE_STATUS_DRIVE_BUSY ) {
        printf( "Busy\n" );
        lprintf_kern( "Busy" );
    }
    if( stat & IDE_STATUS_DRIVE_READY ) {
        printf( "DriveReady\n" );
        lprintf_kern( "DriveReady" );
    }
    if( stat & IDE_STATUS_DRIVE_WRITE_FAULT ) {
        printf( "WriteFault\n" );
        lprintf_kern( "WriteFault" );
    }
    if( stat & IDE_STATUS_DRIVE_SEEK_COMPLETE ) {
        printf( "SeekComplete\n" );
        lprintf_kern( "SeekComplete" );
    }
    if( stat & IDE_STATUS_DRIVE_DATA_REQUEST ) {
        printf( "DataRequest\n" );
        lprintf_kern( "DataRequest\n" );
    }
    if( stat & IDE_STATUS_DRIVE_CORRECTED_DATA ) {
        printf( "CorrectedError\n" );
        lprintf_kern( "CorrectedError" );
    }
    if( stat & IDE_STATUS_DRIVE_INDEX ) {
        printf( "Index\n" );
        lprintf_kern( "Index" );
    }
    if( stat & IDE_STATUS_DRIVE_ERROR ) {
        printf( "Error\n" );
        lprintf_kern( "Error" );
        dump_error();
    }
}

/**
 * Dumps the error status of the
 * disk to the console and simics
 * console.
 *
 */
static void
dump_error()
{
    unsigned int err;

    err = inb( IDE_ERROR_REGISTER );

    printf( "hd error = 0x%x\n", err );
    lprintf_kern( "hd error = 0x%x", err );

    if( err & IDE_ERROR_MARK ) {
       printf( "AddrMarkNotFount\n" );
       lprintf_kern( "AddrMarkNotFound" );
    }
    if( err & IDE_ERROR_TRK0 ) {
        printf( "TrackZeroNotFound\n" );
        lprintf_kern( "TrackZeroNotFound" );
    }
    if( err & IDE_ERROR_ABORT ) {
        printf( "DriveStatusError\n" );
        lprintf_kern( "DriveStatusError" );
    }
    if( err & IDE_ERROR_CHANGERQ ) {
        printf( "MediaChangeRequest\n" );
        lprintf_kern( "MediaChangeRequest" );
    }
    if( err & IDE_ERROR_ID ) {
        printf( "SectorIDNotFound\n" );
        lprintf_kern( "SectorIDNotFound" );
    }
    if( err & IDE_ERROR_CHANGED ) {
        printf( "MediaChanged\n" );
        lprintf_kern( "MediaChanged" );
    }
    if( err & IDE_ERROR_ECC ) {
        printf( "UncorrectableError\n" );
        lprintf_kern( "UncorrectableError" );
    }
    if( err & IDE_ERROR_BADSEC ) {
        printf( "BadSector\n" );
        lprintf_kern( "BadSector" );
    }
}

static unsigned char pcheck_buf[512];
static volatile int pcheck_tested = 0;
static int pcheck_problem = 0;

/**
 * Callback used by partition table checker.
 * Sets pcheck_tested and maybe pcheck_problem.
 *
 */
static void
disk_pcheck_callback(void)
{
    if( pcheck_buf[510] == 0x55 &&
        pcheck_buf[511] == 0xaa ) {

        pcheck_problem = 1;
    }
    pcheck_tested = 1;
    enable_interrupts();
}


/**
 * See if we have a partition "problem" (i.e.,
 * there is a partition label on this disk,
 * in which case we probably don't want to turn
 * a student kernel loose on it).
 *
 */
static int
partition_problem(void)
{
    disk_callback_t save;

    save = current_read_callback;
    current_read_callback = disk_pcheck_callback;

    disk_read_sector(0, pcheck_buf, 1);

    while (!pcheck_tested)
        ;

    current_read_callback = save;

    return (pcheck_problem);
}
