/* -*- Mode: C -*- */

/* trainer/client.C
 * CMUnited99 (code for off-line trainer)
 * Patrick Riley <pfr+@cs.cmu.edu>
 * Computer Science Department
 * Carnegie Mellon University
 * Copyright (C) 1999 Patrick Riley
 *
 * CMUnited-99 was created by Peter Stone, Patrick Riley, and Manuela Veloso
 *
 * You may copy and distribute this program freely as long as you retain this notice.
 * If you make any changes or have any comments we would appreciate a message.
 */

/* this file contains the main program for the off-lien trainer */

#include "client.h"
#include "types.h"
#include "netif.h"
#include "Memory.h"
#include "parse.h"
#include "utils.h"

Bool      wait_for_signals(sigset_t *);
sigset_t  init_handler();
void      sigio_handler(); 
void      sigalrm_handler();

/* Global variables -- don't want to reallocate buffers each time */

sigset_t sigiomask, sigalrmask;

Memory *Mem;

char     recvbuf[MAXMESG];	
char     sendbuf[MAXMESG];	

Socket sock;

int      alrsigs_since_iosig=0;
int      max_alrs_since_io;

/****************************************************************************************/

main(int argc, char *argv[])
{
  Mem = new Memory();

  if ( Mem == NULL ){
    my_error("couldn't allocate Mem");
    exit(0);
  }

  Mem->GetOptions(argc,argv);
  
  sock = init_connection(Mem->SP_host,Mem->SP_coach_port);

  if(sock.socketfd == -1) {
    cerr << "Can't open connection for trainer" << endl;
    exit(0);
  }
  
  Mem->Initialize();

  printf("Trainer started\n");

  max_alrs_since_io =
    ((float)Mem->TP_max_cycles_since_io) * Mem->SP_send_vi_step / Mem->TP_alarm_interval + 1.0;
  
  sigset_t sigfullmask = init_handler();	      

  Eye(TRUE);
  
  while ( Mem->ShutDown == FALSE && wait_for_signals(&sigfullmask) );

  if (sock.socketfd != -1) close_connection(&sock);

  Mem->NotifyTMOfShutdown();
  
  printf("Shutting down trainer");
}





/****************************************************************************/
/****************************************************************************/
/****************************************************************************/

/* set time interval between the sensor receiving and command sending */ 
inline void set_timer() {
  struct itimerval itv;
  itv.it_interval.tv_sec = 0;
  itv.it_interval.tv_usec = Mem->TP_alarm_interval * 1000;
  itv.it_value.tv_sec = 0;
  itv.it_value.tv_usec = Mem->TP_alarm_interval * 1000;
  setitimer(ITIMER_REAL, &itv, NULL);
}

inline void set_timer(int usec) {
  struct itimerval itv;
  itv.it_interval.tv_sec = 0;
  itv.it_interval.tv_usec = Mem->TP_alarm_interval * 1000;
  itv.it_value.tv_sec = 0;
  itv.it_value.tv_usec = usec;
  setitimer(ITIMER_REAL, &itv, NULL);
}

/*****************************************************************************/

sigset_t init_handler() { 
  sigemptyset(&sigiomask);
  sigaddset(&sigiomask, SIGIO);
  
  struct sigaction sigact;
  sigact.sa_flags = 0;
  sigact.sa_mask = sigiomask;

#ifdef Solaris
  sigact.sa_handler = (void (*)(int))sigalrm_handler; 
#else
  sigact.sa_handler = (void (*)(int))sigalrm_handler; 
#endif

  sigaction(SIGALRM, &sigact, NULL);
  sigact.sa_mask = sigalrmask;

#ifdef Solaris
  sigact.sa_handler = (void (*)(int))sigio_handler; 
#else
  sigact.sa_handler = (void (*)(int))sigio_handler; 
#endif

  sigaction(SIGIO, &sigact, NULL);
  set_timer();
  sigprocmask(SIG_UNBLOCK, &sigiomask, NULL);
  sigprocmask(SIG_UNBLOCK, &sigalrmask, NULL);

  sigset_t sigsetmask;
  sigprocmask(SIG_BLOCK, NULL, &sigsetmask);   /* Get's the currently unblocked signals */
  return sigsetmask;   
}


/**************************************************************************/

/* suspend the process until one of the signals comes through */
/* could check for situation to kill client, return FALSE     */
/* i.e. too many actions with no sensory input coming in      */
Bool wait_for_signals(sigset_t *mask){
  sigsuspend(mask);
  return TRUE;
}


/***************************************************************************/

/* SIGIO handler: receive and parse messages from server */
void sigio_handler() {
  sigprocmask(SIG_BLOCK, &sigalrmask, NULL);  
  int counter = 0;

  while (receive_message(recvbuf, &sock) == 1) {
    SenseType st = Parse(recvbuf, Mem);
    switch (st) {
    case ST_None:
      my_error("Could not incorporate sense");
      break;
    case ST_See:
      Mem->NotifyTMOfSight();
      break;
    case ST_HearPlayer:
      /* we may want to listen to the players sometime, but not now */
      /* Mem->NotifyTMOfSound(); */
      break;
    case ST_HearReferee:
      /* the info is already incorporated... */
      break;
    case ST_HearCoach:
    case ST_HearTrainer:
      break;
    case ST_Ok:
    case ST_Error:
      break;
    default:      
      my_error("Illegal value of SenseType: %d", st);
      break;
    }
    
    counter++;
  }

  sigprocmask(SIG_UNBLOCK, &sigalrmask, NULL);  

  alrsigs_since_iosig = 0;
  
  // if (counter>1) printf("Got %d messages\n",counter);
}

/*****************************************************************************/

/* we just use this to recognize when the server dies */
void sigalrm_handler() 
{
  alrsigs_since_iosig++;
  if (alrsigs_since_iosig > max_alrs_since_io) {
    my_error("Too many alarms between ios from server. Shutting down");
    Mem->ShutDown = TRUE;
  }
  
}

  
int client_send_message(char* buf, Socket *sock)
{
  if (Mem->MP_save_log)
    Mem->LogFile << ">> " << buf << endl << endl;
  return send_message(buf, sock);
}



/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/

const char* PmodeStrings[PM_MAX+1] = PLAYMODE_STRINGS;			

void ChangeMode(Pmode pm)
{
  sprintf(sendbuf, "(change_mode %s)", PmodeStrings[pm]);
  client_send_message(sendbuf, &sock);
  Mem->SetPlayMode(pm);
}

void CheckBall()
{
  client_send_message("(check_ball)", &sock);
}

void Ear(Bool OnOff)
{
  sprintf(sendbuf, "(ear %s)", OnOff ? "on" : "off");
  client_send_message(sendbuf, &sock);
}

void Eye(Bool OnOff)
{
  sprintf(sendbuf, "(eye %s)", OnOff ? "on" : "off");
  client_send_message(sendbuf, &sock);
}

void Look()
{
  client_send_message("(look)", &sock);
}

void MoveBall(Vector pos)
{
  sprintf(sendbuf, "(move (ball) %f %f)", pos.x, pos.y);
  client_send_message(sendbuf, &sock);
}

void MoveBall(Vector pos, Vector vel)
{
  /* have to send an angle if we send a velocity */
  sprintf(sendbuf, "(move (ball) %f %f 0 %f %f)", pos.x, pos.y, vel.x, vel.y);
  client_send_message(sendbuf, &sock);
}

void MovePlayer(TeamSide side, Unum num, Vector pos, float ang)
{
  if (ang == NOANG)
    sprintf(sendbuf, "(move (player %s %d) %f %f)",
	    (side == TS_Left)?Mem->MP_left_team_name:Mem->MP_right_team_name,
	    num, pos.x, pos.y);
  else
    sprintf(sendbuf, "(move (player %s %d) %f %f %f)",
	    (side == TS_Left)?Mem->MP_left_team_name:Mem->MP_right_team_name,
	    num, pos.x, pos.y, ang);
  client_send_message(sendbuf, &sock);
}

void Say(char* mess)
{
  /* the 't' indicates to our clients that this message is from a trainer */
  sprintf(sendbuf, "(say training %s)", mess);
  client_send_message(sendbuf, &sock);
}


