/* -*- Mode: C++ -*- */

/* trainer/TMturnball2.C
 * CMUnited99 (code for off-line trainer)
 * Patrick Riley <pfr+@cs.cmu.edu>
 * Computer Science Department
 * Carnegie Mellon University
 * Copyright (C) 1999 Patrick Riley
 *
 * CMUnited-99 was created by Peter Stone, Patrick Riley, and Manuela Veloso
 *
 * You may copy and distribute this program freely as long as you retain this notice.
 * If you make any changes or have any comments we would appreciate a message.
 */

/* this is a sample training module. See the README for details about this program */

/* this is the training module for the turnball2 behavior */

#include <unistd.h>
#include <iostream.h>
#include <fstream.h>
#include "client.h"
#include "utils.h"
#include "TMturnball2.h"
#include "MemPosition.h"

const char* TMTurnball2::Name = "Turnball2";

Bool TMTurnball2::Initialize()
{
  ChangeMode(PM_Play_On);
  MoveBall(10, 10); /* so we can move the player to middle */

  time_out_of_range = Mem->TP_turnball2_max_time_out+1;
  target_angle = 180; 
  curr_turn_time = 0;
  num_losses = -1; /* it's a loss right at the beginning */
  num_attempts = 0;

  cout << "Turnball2 Training module" << endl;
  
  return TRUE;
}

/* ranges to check:
   -max_turn_kick_pow: 5    - 60  by 5      num: 10
   -opt_ctrl_dist:     .35   - <1   by .05    num: 11)
   -closest_margin:    NOT VARIED.385 - .8  by .05   num: 9
   -dokick_factor:     NOT VARIED .1   - .7  by .02    num: 30  */
void TMTurnball2::InitializeEpochController(EpochController* pEC)
{
  pEC->num_vars = 4;
  strcpy(pEC->option_names[0], "max_turn_kick_pow");
  strcpy(pEC->option_names[1], "opt_ctrl_dist");
  strcpy(pEC->option_names[2], "closest_margin");
  strcpy(pEC->option_names[3], "dokick_factor");
  pEC->option_types[0] = TVT_Int;
  pEC->option_types[1] = TVT_Float;
  pEC->option_types[2] = TVT_Float;
  pEC->option_types[3] = TVT_Float;

  pEC->num_epochs = 1;
  pEC->training_epochs[0][0].i = 20;
  pEC->training_epochs[0][1].f = 1.6;
  pEC->training_epochs[0][2].f = .97;
  pEC->training_epochs[0][3].f = .22; 

  pEC->num_epochs = 0;
  for (int max_pow = 5; max_pow <= 60; max_pow += 5) {
    for (float opt_ctrl_dist = .35 ; opt_ctrl_dist < 1; opt_ctrl_dist += .05) {
      if (pEC->num_epochs >= pEC->MAX_NUM_EPOCHS) {
	my_error("Too many epochs wanted for turnball2!");
	return;
      }
      pEC->training_epochs[pEC->num_epochs][0].i = max_pow;
      pEC->training_epochs[pEC->num_epochs][1].f = opt_ctrl_dist;
      pEC->training_epochs[pEC->num_epochs][2].f = .585;
      pEC->training_epochs[pEC->num_epochs][3].f = .4;
      pEC->num_epochs++;
    }    
  }
  
  
#ifdef OLD_PARAMETER_VARIATIONS
  pEC->num_epochs = 0;
  for (float closest_margin=.385;
       closest_margin <= .8;
       closest_margin+=.05) {
    for (float dokick_factor=.1;
	 dokick_factor <= .7;
	 dokick_factor+=.02) {
      if (pEC->num_epochs >= pEC->MAX_NUM_EPOCHS) {
	my_error("Too many epochs wanted for turnball2!");
	return;
      }
      pEC->training_epochs[pEC->num_epochs][0].i = 50;
      pEC->training_epochs[pEC->num_epochs][1].f = .8;
      pEC->training_epochs[pEC->num_epochs][2].f = closest_margin;
      pEC->training_epochs[pEC->num_epochs][3].f = dokick_factor; 
      pEC->num_epochs++;
    }
  }
#endif
}


/* Does the following:
   -insures the player is at (0,0)
   -the player tries to go from one side and stop the ball
   -if the ball is out of kickable range for xx cycles, records a loss
*/
/* Assumes: -The relevant player is the first one on the left team */
void TMTurnball2::NewSightHandler() 
{
  
  pPlayerInfo pPI = Mem->GetPlayer(TS_Left, 1);
  if (pPI == NULL) {
    cout << "Trainer: Waiting for player to connect" << endl;
    return; /* no player yet */
  }
  
  if (pPI->pos != Vector(0,0)) {
    MovePlayer(TS_Left, 1, 0, 0);
  }

  curr_turn_time++;
  
  pBallInfo pBI = Mem->GetBall();
  if (pBI->pos.dist(pPI->pos) > Mem->SP_kickable_area) {
    time_out_of_range++;
    if (time_out_of_range >= Mem->TP_turnball2_max_time_out) {
      if (!Mem->TP_use_epochs)
	cout << "Client lost the ball" << endl;
      num_losses++;
      ResetBall();
      num_attempts++;
    }
  } else if (curr_turn_time > Mem->TP_turnball2_max_turn_time) {
    /* see if the ball got to the right place */
    float ball_ang = (pBI->pos - pPI->pos).dir();
    NormalizeAngleDeg(&ball_ang);
    cout << "ball_ang: " << ball_ang << endl;
    if ( fabs(ball_ang) > Mem->TP_turnball2_angle_buf) {
      if (!Mem->TP_use_epochs)
	cout << "Client failed to get ball to place" << endl;
      num_failures++;
    } else {
      if (!Mem->TP_use_epochs)
	cout << "Client succeded" << endl;
      num_successes++;
    }
    
    ResetBall();
    num_attempts++;
  }
  
}

void TMTurnball2::ResetBall()
{  
  MoveBall(Polar2Vector(Mem->TP_turnball_start_per * Mem->SP_kickable_area,
			range_random(-180, 180)));
  time_out_of_range = 0;
  curr_turn_time = 0;
  Say(""); /* so that the player forgets where the ball is */
}

void TMTurnball2::LogHeader(EpochController* pEC)
{
  ofstream outfile;

  outfile.open(Mem->TP_training_log_fn, ios::out);
  if (!outfile) {
    my_error("Could not open data file: %s", Mem->TP_training_log_fn);
    return;
  }
    
  /* write header */
  outfile << "Training Info for Turnball2 behavior" << endl;
  outfile << "Format: ";
  pEC->WriteOptionNames(outfile);
  outfile << "num_attempts num_successes num_losses num_failures";
  outfile << "perc_succ perc_loss perc_fail" << endl;
  outfile << "Epoch Length is: " << Mem->TP_epoch_length << endl;
  outfile.close();
}


void TMTurnball2::LogPerformance(EpochController* pEC)
{
  ofstream outfile;

  outfile.open(Mem->TP_training_log_fn, ios::app);
  if (!outfile) {
    my_error("Could not open data file: %s", Mem->TP_training_log_fn);
    return;
  }

  pEC->WriteCurrentOptionValues(outfile);
  outfile << num_attempts  << '\t'
	  << num_successes << '\t'
	  << num_losses    << '\t'
	  << num_failures  << '\t';
  outfile << ((float)num_successes)/num_attempts << '\t'
	  << ((float)num_losses)/num_attempts << '\t'
	  << ((float)num_failures)/num_attempts << '\t';
  outfile << endl;
  outfile.close();
}

void TMTurnball2::ResetForEpoch()
{
  /* reset the counting values */
  num_attempts = 0;
  num_successes = 0;
  num_losses = 0;
  num_failures = 0;
  /* reset the ball */
  ResetBall();
}

