/* -*- Mode: C++ -*- */

/* trainer/TMturnball.C
 * CMUnited99 (code for off-line trainer)
 * Patrick Riley <pfr+@cs.cmu.edu>
 * Computer Science Department
 * Carnegie Mellon University
 * Copyright (C) 1999 Patrick Riley
 *
 * CMUnited-99 was created by Peter Stone, Patrick Riley, and Manuela Veloso
 *
 * You may copy and distribute this program freely as long as you retain this notice.
 * If you make any changes or have any comments we would appreciate a message.
 */

/* this is a sample training module. See the README for details about this program */


/* this is the training module for the turnball behavior */

#include <unistd.h>
#include <iostream.h>
#include <fstream.h>
#include "client.h"
#include "utils.h"
#include "TMturnball.h"
#include "MemPosition.h"

const char* TMTurnball::Name = "Turnball";

Bool TMTurnball::Initialize()
{
  ChangeMode(PM_Play_On);
  MoveBall(10, 10); /* so we can move the player to middle */

  time_out_of_range = Mem->TP_turnball_max_time_out+1;
  target_angle = 180; 
  curr_turn_time = 0;
  num_losses = -1; /* it's a loss right at the beginning */
  turn_dir = TURN_CCW;

  cout << "Turnball Training module" << endl;
  
  return TRUE;
}

/* ranges to check:
   -max_turn_kick_pow: NOT VARIED5    - 50  by 5      num: 10
   -opt_ctrl_dist:     NOT VARIED (fix at .8) (.5   - 1   by .05    num: 11)
   -closest_margin:    .385 - .8  by .05   num: 9
   -dokick_factor:     .1   - .7  by .02    num: 30  */
void TMTurnball::InitializeEpochController(EpochController* pEC)
{
#ifdef OLD_PARAMETER_VARIATIONS
  pEC->num_vars = 4;
  strcpy(pEC->option_names[0], "max_turn_kick_pow");
  strcpy(pEC->option_names[1], "opt_ctrl_dist");
  strcpy(pEC->option_names[2], "closest_margin");
  strcpy(pEC->option_names[3], "dokick_factor");
  pEC->option_types[0] = TVT_Int;
  pEC->option_types[1] = TVT_Float;
  pEC->option_types[2] = TVT_Float;
  pEC->option_types[3] = TVT_Float;

  pEC->num_epochs = 1;
  pEC->training_epochs[0][0].i = 20;
  pEC->training_epochs[0][1].f = 1.6;
  pEC->training_epochs[0][2].f = .97;
  pEC->training_epochs[0][3].f = .22; 
#endif
  
  pEC->num_epochs = 0;
  for (float closest_margin=.385;
       closest_margin <= .8;
       closest_margin+=.05) {
    for (float dokick_factor=.1;
	 dokick_factor <= .7;
	 dokick_factor+=.02) {
      if (pEC->num_epochs >= pEC->MAX_NUM_EPOCHS) {
	my_error("Too many epochs wanted for turnball!");
	return;
      }
      pEC->training_epochs[pEC->num_epochs][0].i = 50;
      pEC->training_epochs[pEC->num_epochs][1].f = .8;
      pEC->training_epochs[pEC->num_epochs][2].f = closest_margin;
      pEC->training_epochs[pEC->num_epochs][3].f = dokick_factor; 
      pEC->num_epochs++;
    }
  }
}


/* Does the following:
   -insures the player is at (0,0)
   -if the ball is out of kickable range for 5 cycles, puts it next to player
   -counts number of times lost
   -counts number of times ball goes around
   -uses communication to tell the client what params to change */
/* Assumes: -The relevant player is the first one on the left team
            -Any said messages get through */
void TMTurnball::NewSightHandler()
{
  
  pPlayerInfo pPI = Mem->GetPlayer(TS_Left, 1);
  if (pPI == NULL) {
    cout << "Trainer: Waiting for player to connect" << endl;
    return; /* no player yet */
  }
  
  if (pPI->pos != Vector(0,0)) {
    MovePlayer(TS_Left, 1, 0, 0);
  }

  curr_turn_time++;
  
  pBallInfo pBI = Mem->GetBall();
  if (pBI->pos.dist(pPI->pos) > Mem->SP_kickable_area) {
    time_out_of_range++;
    if (time_out_of_range >= Mem->TP_turnball_max_time_out) {
      num_losses++;
      ResetBall();
    }
  } else {
    float ball_ang = (pBI->pos - pPI->pos).dir();
    if ( (target_angle == 0 &&
	  turn_dir*ball_ang < 0 &&
	  turn_dir*ball_ang > -90) ||
	 (target_angle == 180 &&
	  turn_dir*GetNormalizeAngleDeg(ball_ang+180) < 0 &&
	  turn_dir*GetNormalizeAngleDeg(ball_ang+180) > -90) ) {
      /* a successful half turn */
      /*printf("Time: %3d\tturnball\t(ball_ang, targ_ang)=(%f,%d)\n",
	     Mem->GetTime(), ball_ang, target_angle);*/
      sdsTurnTime.AddPoint((float)curr_turn_time);
      curr_turn_time = 0;
      target_angle = (target_angle + 180) % 360;
    }
  }
  
}

void TMTurnball::ResetBall()
{
  MoveBall(Mem->TP_turnball_start_per * Mem->SP_kickable_area, 0);
  time_out_of_range = 0;
  target_angle = 180;
  curr_turn_time = 0;
}

void TMTurnball::LogHeader(EpochController* pEC)
{
  ofstream outfile;

  outfile.open(Mem->TP_training_log_fn, ios::out);
  if (!outfile) {
    my_error("Could not open data file: %s", Mem->TP_training_log_fn);
    return;
  }
    
  /* write header */
  outfile << "Training Info for Turnball behavior" << endl;
  outfile << "Format: ";
  pEC->WriteOptionNames(outfile);
  outfile << "num_losses num half-turns mean time var stdev" << endl;
  outfile << "Epoch Length is: " << Mem->TP_epoch_length << endl;
  outfile.close();
}


void TMTurnball::LogPerformance(EpochController* pEC)
{
  ofstream outfile;

  outfile.open(Mem->TP_training_log_fn, ios::app);
  if (!outfile) {
    my_error("Could not open data file: %s", Mem->TP_training_log_fn);
    return;
  }

  pEC->WriteCurrentOptionValues(outfile);
  outfile << num_losses << " ";
  sdsTurnTime.WriteCompactInfoToFile(outfile, TRUE);
  outfile << endl;
  outfile.close();
}

void TMTurnball::ResetForEpoch()
{
  /* reset the counting values */
  num_losses = 0;
  sdsTurnTime.Reset();
  /* reset the ball */
  ResetBall();
}

