/* -*- Mode: C++ -*- */

/* trainer/TMshot.C
 * CMUnited99 (code for off-line trainer)
 * Patrick Riley <pfr+@cs.cmu.edu>
 * Computer Science Department
 * Carnegie Mellon University
 * Copyright (C) 1999 Patrick Riley
 *
 * CMUnited-99 was created by Peter Stone, Patrick Riley, and Manuela Veloso
 *
 * You may copy and distribute this program freely as long as you retain this notice.
 * If you make any changes or have any comments we would appreciate a message.
 */

/* this is a sample training module. See the README for details about this program */


/* this is the training module for the shot behavior */

#include <unistd.h>
#include <iostream.h>
#include <fstream.h>
#include <iomanip.h>
#include "client.h"
#include "utils.h"
#include "TMshot.h"
#include "MemPosition.h"


const char* TMShot::Name = "Shot";

Bool TMShot::Initialize()
{
  Ear(TRUE);

  num_shots = num_goals = num_saves = 0;

  starting_dist = Mem->TP_shot_min_dist;

  should_reset = TRUE;
  
  /* now we need to adavance our values to the correct place for the
     given epoch start */
  for (int i = 0; i < Mem->TP_epoch_start; i++)
    IncrementDist();

  cout << "Shot Training module" << endl;
  
  return TRUE;
}

/* ranges to check:
   */
void TMShot::InitializeEpochController(EpochController* pEC)
{
  pEC->num_vars = 0;
  pEC->num_epochs = 0;

}


/* Does the following:
   */
void TMShot::NewSightHandler()
{
  
  pPlayerInfo pPIgoalie = Mem->GetPlayer(TS_Left, 1);
  pPlayerInfo pPIstriker = Mem->GetPlayer(TS_Right, 1);
  if (pPIgoalie == NULL || pPIstriker == NULL) {
    cout << "Trainer: Waiting for players to connect" << endl;
    return; /* no player yet */
  }

  if (should_reset) {    
    /* if we're not using epochs, pick a ranodm dist */
    if (!Mem->TP_use_epochs)
      starting_dist = range_random(Mem->TP_shot_min_dist, Mem->TP_shot_max_dist);
    
    float ang = range_random(-Mem->TP_shot_max_ang, Mem->TP_shot_max_ang);
    Vector player_pos = Vector(-Mem->SP_pitch_length / 2, 0) +
      Polar2Vector(starting_dist, ang);
    MovePlayer(TS_Right, 1, player_pos);
    MoveBall(player_pos + Polar2Vector(.6, 60));
    MovePlayer(TS_Left, 1, -Mem->SP_pitch_length/2 + 3, 0, ang);
    
    if (Mem->GetPlayMode() != PM_Play_On)
      ChangeMode(PM_Play_On);

    time_not_kickable = 0;
    curr_shot_time = 0;
    num_shots++;

    should_reset = FALSE;
  } else {

    pBallInfo pBI = Mem->GetBall();

    if (pBI->pos.dist(pPIstriker->pos) > Mem->SP_kickable_area) {
      if (time_not_kickable++ == 0) {
	if (!Mem->TP_use_epochs)
	  cout << "Shot vel: " << pBI->vel.mod() << endl;
      }
    }
    
    switch (Mem->GetPlayMode()) {
    case PM_Play_On:
      //if (!Mem->TP_use_epochs) cout << "In play_on mode" << endl;
      if (++curr_shot_time > Mem->TP_shot_max_time) {
	if (!Mem->TP_use_epochs) cout << "Time: " << Mem->GetTime() << "\tFailure" << endl;
	num_failed++;
	should_reset = TRUE;
      }
      break;
    case PM_After_Goal_Right:
      if (!Mem->TP_use_epochs) cout << "Time: " << Mem->GetTime() << "\tGoal!" << endl;
      num_goals++;
      should_reset = TRUE;
      break;
    default:
      /* looks like the goalie saved it */
      if (!Mem->TP_use_epochs) cout << "Time: " << Mem->GetTime() << "\tSaved!" << endl;
      num_saves++;
      should_reset = TRUE;
      break;
    }
  
  }
  
}

void TMShot::LogHeader(EpochController* pEC)
{
  ofstream outfile;

  outfile.open(Mem->TP_training_log_fn, ios::out);
  if (!outfile) {
    my_error("Could not open data file: %s", Mem->TP_training_log_fn);
    return;
  }
    
  /* write header */
  outfile << "Training Info for Shot behavior" << endl;
  outfile << "Format: ";
  pEC->WriteOptionNames(outfile);
  outfile << "shot_dist ";
  outfile << "num_shots num_goals num_saves num_failed ";
  outfile << "perc_goals perc_saves perc_fails" << endl;;
    
  outfile << "Epoch Length is: " << Mem->TP_epoch_length << endl;
}


void TMShot::LogPerformance(EpochController* pEC)
{
  ofstream outfile;

  outfile.open(Mem->TP_training_log_fn, ios::app);
  if (!outfile) {
    my_error("Could not open data file: %s", Mem->TP_training_log_fn);
    return;
  }

  pEC->WriteCurrentOptionValues(outfile);
  outfile << setprecision(2) << starting_dist << '\t';
  outfile << num_shots << '\t'
	  << num_goals << '\t'
	  << num_saves << '\t'
	  << num_failed << '\t';
  outfile << ((float)num_goals) / num_shots << '\t'
	  << ((float)num_saves) / num_shots << '\t'
	  << ((float)num_failed) / num_shots << '\t';
  outfile << endl;
  outfile.close();

  // this is a little weird of a place to do this, but hey who cares
  IncrementDist();
}

void TMShot::ResetForEpoch()
{
  /* reset the counting values */
  num_shots = num_goals = num_saves = num_failed = 0;

  /* reset the ball */
  should_reset = TRUE;
}

void TMShot::IncrementDist()
{
  starting_dist += Mem->TP_shot_dist_inc;
  if (starting_dist > Mem->TP_shot_max_dist)
    starting_dist = Mem->TP_shot_min_dist;
}




