/* -*- Mode: C++ -*- */

/* trainer/TMhardkick.C
 * CMUnited99 (code for off-line trainer)
 * Patrick Riley <pfr+@cs.cmu.edu>
 * Computer Science Department
 * Carnegie Mellon University
 * Copyright (C) 1999 Patrick Riley
 *
 * CMUnited-99 was created by Peter Stone, Patrick Riley, and Manuela Veloso
 *
 * You may copy and distribute this program freely as long as you retain this notice.
 * If you make any changes or have any comments we would appreciate a message.
 */

/* this is a sample training module. See the README for details about this program */

/* this is the training module for the hardkick behavior */

#include <unistd.h>
#include <iostream.h>
#include <fstream.h>
#include "client.h"
#include "utils.h"
#include "TMhardkick.h"
#include "MemPosition.h"

const char* TMHardKick::Name = "HardKick";

Bool TMHardKick::Initialize()
{
  ChangeMode(PM_Play_On);
  MoveBall(10, 10); /* so we can move the player to middle */

  time_out_of_range = Mem->TP_hardkick_max_time_out+1;
  starting_angle = -180;
  starting_per   = 0.0;
  curr_kick_time = 0;

  HardKickLogFile.open("TrainingData/some_hard_kicks.log");
  
  cout << "HardKick Training module" << endl;
  
  return TRUE;
}

/* ranges to check:
   -hard_kick_dist_buffer: 0 - .3 by .05 7
   -player_ang           : 0 - 180 by 20 10
   */
void TMHardKick::InitializeEpochController(EpochController* pEC)
{
  pEC->num_vars = 2;
  strcpy(pEC->option_names[0], "hard_kick_dist_buffer");
  strcpy(pEC->option_names[1], "hardest_kick_player_ang");
  pEC->option_types[0] = TVT_Float;
  pEC->option_types[1] = TVT_Int;
  
  pEC->num_epochs = 0;
  for (float buf=0.0;
       buf < .32;
       buf += .05) {
    for (int ang=0;
	 ang <= 180;
	 ang += 20) {
      if (pEC->num_epochs >= pEC->MAX_NUM_EPOCHS) {
	my_error("Too many epochs wanted for hardkick!");
	return;
      }
      pEC->training_epochs[pEC->num_epochs][0].f = buf;
      pEC->training_epochs[pEC->num_epochs][1].i = ang;
      pEC->num_epochs++;
    }
  }
}


/* Does the following:
   -insures the player is at (0,0)
   -if the ball is out of kickable range for xx cycles,
     puts it next to player, at an angle determined by starting_angle, which
     goes around player
   -records angle of kicked ball
   -records velocity immediately on leaving kickable area
   -records angle just before reset
   */
/* Assumes: -The relevant player is the first one on the left team */
void TMHardKick::NewSightHandler()
{
  
  pPlayerInfo pPI = Mem->GetPlayer(TS_Left, 1);
  if (pPI == NULL) {
    cout << "Trainer: Waiting for player to connect" << endl;
    return; /* no player yet */
  }
  
  if (pPI->pos != Vector(0,0)) {
    MovePlayer(TS_Left, 1, 0, 0);
  }

  
  pBallInfo pBI = Mem->GetBall();
  if (pBI->pos.dist(pPI->pos) > Mem->SP_kickable_area) {
    /* out of kickable range */
    time_out_of_range++;
    if (time_out_of_range == 1) {
      /* this is the first cycle out, so record the velocity and time */
      if (!Mem->TP_use_epochs)
	cout << "Kick with vel: " << pBI->vel.mod()
	     << "\ttime: " << curr_kick_time << endl;
      sdsKickVel.AddPoint(pBI->vel.mod());
      sdsKickTime.AddPoint((float)curr_kick_time);

      if (pBI->vel.mod() > 1.5) {
	HardKickLogFile << pBI->vel.mod() << " "
			<< starting_angle << " "
			<< starting_per   << " ";
	/* NOTE TO SELF: this is kind of cheating */
	Mem->epoch_cont.WriteCurrentOptionValues(HardKickLogFile);
	HardKickLogFile << endl;
      }
    }
    if (time_out_of_range >= Mem->TP_hardkick_max_time_out) {
      /* about to reset, so record angle */
      sdsKickAngle.AddPoint(pBI->pos.dir());
      starting_angle += Mem->TP_hardkick_start_ang_inc;
      if (starting_angle == 180) {
	/* advance starting percent */
	starting_per += Mem->TP_hardkick_start_per_inc;
	if (starting_per > 1.05) /* watch rounding error */
	  starting_per = 0.0;
	starting_angle = -180;
      }
      //cout << "Starting angle: " << starting_angle
      //<< "\tper: " << starting_per << endl;
      ResetBall();
    }
  } else {
    if (++curr_kick_time > Mem->TP_hardkick_max_time_out) {
      /* player has been trying to kick for a long time, give up */
      MoveBall(10, 0);
    }
    
  }
  
}

void TMHardKick::ResetBall()
{
  float dist = Mem->SP_player_size + Mem->SP_ball_size +
    starting_per * Mem->SP_kickable_margin;
  MoveBall(dist * Cos(starting_angle), dist * Sin(starting_angle));
  time_out_of_range = 0;
  curr_kick_time = 0;
}

void TMHardKick::LogHeader(EpochController* pEC)
{
  ofstream outfile;

  outfile.open(Mem->TP_training_log_fn, ios::out);
  if (!outfile) {
    my_error("Could not open data file: %s", Mem->TP_training_log_fn);
    return;
  }
    
  /* write header */
  outfile << "Training Info for HardKick behavior" << endl;
  outfile << "Format: ";
  pEC->WriteOptionNames(outfile);
  outfile << "num_kicks ";
  outfile << "KickVel: mean var stdev ";
  outfile << "KickTime: mean var stdev ";
  outfile << "KickAngle: mean var stdev " << endl;
  outfile << "Epoch Length is: " << Mem->TP_epoch_length << endl;
  outfile.close();
}


void TMHardKick::LogPerformance(EpochController* pEC)
{
  ofstream outfile;

  outfile.open(Mem->TP_training_log_fn, ios::app);
  if (!outfile) {
    my_error("Could not open data file: %s", Mem->TP_training_log_fn);
    return;
  }

  pEC->WriteCurrentOptionValues(outfile);
  outfile << " " << sdsKickVel.GetNumPoints() << "    ";
  sdsKickVel.WriteCompactInfoToFile(outfile, FALSE);
  outfile << "    ";
  sdsKickTime.WriteCompactInfoToFile(outfile, FALSE);
  outfile << "    ";
  sdsKickAngle.WriteCompactInfoToFile(outfile, FALSE);
  outfile << endl;
  outfile.close();

  HardKickLogFile.close();
  HardKickLogFile.open("TrainingData/some_hard_kicks.log");

}

void TMHardKick::ResetForEpoch()
{
  /* reset my ball move values */
  starting_angle = -180;
  starting_per   = 0.0;
  /* reset the counting values */
  sdsKickVel.Reset();
  sdsKickTime.Reset();
  sdsKickAngle.Reset();
  /* reset the ball */
  ResetBall();
}

