/* -*- Mode: C++ -*- */

/* trainer/TMgoalie.C
 * CMUnited99 (code for off-line trainer)
 * Patrick Riley <pfr+@cs.cmu.edu>
 * Computer Science Department
 * Carnegie Mellon University
 * Copyright (C) 1999 Patrick Riley
 *
 * CMUnited-99 was created by Peter Stone, Patrick Riley, and Manuela Veloso
 *
 * You may copy and distribute this program freely as long as you retain this notice.
 * If you make any changes or have any comments we would appreciate a message.
 */

/* this is a sample training module. See the README for details about this program */

/* this is the training module for the goalie behavior */

#include <unistd.h>
#include <iostream.h>
#include <iomanip.h>
#include <fstream.h>
#include "client.h"
#include "utils.h"
#include "TMgoalie.h"
#include "MemPosition.h"

#define Debug(x) 

const char* TMGoalie::Name = "Goalie";

inline TMGoalie::SpotInfo* TMGoalie::Ref(SpotInfo* aSI, int x, int y)
    { return &aSI[x * Mem->TP_goalie_rect_x_steps + y]; }



Bool TMGoalie::Initialize()
{
  curr_wait_time = 0;

  curr_targ = Mem->SP_goal_width; // to make sure it advances
  shots_from_spot = Mem->TP_goalie_shots_per_loc; // to make sure it advances

  ball_rect=Rectangle(Mem->TP_goalie_rect_left_x,
		      Mem->TP_goalie_rect_left_x + Mem->TP_goalie_rect_length,
		      -Mem->TP_goalie_rect_width / 2,
		       Mem->TP_goalie_rect_width / 2);
  curr_spot = ball_rect.BottomLeftCorner();
  curr_spot_grid_x = curr_spot_grid_y = 0;

  aSpotInfo = new SpotInfo[Mem->TP_goalie_rect_x_steps *
			   Mem->TP_goalie_rect_x_steps];
  
  cout << "Goalie Training module" << endl;

  Ear(TRUE);
  
  return TRUE;
}

/* ranges to check:
 */
void TMGoalie::InitializeEpochController(EpochController* pEC)
{
  pEC->num_vars = 0;
  pEC->num_epochs = 1;
}


/* Does the following:
*/
/* Assumes: -The relevant player is the first one on the left team */
void TMGoalie::NewSightHandler()
{
  
  pPlayerInfo pPI = Mem->GetPlayer(TS_Left, 1);
  if (pPI == NULL) {
    cout << "Trainer: Waiting for player to connect" << endl;
    return; /* no player yet */
  }

  pBallInfo pBI = Mem->GetBall();

  switch (Mem->GetPlayMode()) {
  case PM_Play_On:
    if (pBI->vel.mod() == 0 && ball_rect.IsWithin(pBI->pos)) {
      if (++curr_wait_time > Mem->TP_goalie_set_wait_time) {
	/* pick a shot target */
	Debug(cout << "Time: " << Mem->GetTime() << "\tKicking the ball" << endl);
	Vector targ;
	if (Mem->TP_goalie_target_random)
	  targ=Vector (-Mem->SP_pitch_length / 2,
		       range_random(-Mem->SP_goal_width / 2,
				    Mem->SP_goal_width / 2));
	else {
	  curr_targ += Mem->SP_goal_width / Mem->TP_goalie_num_shot_targets;
	  if (curr_targ > (Mem->SP_goal_width / 2) -
	      Mem->TP_goalie_corner_buffer)
	    curr_targ = -Mem->SP_goal_width / 2 +
	      Mem->TP_goalie_corner_buffer;
	  targ=Vector (-Mem->SP_pitch_length / 2, curr_targ);
	}
	
	Vector traj = targ - pBI->pos;
	traj *= Mem->SP_ball_speed_max / traj.mod();
	MoveBall(pBI->pos, traj); /* "kick" the ball */
	curr_shot_time = 0;
	Ref(aSpotInfo, curr_spot_grid_x, curr_spot_grid_y)->num_shots++;
      }
    } else {
      /* we have a shot in progress */
      curr_wait_time = 0;
      Debug(cout << "Time: " << Mem->GetTime() << "\tShot in progress" << endl);
      if (++curr_shot_time > Mem->TP_goalie_max_shot_time)
	ResetBallAndPlayer();
    }
    break;
  case PM_After_Goal_Right:
    Debug(cout << "Time: " << Mem->GetTime() << "\tGoal!" << endl);
    if (++curr_wait_time > Mem->TP_goalie_reset_wait_time) {
      Ref(aSpotInfo, curr_spot_grid_x, curr_spot_grid_y)->num_goals++;
      ResetBallAndPlayer();
    }
    break;
  default:
    /* looks like the goalie saved it */
    Debug(cout << "Time: " << Mem->GetTime() << "\tGoalie saved it" << endl);	
    if (++curr_wait_time > Mem->TP_goalie_reset_wait_time) {
      ResetBallAndPlayer();
    }
    break;
  }
  
  
}

void TMGoalie::ResetBallAndPlayer()
{
  MovePlayer(TS_Left, 1,
	     -Mem->SP_pitch_length / 2 + Mem->SP_goal_area_length, 0, 0);
  if (Mem->TP_goalie_spot_random) {
    Debug(cout << "Setting the ball randomly" << endl);
    MoveBall(ball_rect.random());
  } else {
    if (++shots_from_spot > Mem->TP_goalie_shots_per_loc) {
      shots_from_spot = 0;
      curr_spot.x += ball_rect.Width() / Mem->TP_goalie_rect_x_steps;
      curr_spot_grid_x++;
      if (!ball_rect.IsWithin(curr_spot)) {
	curr_spot.x = ball_rect.LeftX();
	curr_spot_grid_x = 0;
	curr_spot.y += ball_rect.Height() / Mem->TP_goalie_rect_y_steps;
	curr_spot_grid_y++;
	if (!ball_rect.IsWithin(curr_spot)) {
	  curr_spot_grid_y = 0;
	  curr_spot.y = ball_rect.TopY();
	}
      }
    }
    MoveBall(curr_spot);
  }
  
  
  ChangeMode(PM_Play_On);
  curr_wait_time = 0;
}



void TMGoalie::LogHeader(EpochController* pEC)
{
  ofstream outfile;

  outfile.open(Mem->TP_training_log_fn, ios::out);
  if (!outfile) {
    my_error("Could not open data file: %s", Mem->TP_training_log_fn);
    return;
  }
    
  /* write header */
  outfile << "Training Info for Goalie behavior" << endl;
  outfile << "Format: ";
  outfile << "Epoch Length is: " << Mem->TP_epoch_length << endl;
}


void TMGoalie::LogPerformance(EpochController* pEC)
{
  ofstream outfile;

  outfile.open(Mem->TP_training_log_fn, ios::app);
  if (!outfile) {
    my_error("Could not open data file: %s", Mem->TP_training_log_fn);
    return;
  }

  for (int y=0; y < Mem->TP_goalie_rect_y_steps; y++) {
    for (int x=0; x < Mem->TP_goalie_rect_x_steps; x++) {
      outfile << Ref(aSpotInfo, x, y)->num_goals
	      << "/"
	      << Ref(aSpotInfo, x, y)->num_shots
	      << "\t";
    }
    outfile << endl;
  }
  outfile << endl;
}

void TMGoalie::ResetForEpoch()
{
  /* reset the counting values */
  for (int i = 0;
       i < Mem->TP_goalie_rect_x_steps * Mem->TP_goalie_rect_x_steps;
       i++)
    aSpotInfo[i].num_goals = aSpotInfo[i].num_shots = 0;
  
  /* reset the ball */
  ResetBallAndPlayer();
}

