/* -*- Mode: C++ -*- */

/* trainer/TMdribble.C
 * CMUnited99 (code for off-line trainer)
 * Patrick Riley <pfr+@cs.cmu.edu>
 * Computer Science Department
 * Carnegie Mellon University
 * Copyright (C) 1999 Patrick Riley
 *
 * CMUnited-99 was created by Peter Stone, Patrick Riley, and Manuela Veloso
 *
 * You may copy and distribute this program freely as long as you retain this notice.
 * If you make any changes or have any comments we would appreciate a message.
 */

/* this is a sample training module. See the README for details about this program */

/* this is the training module for the dribble behavior */

#include <unistd.h>
#include <iostream.h>
#include <iomanip.h>
#include <fstream.h>
#include "client.h"
#include "utils.h"
#include "TMdribble.h"
#include "MemPosition.h"

const char* TMDribble::Name = "Dribble";

Bool TMDribble::Initialize()
{
  ChangeMode(PM_Play_On);

  time_out_of_range = 0;
  curr_drib_time = 0;
  num_losses = 0;
  time_in_control = 0;
  time_out_of_control = 0;

  target = Vector(10,10);
  
  FirstTime = TRUE;

  sdsDribTime.SetPrecision(3);
  sdsDistOff.SetPrecision(3);

  cout << "Dribble Training module" << endl;
  
  return TRUE;
}

/* ranges to check:
   dribble_ball_dist:   .4 - 1.0    by .05    num: 12
   collision_buffer:    .05 - .3     by .05    num: 6 -->
   kickable_buffer:     .05 - .3     by .05    num: 6 __> same value
   dribble_dash_power:   50 - 100   by 10     num: 5
 */
void TMDribble::InitializeEpochController(EpochController* pEC)
{
  pEC->num_vars = 4;
  strcpy(pEC->option_names[0], "dribble_ball_dist");
  strcpy(pEC->option_names[1], "collision_buffer");
  strcpy(pEC->option_names[2], "kickable_buffer");
  strcpy(pEC->option_names[3], "dribble_dash_pow");
  pEC->option_types[0] = TVT_Float;
  pEC->option_types[1] = TVT_Float;
  pEC->option_types[2] = TVT_Float;
  pEC->option_types[3] = TVT_Int;

#define NEW_VALS
  
#ifdef NEW_VALS
  pEC->num_epochs = 0;
  for (float bdist=.4;
       bdist <= 1.0;
       bdist += .05) {
    for (float buf =0.05;
	 buf <= .31;
	 buf += .05) {
      for (int pow = 50;
	   pow <= 100;
	   pow += 10) {
	if (pEC->num_epochs >= pEC->MAX_NUM_EPOCHS) {
	  my_error("Too many epochs wanted for dribble!");
	  return;
	}
	pEC->training_epochs[pEC->num_epochs][0].f = bdist;
	pEC->training_epochs[pEC->num_epochs][1].f = buf;
	pEC->training_epochs[pEC->num_epochs][2].f = buf;
	pEC->training_epochs[pEC->num_epochs][3].i = pow;
	pEC->num_epochs++;
      }
    }
  }
#endif
  
#ifdef OLD_VALS
  pEC->num_epochs = 1;
  pEC->training_epochs[0][0].f = 1.2;
  pEC->training_epochs[0][1].f = .1;
  pEC->training_epochs[0][2].f = .1;
  pEC->training_epochs[0][3].i = 75;
#endif  

}


/* Does the following:
   -if the ball is out of range for xx cycles, puts it next to player
   -records how long it takes to get from place to place
   -counts number of times the ball is lost
*/
/* Assumes: -The relevant player is the first one on the left team */
void TMDribble::NewSightHandler()
{
  
  pPlayerInfo pPI = Mem->GetPlayer(TS_Left, 1);
  if (pPI == NULL) {
    cout << "Trainer: Waiting for player to connect" << endl;
    return; /* no player yet */
  }
  
  if (FirstTime) {
    if (!Mem->TP_use_epochs)
      ResetForEpoch();
    FirstTime = FALSE;
  }

  curr_drib_time++;

  Line l = LineFromTwoPoints(target, -target);
  sdsDistOff.AddPoint(l.dist(pPI->pos));
		      
  pBallInfo pBI = Mem->GetBall();
  if (pBI->pos.dist(pPI->pos) > Mem->SP_kickable_area) {
    /* ball is NOT in kickable area */
    time_out_of_control++;
    time_out_of_range++;
    if (time_out_of_range >= Mem->TP_dribble_max_time_out) {
      num_losses++;
      ResetBall(pPI->pos);
    }
  } else {
    /* ball is in kickable area */
    time_in_control++;
    time_out_of_range = 0;
    if (target.dist(pPI->pos) < targ_buffer) {
      /* the agent reached the target */
      sdsDribTime.AddPoint((float)curr_drib_time);
      curr_drib_time = 0;
      target = -target;
    }
  }  
}

void TMDribble::ResetBall(Vector ppos)
{
  Vector offset;
  offset = Polar2Vector(Mem->TP_dribble_start_ball_dist,
			(target - ppos).dir());
  MoveBall(ppos + offset);
}

void TMDribble::LogHeader(EpochController* pEC)
{
  ofstream outfile;

  outfile.open(Mem->TP_training_log_fn, ios::out);
  if (!outfile) {
    my_error("Could not open data file: %s", Mem->TP_training_log_fn);
    return;
  }
    
  /* write header */
  outfile << "Training Info for Dribble behavior" << endl;
  outfile << "Format: ";
  pEC->WriteOptionNames(outfile);
  outfile << "num_losses perc_time_control mean-dist-off var-dist-off stdev-dist-off num-dribs mean-time var-time stdev-time" << endl;
  outfile << "Epoch Length is: " << Mem->TP_epoch_length << endl;
  outfile.close();
}


void TMDribble::LogPerformance(EpochController* pEC)
{
  ofstream outfile;

  outfile.open(Mem->TP_training_log_fn, ios::app);
  if (!outfile) {
    my_error("Could not open data file: %s", Mem->TP_training_log_fn);
    return;
  }

  pEC->WriteCurrentOptionValues(outfile);
  float perc_in_control = ((float)time_in_control) / (time_in_control+time_out_of_control);
  outfile << num_losses << "\t"
	  << setprecision(2) << perc_in_control << "\t";
  sdsDistOff.WriteCompactInfoToFile(outfile, FALSE);
  outfile << "\t";
  sdsDribTime.WriteCompactInfoToFile(outfile, TRUE);
  outfile << endl;
  outfile.close();
}

void TMDribble::ResetForEpoch()
{
  /* reset the counting values */
  num_losses = 0;
  sdsDribTime.Reset();
  time_in_control = 0;
  time_out_of_control = 0;
  
  /* reset the ball */
  time_out_of_range = 0;
  curr_drib_time = 0;
  MovePlayer(TS_Left, 1, -target);
  ResetBall(-target);
}

